﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkAssert.h>
#include <nn/audio/server/audio_UserServiceHipcServer.h>
#include <nn/audio/audio_AudioOutApi.h>
#include <nn/audio/audio_AudioOutTypes.private.h>
#include <nn/audio/audio_Result.private.h>
#include <nn/dd/dd_DeviceAddressSpace.h>
#include <nn/os/os_TransferMemory.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_Types.h>
#include <nn/applet/applet_Apis.h>
#include "audio_CreateAudioOutManager.h"
#include "../common/audio_Util.h"

namespace nn {
namespace audio {

extern int g_AudioOutCountMax;

namespace {
class AudioOutMap
{
public:
    AudioOutMap()
    : m_totalUsed(0)
    {
        os::InitializeMutex(&m_mutex, true, 0);
        std::memset(m_audioOutPairs, 0, sizeof(m_audioOutPairs));
    }

    ~AudioOutMap()
    {
        os::FinalizeMutex(&m_mutex);
    }

    void AddPair(const AudioOut* key, void* handle) NN_NOEXCEPT
    {
        os::LockMutex(&m_mutex);
        NN_SDK_ASSERT(m_totalUsed!=g_AudioOutCountMax,"Opening too many Audio Outs, can only open %d\n", g_AudioOutCountMax);

        for( int i = 0; i < g_AudioOutCountMax; ++i )
        {
            if( m_audioOutPairs[i].pAudioOutKey == nullptr )
            {
                NN_SDK_ASSERT( m_audioOutPairs[i].pAudioOutHandle == nullptr );
                m_audioOutPairs[i].pAudioOutKey = key;
                m_audioOutPairs[i].pAudioOutHandle = handle;
                ++m_totalUsed;
                break;
            }
        }
        os::UnlockMutex(&m_mutex);
    }

    void RemovePair(const AudioOut* key) NN_NOEXCEPT
    {
        os::LockMutex(&m_mutex);
        for( int i = 0; i < g_AudioOutCountMax; ++i )
        {
            if( m_audioOutPairs[i].pAudioOutKey == key )
            {
                m_audioOutPairs[i].pAudioOutKey = nullptr;
                m_audioOutPairs[i].pAudioOutHandle = nullptr;
                --m_totalUsed;
                break;
            }
        }
        os::UnlockMutex(&m_mutex);
    }

    void* FindPair(const AudioOut* key) NN_NOEXCEPT
    {
        os::LockMutex(&m_mutex);
        for( int i = 0; i < g_AudioOutCountMax; ++i )
        {
            if( m_audioOutPairs[i].pAudioOutKey == key )
            {
                os::UnlockMutex(&m_mutex);
                return m_audioOutPairs[i].pAudioOutHandle;
            }
        }
        os::UnlockMutex(&m_mutex);
        return nullptr;
    }

private:
    struct AudioOutPair
    {
        const AudioOut* pAudioOutKey;
        void* pAudioOutHandle;
    };

    // Allocate max size of AudioOut
    NN_STATIC_ASSERT(nn::audio::AudioOutCountMax < nn::audio::AudioOutCountMaxForLibraryApplet);
    AudioOutPair m_audioOutPairs[nn::audio::AudioOutCountMaxForLibraryApplet];
    int m_totalUsed;
    os::MutexType m_mutex;
};

AudioOutMap g_AudioOutMap;
}

#define NN_AUDIO_USE_SYSTEM_PROCESS

nn::sf::SharedPointer<detail::IAudioOutManager> CreateAudioOutManager() NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
#if defined(NN_AUDIO_USE_SYSTEM_PROCESS)
    return CreateAudioOutManagerByHipc();
#else
    // DFC に切り替えれば audio process なしでも動作可能
    return CreateAudioOutManagerByDfc();
#endif // #if defined(NN_AUDIO_USE_SYSTEM_PROCESS)
#else
    return CreateAudioOutManagerByDfc();
#endif
}

int ListAudioOuts(nn::audio::AudioOutInfo* audioInfo, int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(audioInfo);
    NN_SDK_REQUIRES_GREATER_EQUAL(count, 0);

    int amountRet = 0;

    if( count > 0 )
    {
        nn::sf::OutBuffer outAudioBuffer(reinterpret_cast<char*>(audioInfo), count * sizeof(*audioInfo));
        auto audioOutManager = CreateAudioOutManager();
        NN_RESULT_DO( audioOutManager->ListAudioOutsAuto(outAudioBuffer, &amountRet) );
    }

    return amountRet;
}

nn::Result OpenDefaultAudioOut(nn::audio::AudioOut* pAudioOut, const nn::audio::AudioOutParameter& parameter) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pAudioOut);
    NN_SDK_REQUIRES(parameter._reserved == common::AudioOutInitializedMagic, "paraemter is not initialized with nn::audio::InitializeAudioOutParameter()");
    return OpenAudioOut(pAudioOut, "", parameter);
}

Result OpenDefaultAudioOut(AudioOut* pOutAudioOut, nn::os::SystemEvent* pOutSystemEvent, const AudioOutParameter& parameter) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutSystemEvent);
    NN_RESULT_DO(OpenDefaultAudioOut(pOutAudioOut, parameter));

    nn::sf::NativeHandle systemEventHandle;
    NN_RESULT_DO(static_cast<detail::IAudioOut*>(g_AudioOutMap.FindPair(pOutAudioOut))->RegisterBufferEvent(&systemEventHandle));
    pOutSystemEvent->AttachReadableHandle(systemEventHandle.GetOsHandle(), systemEventHandle.IsManaged(), nn::os::EventClearMode_AutoClear);
    systemEventHandle.Detach();

    NN_RESULT_SUCCESS;
}

nn::Result OpenAudioOut(nn::audio::AudioOut* pAudioOut, const char* name, const nn::audio::AudioOutParameter& parameter) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pAudioOut);
    NN_SDK_REQUIRES_NOT_NULL(name);
    NN_SDK_REQUIRES(parameter._reserved == common::AudioOutInitializedMagic, "paraemter is not initialized with nn::audio::InitializeAudioOutParameter()");
    NN_SDK_REQUIRES(parameter.channelCount == 0 || parameter.channelCount == 2 || parameter.channelCount == 6);
    NN_SDK_REQUIRES(parameter.sampleRate >= 0);

    bool canOpen = g_AudioOutMap.FindPair(pAudioOut) == nullptr;
    NN_SDK_REQUIRES(canOpen, "AudioOut %p has already been opened", pAudioOut);
    NN_RESULT_THROW_UNLESS(canOpen, ResultAlreadyOpen());

    nn::audio::detail::AudioOutParameterInternal audioOutParam;

    nn::sf::SharedPointer<detail::IAudioOut> audioOutShared;
    nn::sf::InBuffer inNameBuff(name, nn::audio::AudioOut::NameLength);

    char nameOut[nn::audio::AudioOut::NameLength];
    nn::sf::OutBuffer outNameBuff(nameOut, nn::audio::AudioOut::NameLength);
    auto processHandle = nn::os::InvalidNativeHandle;
    auto appletResourceUserId = nn::applet::AppletResourceUserId::GetInvalidId();

    #if defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK2) || defined(NN_BUILD_CONFIG_HARDWARE_NX)
        processHandle = nn::dd::GetCurrentProcessHandle();
        NN_ABORT_UNLESS(processHandle != nn::os::InvalidNativeHandle);
        appletResourceUserId = applet::GetAppletResourceUserId();
    #endif

    auto audioOutManager = CreateAudioOutManager();

    NN_RESULT_DO(audioOutManager->OpenAudioOut(&audioOutShared, inNameBuff, parameter, nn::sf::NativeHandle(processHandle, false), &audioOutParam, outNameBuff, appletResourceUserId) );

    g_AudioOutMap.AddPair(pAudioOut, audioOutShared.Detach());

    pAudioOut->sampleRate = audioOutParam.sampleRate;
    pAudioOut->channelCount = audioOutParam.channelCount;
    pAudioOut->sampleFormat = static_cast<nn::audio::SampleFormat>(audioOutParam.sampleFormat);
    pAudioOut->state = static_cast<nn::audio::AudioOutState>(audioOutParam.state);
    memset(pAudioOut->_reserved, 0, sizeof(pAudioOut->_reserved));
    strcpy(pAudioOut->name, nameOut);

    NN_RESULT_SUCCESS;
}

Result OpenAudioOut(AudioOut* pOutAudioOut, nn::os::SystemEvent* pOutSystemEvent, const char* name, const AudioOutParameter& parameter) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutSystemEvent);

    NN_RESULT_DO(OpenAudioOut(pOutAudioOut, name, parameter));

    nn::sf::NativeHandle systemEventHandle;
    NN_RESULT_DO(static_cast<detail::IAudioOut*>(g_AudioOutMap.FindPair(pOutAudioOut))->RegisterBufferEvent(&systemEventHandle));
    pOutSystemEvent->AttachReadableHandle(systemEventHandle.GetOsHandle(), systemEventHandle.IsManaged(), nn::os::EventClearMode_AutoClear);
    systemEventHandle.Detach();

    NN_RESULT_SUCCESS;
}

void CloseAudioOut(AudioOut* pAudioOut) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pAudioOut);

     void* handle = g_AudioOutMap.FindPair(pAudioOut);
     if( handle == nullptr )
     {
         return;
     }

     NN_SDK_REQUIRES_EQUAL(GetAudioOutState(pAudioOut), AudioOutState_Stopped);
     pAudioOut->state = AudioOutState_Stopped;

     nn::sf::ReleaseSharedObject(static_cast<detail::IAudioOut*>(handle));
     g_AudioOutMap.RemovePair(pAudioOut);
}

nn::Result StartAudioOut(AudioOut* pAudioOut) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pAudioOut);
    NN_SDK_REQUIRES_EQUAL(GetAudioOutState(pAudioOut), AudioOutState_Stopped);

    void* handle = g_AudioOutMap.FindPair(pAudioOut);
    NN_SDK_ASSERT(handle);
    return static_cast<detail::IAudioOut*>(handle)->Start();
}

void StopAudioOut(nn::audio::AudioOut* pAudioOut) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pAudioOut);
    NN_SDK_REQUIRES_EQUAL(GetAudioOutState(pAudioOut), AudioOutState_Started);

    void* handle = g_AudioOutMap.FindPair(pAudioOut);
    NN_SDK_ASSERT(handle);
    static_cast<detail::IAudioOut*>(handle)->Stop();
}

nn::audio::AudioOutState GetAudioOutState(const nn::audio::AudioOut* pAudioOut) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pAudioOut);
    uint32_t audioState;

    void* handle = g_AudioOutMap.FindPair(pAudioOut);
    if( handle == nullptr )
    {
        return pAudioOut->state;
    }

    NN_ABORT_UNLESS_RESULT_SUCCESS( static_cast<detail::IAudioOut*>(handle)->GetAudioOutState(&audioState) );

    return static_cast<AudioOutState>(audioState);
}

const char* GetAudioOutName(const nn::audio::AudioOut* pAudioOut) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pAudioOut);
    return pAudioOut->name;
}

int32_t GetAudioOutSampleRate(const nn::audio::AudioOut* pAudioOut) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pAudioOut);
    return pAudioOut->sampleRate;
}

int32_t GetAudioOutChannelCount(const nn::audio::AudioOut* pAudioOut) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pAudioOut);
    return pAudioOut->channelCount;
}

nn::audio::SampleFormat GetAudioOutSampleFormat(const nn::audio::AudioOut* pAudioOut) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pAudioOut);
    return pAudioOut->sampleFormat;
}

bool AppendAudioOutBuffer(nn::audio::AudioOut* pAudioOut, nn::audio::AudioOutBuffer* outAudioBuffer) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pAudioOut);
    NN_SDK_REQUIRES_NOT_NULL(outAudioBuffer);
    NN_SDK_REQUIRES_NOT_NULL(outAudioBuffer->buffer);
    NN_SDK_REQUIRES_ALIGNED(outAudioBuffer->buffer, nn::audio::AudioOutBuffer::AddressAlignment);
    NN_SDK_REQUIRES_ALIGNED(outAudioBuffer->bufferSize, nn::audio::AudioOutBuffer::SizeGranularity);
    NN_SDK_REQUIRES(outAudioBuffer->size % (nn::audio::GetAudioOutChannelCount(pAudioOut) * nn::audio::GetSampleByteSize(nn::audio::GetAudioOutSampleFormat(pAudioOut))) == 0);

    void* handle = g_AudioOutMap.FindPair(pAudioOut);
    NN_SDK_ASSERT(handle);
    nn::sf::InBuffer inBuf( reinterpret_cast<char*>(outAudioBuffer), sizeof( *outAudioBuffer ) );

    const auto result = static_cast<detail::IAudioOut*>(handle)->AppendAudioOutBufferAuto(inBuf, reinterpret_cast<uintptr_t>(outAudioBuffer));

    if(ResultBufferCountMax::Includes(result))
    {
        return false;
    }
    else
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        return true;
    }
}

AudioOutBuffer* GetReleasedAudioOutBuffer(AudioOut* pAudioOut) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pAudioOut);
    int countRet;
    int count = 1;

    //Right now just returning a pointer of the original buffer, in the future it should return an array of pointers
    uint64_t audioOutArray;
    nn::sf::OutBuffer outArray(reinterpret_cast<char *>(&audioOutArray), count*sizeof(audioOutArray));

    void* handle = g_AudioOutMap.FindPair(pAudioOut);
    NN_SDK_ASSERT(handle);
    static_cast<detail::IAudioOut*>(handle)->GetReleasedAudioOutBuffersAuto(outArray, &countRet);

    AudioOutBuffer* outBuffer = reinterpret_cast<AudioOutBuffer*>(audioOutArray);
    return outBuffer;
}

bool ContainsAudioOutBuffer(const AudioOut* pAudioOut, const AudioOutBuffer* pAudioOutBuffer) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pAudioOut);
    NN_SDK_REQUIRES_NOT_NULL(pAudioOutBuffer);
    bool contains;

    void* handle = g_AudioOutMap.FindPair(pAudioOut);
    NN_SDK_ASSERT(handle);
    NN_ABORT_UNLESS_RESULT_SUCCESS( static_cast<detail::IAudioOut*>(handle)->ContainsAudioOutBuffer(reinterpret_cast<uintptr_t>(pAudioOutBuffer), &contains) );

    return contains;
}

nn::Result RegisterBufferEvent(nn::audio::AudioOut* pAudioOut, nn::os::SystemEvent* pBufferEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pAudioOut);
    NN_SDK_REQUIRES_NOT_NULL(pBufferEvent);
    nn::sf::NativeHandle systemEventHandle;

    void* handle = g_AudioOutMap.FindPair(pAudioOut);
    NN_SDK_ASSERT(handle);
    NN_RESULT_DO(static_cast<detail::IAudioOut*>(handle)->RegisterBufferEvent(&systemEventHandle));

    pBufferEvent->AttachReadableHandle(systemEventHandle.GetOsHandle(), systemEventHandle.IsManaged(), nn::os::EventClearMode_AutoClear);
    systemEventHandle.Detach();
    NN_RESULT_SUCCESS;
}

int GetAudioOutBufferCount(const AudioOut* pAudioOut) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pAudioOut);
    void* handle = g_AudioOutMap.FindPair(pAudioOut);
    NN_SDK_ASSERT(handle);
    int bufferCount = 0;
    detail::IAudioOut* audioOut = static_cast<detail::IAudioOut*>(handle);
    NN_ABORT_UNLESS_RESULT_SUCCESS( audioOut->GetAudioOutBufferCount(&bufferCount));
    return bufferCount;
}

int64_t GetAudioOutPlayedSampleCount(const AudioOut* pAudioOut) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pAudioOut);
    void* handle = g_AudioOutMap.FindPair(pAudioOut);
    NN_SDK_ASSERT(handle);
    int64_t sampleCount;
    detail::IAudioOut* audioOut = static_cast<detail::IAudioOut*>(handle);
    NN_ABORT_UNLESS_RESULT_SUCCESS(audioOut->GetAudioOutPlayedSampleCount(&sampleCount));
    return sampleCount;
}

bool FlushAudioOutBuffers(AudioOut* pAudioOut) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pAudioOut);
    // TODO: This limitation will be removed on NxAddon 5.x.x.
    NN_SDK_ASSERT_EQUAL(GetAudioOutState(pAudioOut), AudioOutState_Stopped);
    void* handle = g_AudioOutMap.FindPair(pAudioOut);
    NN_SDK_ASSERT(handle);

    bool isAllBufferFlushed = false;

    detail::IAudioOut* audioOut = static_cast<detail::IAudioOut*>(handle);
    NN_ABORT_UNLESS_RESULT_SUCCESS(audioOut->FlushAudioOutBuffers(&isAllBufferFlushed));

    return isAllBufferFlushed;
}

void SetAudioOutVolume(AudioOut* pAudioOut, float volume) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pAudioOut);
    NN_SDK_ASSERT_MINMAX(volume, AudioOut::GetVolumeMin(), AudioOut::GetVolumeMax());
    void* handle = g_AudioOutMap.FindPair(pAudioOut);
    NN_SDK_ASSERT(handle);

    detail::IAudioOut* audioOut = static_cast<detail::IAudioOut*>(handle);
    NN_ABORT_UNLESS_RESULT_SUCCESS(audioOut->SetAudioOutVolume(volume));
}

float GetAudioOutVolume(const AudioOut* pAudioOut) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pAudioOut);
    void* handle = g_AudioOutMap.FindPair(pAudioOut);
    NN_SDK_ASSERT(handle);

    float volume = 0.0f;
    detail::IAudioOut* audioOut = static_cast<detail::IAudioOut*>(handle);
    NN_ABORT_UNLESS_RESULT_SUCCESS(audioOut->GetAudioOutVolume(&volume));

    return volume;
}

}  // namespace audio
}  // namespace nn
