﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkAssert.h>
#include <nn/audio/server/audio_UserServiceHipcServer.h>
#include <nn/audio/audio_AudioInApi.h>
#include <nn/audio/audio_AudioInApi-spec.nx.h>
#include <nn/audio/audio_Result.private.h>
#include <nn/dd/dd_DeviceAddressSpace.h>
#include <nn/os/os_TransferMemory.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_Types.h>
#include <nn/applet/applet_Apis.h>

#include "audio_CreateAudioInManager.h"
#include "../common/audio_Util.h"
#include "../common/audio_AudioInPrivate.h"

namespace nn{
namespace audio {

namespace {

class AudioInMap
{
public:
    AudioInMap()
    : m_totalUsed(0)
    {
        os::InitializeMutex(&m_mutex, true, 0);
        std::memset(m_audioInPairs, 0, sizeof(m_audioInPairs));
    }

    ~AudioInMap()
    {
        os::FinalizeMutex(&m_mutex);
    }

    void AddPair(const AudioIn* key, void* handle) NN_NOEXCEPT
    {
        os::LockMutex(&m_mutex);
        NN_SDK_ASSERT(m_totalUsed!=nn::audio::AudioInCountMax,"Opening too many Audio Ins, can only open %d\n", nn::audio::AudioInCountMax);

        for( int i = 0; i < nn::audio::AudioInCountMax; ++i )
        {
            if( m_audioInPairs[i].pAudioInKey == nullptr )
            {
                NN_SDK_ASSERT( m_audioInPairs[i].pAudioInHandle == nullptr );
                m_audioInPairs[i].pAudioInKey = key;
                m_audioInPairs[i].pAudioInHandle = handle;
                ++m_totalUsed;
                break;
            }
        }
        os::UnlockMutex(&m_mutex);
    }

    void RemovePair(const AudioIn* key) NN_NOEXCEPT
    {
        os::LockMutex(&m_mutex);
        for( int i = 0; i < nn::audio::AudioInCountMax; ++i )
        {
            if( m_audioInPairs[i].pAudioInKey == key )
            {
                m_audioInPairs[i].pAudioInKey = nullptr;
                m_audioInPairs[i].pAudioInHandle = nullptr;
                --m_totalUsed;
                break;
            }
        }
        os::UnlockMutex(&m_mutex);
    }

    void* FindPair(const AudioIn* key) NN_NOEXCEPT
    {
        os::LockMutex(&m_mutex);
        for( int i = 0; i < nn::audio::AudioInCountMax; ++i )
        {
            if( m_audioInPairs[i].pAudioInKey == key )
            {
                os::UnlockMutex(&m_mutex);
                return m_audioInPairs[i].pAudioInHandle;
            }
        }
        os::UnlockMutex(&m_mutex);
        return nullptr;
    }

private:
    struct AudioInPair
    {
        const AudioIn* pAudioInKey;
        void* pAudioInHandle;
    };

    AudioInPair m_audioInPairs[nn::audio::AudioInCountMax];
    int m_totalUsed;
    os::MutexType m_mutex;
};

AudioInMap g_AudioInMap;

#if defined(NN_BUILD_CONFIG_OS_HORIZON) && defined(NN_BUILD_CONFIG_SPEC_NX)
bool g_IsAudioInEnabled = false;
#else
bool g_IsAudioInEnabled = true;
#endif

}
#define NN_AUDIO_USE_SYSTEM_PROCESS

nn::sf::SharedPointer<detail::IAudioInManager> CreateAudioInManager() NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
#if defined(NN_AUDIO_USE_SYSTEM_PROCESS)
    return CreateAudioInManagerByHipc();
#else
    return CreateAudioInManagerByDfc();
#endif
#else
    return CreateAudioInManagerByDfc();
#endif
}

void SetAudioInEnabled(bool enable) NN_NOEXCEPT
{
    g_IsAudioInEnabled = enable;
}

bool IsAudioInEnabled() NN_NOEXCEPT
{
    return g_IsAudioInEnabled;
}

int ListAudioIns(AudioInInfo* audioInfo, int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(audioInfo);
    NN_SDK_REQUIRES_GREATER_EQUAL(count, 0);

    if (!g_IsAudioInEnabled)
    {
        return 0;
    }

    int amountRet = 0;

    if( count > 0 )
    {
        nn::sf::OutBuffer inAudioBuffer(reinterpret_cast<char*>(audioInfo), count * sizeof(*audioInfo));
        auto audioInManager = CreateAudioInManager();
        NN_RESULT_DO( audioInManager->ListAudioInsAutoFiltered(inAudioBuffer, &amountRet) );
    }

    return amountRet;
}

Result OpenDefaultAudioIn(nn::audio::AudioIn* pAudioIn, const AudioInParameter& parameter) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pAudioIn);
    NN_SDK_REQUIRES(parameter._reserved == common::AudioInInitializedMagic, "paraemter is not initialized with nn::audio::InitializeAudioInParameter()");

    if (!g_IsAudioInEnabled)
    {
        NN_RESULT_THROW(ResultDefaultAudioInNotAvailable());
    }

    return OpenAudioIn(pAudioIn, "", parameter);
}

Result OpenDefaultAudioIn(AudioIn* pOutAudioIn, nn::os::SystemEvent* pOutSystemEvent, const AudioInParameter& parameter) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutSystemEvent);

    NN_RESULT_DO(OpenDefaultAudioIn(pOutAudioIn, parameter));

    nn::sf::NativeHandle systemEventHandle;
    NN_RESULT_DO(static_cast<detail::IAudioIn*>(g_AudioInMap.FindPair(pOutAudioIn))->RegisterBufferEvent(&systemEventHandle));
    pOutSystemEvent->AttachReadableHandle(systemEventHandle.GetOsHandle(), systemEventHandle.IsManaged(), nn::os::EventClearMode_AutoClear);
    systemEventHandle.Detach();

    NN_RESULT_SUCCESS;
}

Result OpenAudioIn(AudioIn* pOutAudioIn, nn::os::SystemEvent* pOutSystemEvent, const char* name, const AudioInParameter& parameter) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutSystemEvent);

    NN_RESULT_DO(OpenAudioIn(pOutAudioIn, name, parameter));

    nn::sf::NativeHandle systemEventHandle;
    NN_RESULT_DO(static_cast<detail::IAudioIn*>(g_AudioInMap.FindPair(pOutAudioIn))->RegisterBufferEvent(&systemEventHandle));
    pOutSystemEvent->AttachReadableHandle(systemEventHandle.GetOsHandle(), systemEventHandle.IsManaged(), nn::os::EventClearMode_AutoClear);
    systemEventHandle.Detach();

    NN_RESULT_SUCCESS;
}

Result OpenAudioIn(nn::audio::AudioIn* pAudioIn, const char* name, const nn::audio::AudioInParameter& parameter) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pAudioIn);
    NN_SDK_REQUIRES_NOT_NULL(name);
    NN_SDK_REQUIRES(parameter._reserved == common::AudioInInitializedMagic, "parameter is not initialized with nn::audio::InitializeAudioInParameter()");
    NN_SDK_REQUIRES(parameter.sampleRate >= 0);

    bool canOpen = g_AudioInMap.FindPair(pAudioIn) == nullptr;
    NN_SDK_REQUIRES(canOpen, "AudioIn %p has already been opened", pAudioIn);
    NN_RESULT_THROW_UNLESS(canOpen, ResultAlreadyOpen());

    if (!g_IsAudioInEnabled && strnlen(name, AudioIn::NameLength) == 0)
    {
        NN_RESULT_THROW(ResultDefaultAudioInNotAvailable());
    }

    nn::audio::detail::AudioInParameterInternal audioInParam;

    nn::sf::SharedPointer<detail::IAudioIn> audioInShared;
    nn::sf::InBuffer inNameBuff(name, nn::audio::AudioIn::NameLength);

    char nameIn[nn::audio::AudioIn::NameLength];
    nn::sf::OutBuffer outNameBuff(nameIn, nn::audio::AudioIn::NameLength);
    auto processHandle = nn::os::InvalidNativeHandle;
    auto appletResourceUserId = nn::applet::AppletResourceUserId::GetInvalidId();

    #if defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK2) || defined(NN_BUILD_CONFIG_HARDWARE_NX)
        processHandle = nn::dd::GetCurrentProcessHandle();
        NN_ABORT_UNLESS(processHandle != nn::os::InvalidNativeHandle);
        appletResourceUserId = applet::GetAppletResourceUserId();
    #endif
    auto audioInManager = CreateAudioInManager();
    nn::Result ret = (audioInManager->OpenAudioInProtocolSpecified(common::AudioInProtocolFlags::GetAudioInNameFix, &audioInShared, inNameBuff, parameter, nn::sf::NativeHandle(processHandle, false), &audioInParam, outNameBuff, appletResourceUserId) );
    if( ret.IsFailure() )
    {
        void* deadHandle = audioInShared.Detach();
        if( deadHandle == nullptr )
        {
            return ret;
        }
        pAudioIn->state = AudioInState_Stopped;
        nn::sf::ReleaseSharedObject(static_cast<detail::IAudioIn*>(deadHandle));
        return ret;
    }
    g_AudioInMap.AddPair(pAudioIn, audioInShared.Detach());

    pAudioIn->sampleRate = audioInParam.sampleRate;
    pAudioIn->channelCount = audioInParam.channelCount;
    pAudioIn->sampleFormat = static_cast<nn::audio::SampleFormat>(audioInParam.sampleFormat);
    pAudioIn->state = static_cast<nn::audio::AudioInState>(audioInParam.state);
    memset(pAudioIn->_reserved, 0, sizeof(pAudioIn->_reserved));
    strcpy(pAudioIn->name, nameIn);
    NN_RESULT_SUCCESS;
}

void CloseAudioIn(AudioIn* pAudioIn) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pAudioIn);
    void* handle = g_AudioInMap.FindPair(pAudioIn);
    if( handle == nullptr )
    {
        return;
    }

    NN_SDK_REQUIRES(GetAudioInState(pAudioIn) == AudioInState_Stopped);
    pAudioIn->state = AudioInState_Stopped;
    nn::sf::ReleaseSharedObject(static_cast<detail::IAudioIn*>(handle));
    g_AudioInMap.RemovePair(pAudioIn);
}

Result StartAudioIn(AudioIn* pAudioIn) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pAudioIn);
    NN_SDK_REQUIRES_EQUAL(GetAudioInState(pAudioIn), AudioInState_Stopped);
    void* handle = g_AudioInMap.FindPair(pAudioIn);
    NN_SDK_ASSERT(handle);
    return static_cast<detail::IAudioIn*>(handle)->Start();
}

void StopAudioIn(AudioIn* pAudioIn) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pAudioIn);
    NN_SDK_REQUIRES_EQUAL(GetAudioInState(pAudioIn), AudioInState_Started);
    void* handle = g_AudioInMap.FindPair(pAudioIn);
    NN_SDK_ASSERT(handle);
    static_cast<detail::IAudioIn*>(handle)->Stop();
}

nn::audio::AudioInState GetAudioInState(const nn::audio::AudioIn* pAudioIn) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pAudioIn);
    uint32_t audioState;

    void* handle = g_AudioInMap.FindPair(pAudioIn);
    if( handle == nullptr )
    {
        return pAudioIn->state;
    }

    NN_ABORT_UNLESS_RESULT_SUCCESS( static_cast<detail::IAudioIn*>(handle)->GetAudioInState(&audioState) );

    return static_cast<AudioInState>(audioState);
}

const char* GetAudioInName(const AudioIn* pAudioIn) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pAudioIn);
    return pAudioIn->name;
}

int GetAudioInSampleRate(const AudioIn* pAudioIn) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pAudioIn);
    return pAudioIn->sampleRate;
}

int GetAudioInChannelCount(const AudioIn* pAudioIn) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pAudioIn);
    return pAudioIn->channelCount;
}

SampleFormat GetAudioInSampleFormat(const AudioIn* pAudioIn) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pAudioIn);
    return pAudioIn->sampleFormat;
}

void SetAudioInBufferInfo(AudioInBuffer* pOutAudioInBuffer, void* buffer, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutAudioInBuffer);
    NN_SDK_REQUIRES_NOT_NULL(buffer);
#if defined(NN_BUILD_CONFIG_ADDRESS_32)
    pOutAudioInBuffer->_bufferU64 = 0;
    pOutAudioInBuffer->_sizeU64 = 0;
#endif
    pOutAudioInBuffer->buffer = buffer;
    pOutAudioInBuffer->size = size;
}

void SetAudioInBufferInfo(AudioInBuffer* pOutAudioInBuffer, void* buffer, size_t bufferSize, size_t dataSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutAudioInBuffer);
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    NN_SDK_REQUIRES_ALIGNED(buffer, AudioInBuffer::AddressAlignment);
    NN_SDK_REQUIRES_GREATER(bufferSize, 0u);
    NN_SDK_REQUIRES_ALIGNED(bufferSize, AudioInBuffer::SizeGranularity);
    NN_SDK_REQUIRES_GREATER(dataSize, 0u);
    NN_SDK_REQUIRES_LESS_EQUAL(dataSize, bufferSize);


#if defined(NN_BUILD_CONFIG_ADDRESS_32)
    pOutAudioInBuffer->_bufferU64 = 0;
    pOutAudioInBuffer->_bufferSizeU64 = 0;
    pOutAudioInBuffer->_sizeU64 = 0;
#endif
    pOutAudioInBuffer->buffer = buffer;
    pOutAudioInBuffer->bufferSize = bufferSize;
    pOutAudioInBuffer->size = dataSize;
}

void* GetAudioInBufferDataPointer(const AudioInBuffer* pAudioInBuffer) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pAudioInBuffer);
    return pAudioInBuffer->buffer;
}

size_t GetAudioInBufferDataSize(const AudioInBuffer* pAudioInBuffer) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pAudioInBuffer);
    return pAudioInBuffer->size;
}

size_t GetAudioInBufferBufferSize(const AudioInBuffer* pAudioInBuffer) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pAudioInBuffer);
    return pAudioInBuffer->bufferSize;
}

bool AppendAudioInBuffer(AudioIn* pAudioIn, AudioInBuffer* pAudioInBuffer) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pAudioIn);
    NN_SDK_REQUIRES_NOT_NULL(pAudioInBuffer);
    NN_SDK_REQUIRES_NOT_NULL(pAudioInBuffer->buffer);
    NN_SDK_REQUIRES_ALIGNED(pAudioInBuffer->buffer, nn::audio::AudioInBuffer::AddressAlignment);
    NN_SDK_REQUIRES_ALIGNED(pAudioInBuffer->bufferSize, nn::audio::AudioInBuffer::SizeGranularity);
    NN_SDK_REQUIRES(pAudioInBuffer->size % (nn::audio::GetAudioInChannelCount(pAudioIn) * nn::audio::GetSampleByteSize(nn::audio::GetAudioInSampleFormat(pAudioIn))) == 0);

    void* handle = g_AudioInMap.FindPair(pAudioIn);
    NN_SDK_ASSERT(handle);
    nn::sf::InBuffer inBuf( reinterpret_cast<char*>(pAudioInBuffer), sizeof( *pAudioInBuffer ) );
    if(strncmp(pAudioIn->name, nn::audio::common::UacDevicePrefix, nn::audio::common::UacDevicePrefixSize) == 0)
    {
        nn::os::TransferMemory transferMemory(pAudioInBuffer->buffer, nn::util::align_up(pAudioInBuffer->size, nn::os::MemoryPageSize), nn::os::MemoryPermission_ReadWrite);
        return static_cast<detail::IAudioIn*>(handle)->AppendUacInBufferAuto(inBuf, reinterpret_cast<uintptr_t>(pAudioInBuffer), nn::sf::NativeHandle(transferMemory.Detach(), true)).IsSuccess();
    }
    else
    {
        const auto result = static_cast<detail::IAudioIn*>(handle)->AppendAudioInBufferAuto(inBuf, reinterpret_cast<uintptr_t>(pAudioInBuffer));

        if(ResultBufferCountMax::Includes(result))
        {
            return false;
        }
        else
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
            return true;
        }
    }

}
int GetAudioInBufferCount(const AudioIn* pAudioIn) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pAudioIn);
    void* handle = g_AudioInMap.FindPair(pAudioIn);
    NN_SDK_REQUIRES_NOT_NULL(handle);
    int bufferCount = 0;
    NN_ABORT_UNLESS_RESULT_SUCCESS(static_cast<detail::IAudioIn*>(handle)->GetAudioInBufferCount(&bufferCount));
    return bufferCount;
}

AudioInBuffer* GetReleasedAudioInBuffer(AudioIn* pAudioIn) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pAudioIn);
    int countRet;
    int count = 1;

    uint64_t audioInArray;
    nn::sf::OutBuffer inArray(reinterpret_cast<char *>(&audioInArray), count*sizeof(audioInArray));

    void* handle = g_AudioInMap.FindPair(pAudioIn);
    NN_SDK_ASSERT(handle);
    static_cast<detail::IAudioIn*>(handle)->GetReleasedAudioInBuffersAuto(inArray, &countRet);

    AudioInBuffer* inBuffer = reinterpret_cast<AudioInBuffer*>(audioInArray);
    return inBuffer;
}

bool ContainsAudioInBuffer(const AudioIn* pAudioIn, const AudioInBuffer* pAudioInBuffer) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pAudioIn);
    NN_SDK_REQUIRES_NOT_NULL(pAudioInBuffer);
    bool contains;

    void* handle = g_AudioInMap.FindPair(pAudioIn);
    NN_SDK_ASSERT(handle);
    NN_ABORT_UNLESS_RESULT_SUCCESS( static_cast<detail::IAudioIn*>(handle)->ContainsAudioInBuffer(reinterpret_cast<uintptr_t>(pAudioInBuffer), &contains) );

    return contains;
}

nn::Result RegisterBufferEvent(nn::audio::AudioIn* pAudioIn, nn::os::SystemEvent* pBufferEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pAudioIn);
    NN_SDK_REQUIRES_NOT_NULL(pBufferEvent);

    nn::sf::NativeHandle systemEventHandle;

    void* handle = g_AudioInMap.FindPair(pAudioIn);
    NN_SDK_ASSERT(handle);
    NN_RESULT_DO(static_cast<detail::IAudioIn*>(handle)->RegisterBufferEvent(&systemEventHandle));

    pBufferEvent->AttachReadableHandle(systemEventHandle.GetOsHandle(), systemEventHandle.IsManaged(), nn::os::EventClearMode_AutoClear);
    systemEventHandle.Detach();
    NN_RESULT_SUCCESS;
}

nn::Result SetAudioInDeviceGain(nn::audio::AudioIn* pAudioIn, float gain) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pAudioIn);
    NN_SDK_REQUIRES_MINMAX(gain, nn::audio::AudioInDeviceGainMin, nn::audio::AudioInDeviceGainMax);
    void* handle = g_AudioInMap.FindPair(pAudioIn);
    NN_SDK_ASSERT(handle);

#if defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK2) || defined(NN_BUILD_CONFIG_HARDWARE_NX)
    NN_RESULT_DO(static_cast<detail::IAudioIn*>(handle)->SetDeviceGain(gain));
    NN_RESULT_SUCCESS;
#else
    NN_UNUSED(gain);
    NN_UNUSED(handle);
    NN_RESULT_THROW(nn::audio::ResultNotSupported());
#endif
}

float GetAudioInDeviceGain(const AudioIn* pAudioIn) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pAudioIn);
    void* handle = g_AudioInMap.FindPair(pAudioIn);
    NN_SDK_ASSERT(handle);

    float gain = nn::audio::AudioInDeviceGainMax;

#if defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK2) || defined(NN_BUILD_CONFIG_HARDWARE_NX)
    NN_ABORT_UNLESS_RESULT_SUCCESS(static_cast<detail::IAudioIn*>(handle)->GetDeviceGain(&gain));
#else
    NN_UNUSED(handle);
#endif

    return gain;

}

} //namespace audio
} //namespace nn
