﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#if defined(__ARM_NEON__) || defined(__ARM_NEON)
#include <arm_neon.h>
#endif // defined(__ARM_NEON__) || defined(__ARM_NEON)

#include <cstdint>
#include <algorithm>
#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>

namespace nn { namespace audio { namespace server { namespace detail {

NN_FORCEINLINE void ApplyUniformGain1(int16_t* output, const int16_t* input, int32_t gain, int sampleCount) NN_NOEXCEPT
{
    const int Q = 15;
    const int gainMax = (1 << Q);

    NN_SDK_ASSERT_LESS_EQUAL(gain, gainMax);
    NN_SDK_ASSERT_GREATER_EQUAL(gain, 0);

    gain = std::min(gainMax, std::max(gain, 0));

    for (auto i = 0; i < sampleCount; ++i)
    {
        output[i] = static_cast<int16_t>((static_cast<int32_t>(input[i]) * gain + (1 << (Q - 1))) >> Q);
    }
}

NN_FORCEINLINE void ApplyUniformGain4(int16_t* output, const int16_t* input, int32_t gain, int sampleCount) NN_NOEXCEPT
{
    const int Q = 15;
    const int gainMax = (1 << Q);

    NN_SDK_ASSERT_LESS_EQUAL(gain, gainMax);
    NN_SDK_ASSERT_GREATER_EQUAL(gain, 0);

    gain = std::min(gainMax, std::max(gain, 0));

#if defined(__ARM_NEON__) || defined(__ARM_NEON)
    int16x4_t gain4 = vdup_n_s16(gain);
    if(gain == gainMax)
    {
        for (int i = sampleCount; i != 0; i -= 4)
        {
            vst1_s16(output, vld1_s16(input));
            input += 4;
            output += 4;
        }
    }
    else
    {
        for (int i = sampleCount; i != 0; i -= 4)
        {
            int16x4_t in4 = vld1_s16(input);
            int32x4_t tmp = vmull_s16(in4, gain4);
            int16x4_t out4 = vqrshrn_n_s32(tmp, Q);
            vst1_s16(output, out4);
            input += 4;
            output += 4;
        }
    }
#else // defined(__ARM_NEON__) || defined(__ARM_NEON)
    ApplyUniformGain1(output, input, gain, sampleCount);
#endif // defined(__ARM_NEON__) || defined(__ARM_NEON)

}

NN_FORCEINLINE void ApplyUniformGain(int16_t* output, const int16_t* input, int32_t gain, int sampleCount) NN_NOEXCEPT
{
    if (!(sampleCount & 0x3))
    {
        return ApplyUniformGain4(output, input, gain, sampleCount);
    }
    else
    {
        return ApplyUniformGain1(output, input, gain, sampleCount);
    }
}

}}}}  // namespace nn::audio::server::detail
