﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_Mutex.h>
#include <nn/os/os_MemoryHeapCommon.h>
#include <nn/audio/audio_AudioInTypes.h>
#include <nn/cduac/cduac_Spec.h>
#include <nn/cduac/cduac_Api.h>
#include <nne/audio/audio.h>
#include "../audio_Mixer.h"
#include "../../common/audio_Util.h"

namespace nn {
namespace audio {
namespace detail {

class UacOutDevice
{
    enum UacReadResult
    {
        UacReadResult_Success,
        UacReadResult_Failure,
        UacReadResult_FailureAlreadyCopiedToUser,
    };

    struct UserBuffer
    {
        UserBuffer()
            : buffer(nullptr)
            , size(0)
            , writePosition(0)
            {}
        UserBuffer(void* _buffer, size_t _size)
            : buffer(_buffer)
            , size(_size)
            , writePosition(0)
        {}
        void* buffer;
        size_t size;
        size_t writePosition;
    };

    class MixBuffer
    {
    public:
        MixBuffer();
        int AppendBytes(void* buffer, int numBytes) NN_NOEXCEPT;
        int GetBytes(void* outBuffer, int numBytes) NN_NOEXCEPT;
        void ReleaseBytes(int numBytes) NN_NOEXCEPT;
        int GetFreeBytes() const NN_NOEXCEPT;
        int GetAppendedBytes() const NN_NOEXCEPT;
        int GetBufferCapacity() const NN_NOEXCEPT;
        int GetEstimatedNextFrameSize(int inputSizeBytes) NN_NOEXCEPT;
        void SetVolume(float volume) NN_NOEXCEPT;
        void Reset() NN_NOEXCEPT;

        bool InitResampler(MixerFormat* inFormat, MixerFormat* outFormat) NN_NOEXCEPT;
        void DisableResampler() NN_NOEXCEPT;
    private:
        static const int ResamplerStateSize = 6008; //Required size for 44.1 -> 48 polyphase SRC

        nn::os::Mutex m_Mutex;
        int8_t m_ResamplerState[ResamplerStateSize];
        MixerFormat* m_pInFormat;
        MixerFormat* m_pOutFormat;
        int32_t m_EstimateSrcOutputCount;
        int32_t m_InputBytesToSamples;
        int32_t m_OutputBytesToSamples;
        int8_t m_Buffer[5760]; //10ms @ 6ch PCM 16 48k
        int16_t m_NumAppendedBytes;
        float m_Volume;
    };

public:
    UacOutDevice() NN_NOEXCEPT;
    static void ThreadFunc(void*);

    static bool IsSupported(nn::cduac::Interface* interface, nn::cduac::Parser* parser) NN_NOEXCEPT;

    bool Update() NN_NOEXCEPT;

    //Plug/Unplug
    void InitializeDevice(const char* deviceName, nn::cduac::Interface* interface, nn::cduac::Host* host, nn::cduac::Parser* parser) NN_NOEXCEPT;
    void FinalizeDevice() NN_NOEXCEPT;

    bool OpenSession(int channelCount) NN_NOEXCEPT;
    void CloseSession() NN_NOEXCEPT;

    //Game Start/Stop
    void StartDevice() NN_NOEXCEPT;
    void StopDevice() NN_NOEXCEPT;

    int AppendBytes(void* buffer, int size) NN_NOEXCEPT;

    bool IsStarted() NN_NOEXCEPT;
    bool IsInitialized() NN_NOEXCEPT;
    const char* GetName() NN_NOEXCEPT;

    int GetSessionId() NN_NOEXCEPT;
    bool SetChannelCount(int channelCount) NN_NOEXCEPT;

    void SetSourceBufferSize(int bufferSize) NN_NOEXCEPT;
    void SetMixVolume(float volume) NN_NOEXCEPT;

    void SetCurrentVolume(float gain) NN_NOEXCEPT;
    void SetCurrentGain(float gain) NN_NOEXCEPT;

    void SetMute(bool mute) NN_NOEXCEPT;

    nn::os::SystemEventType* GetBufferCompletionEvent() NN_NOEXCEPT;
    void ClearBufferCompletionEvent() NN_NOEXCEPT;

private:
    static void SetSamplingFrequency(nn::cduac::AudioSamplingFrequency* outFrequency, uint32_t frequency) NN_NOEXCEPT;
    static int GetEndpointSamplingRateFromFormatType(nn::cduac::AudioStreamingFormatType* format) NN_NOEXCEPT;
    static int SetEndpointSamplingRate(nn::cduac::Interface* interface, nn::cduac::Parser* parser) NN_NOEXCEPT;
    static nn::audio::SampleFormat UacFormatToAudioFormat(uint8_t uacFormat, uint8_t sampleSizeBits) NN_NOEXCEPT;
    static uint8_t FindAltSetting(nn::cduac::Interface* interface, nn::cduac::Parser* parser) NN_NOEXCEPT;

    void StartDeviceInternal() NN_NOEXCEPT;
    void StopDeviceInternal(bool isConnected) NN_NOEXCEPT;
    void SetUacEndpoint() NN_NOEXCEPT;
    void SetUacControls() NN_NOEXCEPT;
    int8_t* ReadDeviceSynchronous(UacReadResult* outResult) NN_NOEXCEPT;
    int8_t* GetNextReadyBuffer() NN_NOEXCEPT;
    void SubmitUacBuffer() NN_NOEXCEPT;
    UacReadResult CheckXferStatus() NN_NOEXCEPT;
    void ResyncUacDevice(bool waitCompletions) NN_NOEXCEPT;
    void SetNextUacFrameNumber() NN_NOEXCEPT;
    void Depop(int16_t* outBuffer, uint32_t sampleCount) NN_NOEXCEPT;
    bool IsDeviceInitialized() NN_NOEXCEPT;

private:
    static const int StackSize = 4096 * 2;
    static const int UacFrameDurationMs = 5;
    static const int UacMaxSamplingRate = 48000;
    static const int UacMaxSamplesPerMs = UacMaxSamplingRate / 1000;
    static const int UacMaxSamplesPerFrame = UacMaxSamplesPerMs * UacFrameDurationMs;
    static const int UacMaxChannelCount = 2;
    static const int UacBufferMaxCount = 3;
    static const int UacSampleSize = 2;
    static const int UacBufferSize = NN_AUDIO_ALIGN_UP(UacMaxSamplesPerFrame * UacMaxChannelCount * UacSampleSize, nn::os::MemoryPageSize);
    static const int UserBufferMaxCount = nne::audio::gmix::NumAudioInBuffers;

    NN_AUDIO_ALIGNAS_UAC_BUFFER int8_t m_UacBuffer[UacBufferMaxCount][UacBufferSize]; // DMA buffers have to be 4K aligned !!!! * 2 to get past IoVa mapping issue
    NN_AUDIO_ALIGNAS_UAC_BUFFER int8_t m_ReportBuffer[nn::os::MemoryPageSize];
    MixBuffer m_MixBuffer;
    Mixer m_Mixer;
    uint32_t m_InputBytes[UacFrameDurationMs];
    nn::usb::XferReport m_XferReport[UacFrameDurationMs * UacBufferMaxCount];
    nn::os::Mutex m_Mutex;
    nn::cduac::Host* m_pHost;
    nn::cduac::Parser* m_pParser;
    nn::cduac::Interface* m_pInterface;
    nn::cduac::AudioControlInputTerminal* m_pInputTerminal;
    nn::cduac::AudioControlFeatureUnit* m_pFeatureUnit;
    nn::os::SystemEventType* m_pStateChangeEvent;
    const char* m_Name;
    MixerFormat m_OutFormat;
    MixerFormat m_InFormat;
    uint32_t m_NextUsbFrame;
    int m_UacBufferIndex;
    int m_UacBufferReadIndex;
    int m_SessionId;
    int m_SourceSize;
    int m_SourceOffset;
    int m_RemainderSample;
    int m_RemainderIntervalMs;
    uint32_t m_OutCount;
    int16_t m_MinVolume[UacMaxChannelCount];
    int16_t m_MaxVolume[UacMaxChannelCount];
    int16_t m_CurrentVolume[UacMaxChannelCount];

    /*
        Layout of m_Controls:
        ch 0: Master
        m_Controls[ch][0]:
            bit 0: Mute
            bit 1: Volume
            bit 2: Bass
            bit 3: Mid
            bit 4: Treble
            bit 5: Graphic Equalizer
            bit 6: Automatic Gain
            bit 7: Delay
        m_Controls[ch][1]:
            bit 0: Bass Boost
            bit 1: Loudness
            bit 2+: Reserved
    */
    int8_t m_Controls[UacMaxChannelCount + 1][2]; //+1 for master channel
    uint8_t m_AltSetting;
    bool m_IsStarted;
    bool m_IsInitialized;
    bool m_IsDeviceInitialized;
    bool m_IsMute[UacMaxChannelCount];
    bool m_IsDeviceStarted;
    bool m_NeedResync;
    bool m_SkipSubmitBuffer;
    float m_MixVolume;
    float m_Gain;
};

}  // namespace detail
}  // namespace audio
}  // namespace nn
