﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "audio_UacManager.h"
#include <nn/os/os_Event.h>
#include <nne/audio/audio.h>
#include <nn/audio/audio_Result.h>
#include <nn/audio/server/audio_FirmwareDebugSettings.h>
#include <nn/audio/detail/audio_Log.h>
#include <nn/nn_SystemThreadDefinition.h>
#include "../audio_AppletVolumeManager.h"

namespace nn {
namespace audio {
namespace detail {

UacManager::UacManager()
    : m_pProcessSemaphore(nullptr)
    , m_pGmixSession(nullptr)
    , m_Mutex(true)
    , m_NumSpeakers(0)
    , m_InInitializedCount(0)
    , m_InStartedCount(0)
    , m_LastGmixPosition(0)
    , m_IsInitialized(false)
    , m_IsAwake(true)
    , m_IsGmixLinked(false)
    , m_CanConnectSpeaker(false)
    , m_SpeakerVolume(1.0f)
{
    memset(m_UpdateThreadStack, 0, sizeof(m_UpdateThreadStack));
    memset(m_AttachThreadStack, 0, sizeof(m_AttachThreadStack));
    for(int i = 0; i < InSessionMax; ++i)
    {
        m_IsMicrophoneLinked[i] = false;
        m_IsMicrophoneUnlinkDeferred[i] = false;
        m_MicrophoneDeviceIdToConnectionId[i] = InvalidConnectionId;
    }

    for(int i = 0; i < OutSessionMax; ++i)
    {
        m_IsSpeakerLinked[i] = false;
        m_IsSpeakerUnlinkDeferred[i] = false;
        m_IsSpeakerLinkDeferred[i] = false;
        m_SpeakerDeviceIdToConnectId[i] = InvalidConnectionId;
    }
}

UacManager::~UacManager()
{
    NN_SDK_REQUIRES(m_InInitializedCount == 0);
    NN_SDK_REQUIRES(m_InStartedCount == 0);

    for(int i = 0; i < InSessionMax; ++i)
    {
        m_MicrophoneDeviceIdToConnectionId[i] = InvalidConnectionId;
    }

    for(int i = 0; i < OutSessionMax; ++i)
    {
        m_SpeakerDeviceIdToConnectId[i] = InvalidConnectionId;
    }
}

UacManager& UacManager::GetInstance() NN_NOEXCEPT
{
    static UacManager instance;
    return instance;
}

nn::os::EventType* UacManager::GetSpeakerAttachEvent() NN_NOEXCEPT
{
    return &m_SpeakerAttachEvent;
}

nn::os::EventType* UacManager::GetSpeakerDetachEvent() NN_NOEXCEPT
{
    return &m_SpeakerDetachEvent;
}

nn::os::EventType* UacManager::GetUnsupportedSpeakerAttachEvent() NN_NOEXCEPT
{
    return &m_UnsupportedSpeakerAttachEvent;
}

int UacManager::GetNumSpeakers() NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(m_Mutex);
    return m_ConnectionManager.GetNumSpeakers();
}

void UacManager::InitializeImpl() NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(m_StateChangeArg.mutex);
    nn::os::InitializeEvent(&m_StateChangeEvent, false, nn::os::EventClearMode_ManualClear);
    nn::os::InitializeEvent(&m_StateChangeCompleteEvent, false, nn::os::EventClearMode_AutoClear);
    nn::os::InitializeEvent(&m_CloseAttachThreadEvent, false, nn::os::EventClearMode_ManualClear);
    nn::os::InitializeEvent(&m_CheckDeferredLinksEvent, false, nn::os::EventClearMode_AutoClear);
    nn::os::InitializeEvent(&m_DeferredLinksCompleteEvent, false, nn::os::EventClearMode_AutoClear);

    nn::os::InitializeMultiWait(&m_UpdateMultiWait);
    nn::os::InitializeMultiWait(&m_AttachMultiWait);

    nn::os::InitializeMultiWaitHolder(&m_StateChangeHolder, &m_StateChangeEvent);
    auto data = m_ConnectionManager.CreateMultiWaitData(UacEventType_StateChange, UacStateChangeType_SetSpeakerSource);
    nn::os::SetMultiWaitHolderUserData(&m_StateChangeHolder, static_cast<uintptr_t>(data));
    nn::os::LinkMultiWaitHolder(&m_UpdateMultiWait, &m_StateChangeHolder);

    nn::os::InitializeMultiWaitHolder(&m_CheckDeferredLinksHolder, &m_CheckDeferredLinksEvent);
    data = m_ConnectionManager.CreateMultiWaitData(UacEventType_CheckDeferredLinks, 0);
    nn::os::SetMultiWaitHolderUserData(&m_CheckDeferredLinksHolder, static_cast<uintptr_t>(data));
    nn::os::LinkMultiWaitHolder(&m_UpdateMultiWait, &m_CheckDeferredLinksHolder);

    m_ConnectionManager.SetAttachEvent(&m_AttachEvent);
    nn::os::InitializeMultiWaitHolder(&m_AttachHolder, &m_AttachEvent);
    nn::os::SetMultiWaitHolderUserData(&m_AttachHolder, static_cast<uintptr_t>(UacEventType_Attach));
    nn::os::LinkMultiWaitHolder(&m_AttachMultiWait, &m_AttachHolder);

    nn::os::InitializeEvent(&m_SpeakerAttachEvent, false, nn::os::EventClearMode_AutoClear);
    nn::os::InitializeEvent(&m_SpeakerDetachEvent, false, nn::os::EventClearMode_AutoClear);
    nn::os::InitializeEvent(&m_UnsupportedSpeakerAttachEvent, false, nn::os::EventClearMode_AutoClear);
    m_ConnectionManager.SetUnsupportedSpeakerAttachEvent(&m_UnsupportedSpeakerAttachEvent);

    nn::os::InitializeMultiWaitHolder(&m_CloseAttachHolder, &m_CloseAttachThreadEvent);
    nn::os::SetMultiWaitHolderUserData(&m_CloseAttachHolder, static_cast<uintptr_t>(UacEventType_CloseAttachThread));
    nn::os::LinkMultiWaitHolder(&m_AttachMultiWait, &m_CloseAttachHolder);

    m_IsInitialized = true;
    SetSpeakerSourceImpl(nullptr, nullptr);
    StartThread();
}

void UacManager::FinalizeImpl() NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(m_StateChangeArg.mutex);
    m_IsInitialized = false;
    StopThread();

    UnlinkAllDevices();
    UnlinkProcessSemaphore();
    DisconnectAllSpeakers();
    m_ConnectionManager.DetachAllDevices();

    nn::os::UnlinkMultiWaitHolder(&m_AttachHolder);
    nn::os::FinalizeMultiWaitHolder(&m_AttachHolder);

    nn::os::UnlinkMultiWaitHolder(&m_StateChangeHolder);
    nn::os::FinalizeMultiWaitHolder(&m_StateChangeHolder);

    nn::os::UnlinkMultiWaitHolder(&m_CloseAttachHolder);
    nn::os::FinalizeMultiWaitHolder(&m_CloseAttachHolder);

    nn::os::UnlinkMultiWaitHolder(&m_CheckDeferredLinksHolder);
    nn::os::FinalizeMultiWaitHolder(&m_CheckDeferredLinksHolder);

    nn::os::FinalizeEvent(&m_SpeakerAttachEvent);
    nn::os::FinalizeEvent(&m_SpeakerDetachEvent);
    nn::os::FinalizeEvent(&m_StateChangeEvent);
    nn::os::FinalizeEvent(&m_CloseAttachThreadEvent);
    nn::os::FinalizeEvent(&m_CheckDeferredLinksEvent);
    nn::os::FinalizeEvent(&m_DeferredLinksCompleteEvent);
    nn::os::FinalizeEvent(&m_StateChangeCompleteEvent);
    nn::os::FinalizeEvent(&m_UnsupportedSpeakerAttachEvent);

    m_ConnectionManager.ClearAttachEvent(&m_AttachEvent);

    nn::os::FinalizeMultiWait(&m_UpdateMultiWait);
    nn::os::FinalizeMultiWait(&m_AttachMultiWait);
}


void UacManager::Initialize() NN_NOEXCEPT
{
    if(!nn::audio::server::IsUacEnabled())
    {
        return;
    }
    m_ConnectionManager.Initialize();
    InitializeImpl();
}

void UacManager::Finalize() NN_NOEXCEPT
{
    if(!nn::audio::server::IsUacEnabled())
    {
        return;
    }
    FinalizeImpl();
    m_ConnectionManager.Finalize();
}

void UacManager::Sleep() NN_NOEXCEPT
{
    if(!nn::audio::server::IsUacEnabled())
    {
        return;
    }
    FinalizeImpl();
    m_ConnectionManager.Sleep();
}

void UacManager::Wake() NN_NOEXCEPT
{
    if(!nn::audio::server::IsUacEnabled())
    {
        return;
    }
    m_ConnectionManager.Wake();
    InitializeImpl();
}

//Called by AudioInSessionAbstractor Open/Close
nn::Result UacManager::OpenUacInSession(UacSessionId* outUacId, const char* pDviceName, server::SessionFormat& format, int sessionId, const nn::applet::AppletResourceUserId& aruid) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(m_StateChangeArg.mutex);
    auto arg = &m_StateChangeArg.parameters.openMicrophone;

    arg->format = format;
    arg->name = pDviceName;
    arg->aruid = &aruid;
    arg->sessionId = sessionId;
    arg->returnId = InvalidUacId;
    m_StateChangeArg.reason = StateChangeReason_OpenMicrophone;
    DoSynchronizedStateChange();

    *outUacId = arg->returnId;
    if(arg->returnId == InvalidUacId)
    {
        NN_DETAIL_AUDIO_ERROR("Device not found (name: %s)!!\n", pDviceName);
        NN_RESULT_THROW(ResultNoAudioDeviceFound());
    }
    NN_RESULT_SUCCESS;
}

nn::Result UacManager::CloseUacInSession(UacSessionId uacId) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(m_StateChangeArg.mutex);
    m_StateChangeArg.uacId = uacId;
    m_StateChangeArg.reason = StateChangeReason_CloseMicrophone;

    DoSynchronizedStateChange();
    NN_RESULT_SUCCESS;
}

UacSessionId UacManager::OpenUacInSessionImpl(const char* pDviceName, server::SessionFormat& format, int sessionId, const nn::applet::AppletResourceUserId& aruid) NN_NOEXCEPT
{
    auto connectionId = m_ConnectionManager.FindConnection(pDviceName, UacConnectionType_Microphone);
    if(connectionId != InvalidConnectionId)
    {
        for(int i = 0; i < InSessionMax; ++i)
        {
            if(m_InDevice[i].CanOpen())
            {
                m_InDevice[i].InitializeDevice(m_ConnectionManager.GetName(connectionId),
                                               m_ConnectionManager.GetInterface(connectionId),
                                               m_ConnectionManager.GetHost(),
                                               m_ConnectionManager.GetParser(),
                                               aruid);
                m_InDevice[i].OpenSession(format, sessionId);
                m_MicrophoneDeviceIdToConnectionId[i] = connectionId;
                return i;
            }
        }
    }
    return InvalidUacId;
}

void UacManager::CloseUacInSessionImpl(UacSessionId uacId) NN_NOEXCEPT
{
    if(m_InDevice[uacId].IsStarted())
    {
        m_InDevice[uacId].StopDevice();
    }

    m_InDevice[uacId].CloseSession();
    m_InDevice[uacId].FinalizeDevice();
    UnlinkMicrophone(uacId, false);
    m_MicrophoneDeviceIdToConnectionId[uacId] = InvalidConnectionId;
}

nn::Result UacManager::StartUacInSession(UacSessionId uacId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(uacId >= 0 && uacId < InSessionMax);
    std::lock_guard<os::Mutex> lock (m_StateChangeArg.mutex);
    m_StateChangeArg.uacId = uacId;
    m_StateChangeArg.reason = StateChangeReason_StartMicrophone;
    DoSynchronizedStateChange();

    NN_RESULT_SUCCESS;
}

nn::Result UacManager::StopUacInSession(UacSessionId uacId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(uacId >= 0 && uacId < InSessionMax);
    std::lock_guard<os::Mutex> lock (m_StateChangeArg.mutex);
    m_StateChangeArg.uacId = uacId;
    m_StateChangeArg.reason = StateChangeReason_StopMicrophone;
    DoSynchronizedStateChange();

    NN_RESULT_SUCCESS;
}

void UacManager::StartUacInSessionImpl(UacSessionId uacId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(uacId >= 0 && uacId < InSessionMax);
    if(m_InDevice[uacId].IsInitialized())
    {
        m_InDevice[uacId].StartDevice();
        LinkMicrophone(uacId);
        ++m_InStartedCount;
    }
}

void UacManager::StopUacInSessionImpl(UacSessionId uacId) NN_NOEXCEPT
{
    if(m_InDevice[uacId].IsStarted())
    {
        UnlinkMicrophone(uacId, false);
        m_InDevice[uacId].StopDevice();
        --m_InStartedCount;
    }
}

UacInDevice* UacManager::GetUacInDevice(UacSessionId uacId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(uacId >= 0 && uacId < InSessionMax);
    if(uacId >= 0 && uacId < InSessionMax)
    {
        return &m_InDevice[uacId];
    }
    return nullptr;
}

bool UacManager::IsUacInDeviceSupported(nn::cduac::Interface* pInterface, nn::cduac::Parser* pParser) NN_NOEXCEPT
{
    return UacInDevice::IsSupported(pInterface, pParser);
}

bool UacManager::IsUacOutDeviceSupported(nn::cduac::Interface* pInterface, nn::cduac::Parser* pParser) NN_NOEXCEPT
{
    return UacOutDevice::IsSupported(pInterface, pParser);
}

int UacManager::ListUacIns(AudioInInfo* outAudioIns, int count) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(m_Mutex);
    int remain = count;
    int index = 0;
    for(int i = 0; i < UacConnectionManager::MaxConnections && remain > 0; ++i)
    {
        auto name = m_ConnectionManager.GetName(i);
        if(name && m_ConnectionManager.IsMicrophone(i))
        {
            strncpy(outAudioIns[index].name, name, nn::audio::AudioIn::NameLength);
            ++index;
            --remain;
        }
    }
    return index;
}

bool UacManager::IsUacInName(const char* name) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(m_Mutex);
    for(int i = 0; i < UacConnectionManager::MaxConnections; ++i)
    {
        auto connectionName = m_ConnectionManager.GetName(i);
        if(connectionName && m_ConnectionManager.IsMicrophone(i) && strcmp(name, connectionName) == 0)
        {
            return true;
        }
    }
    return false;
}

void UacManager::SetSpeakerSource(nne::audio::gmix::Session* session, nn::os::Semaphore* semaphore) NN_NOEXCEPT
{
    if(nn::audio::server::IsUacEnabled())
    {
        std::lock_guard<os::Mutex> lock(m_StateChangeArg.mutex);
        auto arg = &m_StateChangeArg.parameters.setSource;
        arg->session = session;
        arg->semaphore = semaphore;
        m_StateChangeArg.reason = StateChangeReason_SetSpeakerSource;
        if(m_pGmixSession == session)
        {
            NN_SDK_ASSERT(semaphore == m_pProcessSemaphore);
            return;
        }
        DoSynchronizedStateChange();
    }
}

void UacManager::DoSynchronizedStateChange() NN_NOEXCEPT
{
    if(m_IsInitialized)
    {
        nn::os::SignalEvent(&m_StateChangeEvent);
        nn::os::WaitEvent(&m_StateChangeCompleteEvent);
    }
}

void UacManager::StartThread() NN_NOEXCEPT
{
    nn::os::CreateThread(&m_UpdateThread, DeviceManagerUpdateThreadFunc, this, m_UpdateThreadStack, sizeof(m_UpdateThreadStack), NN_SYSTEM_THREAD_PRIORITY(audio, AudioUacUpdateManager));
    nn::os::SetThreadNamePointer(&m_UpdateThread, NN_SYSTEM_THREAD_NAME(audio, AudioUacUpdateManager));
    nn::os::StartThread(&m_UpdateThread);

    nn::os::CreateThread(&m_AttachThread, DeviceManagerAttachThreadFunc, this, m_AttachThreadStack, sizeof(m_AttachThreadStack), NN_SYSTEM_THREAD_PRIORITY(audio, AudioUacAttachManager));
    nn::os::SetThreadNamePointer(&m_AttachThread, NN_SYSTEM_THREAD_NAME(audio, AudioUacAttachManager));
    nn::os::StartThread(&m_AttachThread);
}

void UacManager::StopThread() NN_NOEXCEPT
{
    nn::os::SignalEvent(&m_CloseAttachThreadEvent);
    nn::os::WaitThread(&m_AttachThread);
    nn::os::DestroyThread(&m_AttachThread);

    nn::os::WaitThread(&m_UpdateThread);
    nn::os::DestroyThread(&m_UpdateThread);
}

void UacManager::UpdateUnlinkedMicrophones(UacSessionId id) NN_NOEXCEPT
{
    for(int i = 0; i < InSessionMax; ++i)
    {
        if(i == id)
        {
            continue;
        }
        if(!m_IsMicrophoneLinked[i] && m_InDevice[i].IsStarted())
        {
            UpdateMicrophone(i);
        }
    }
}

void UacManager::UpdateMicrophone(UacSessionId id) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(id >=0 && id < InSessionMax);
    auto& device = m_InDevice[id];

    auto running = device.Update();

    auto connectionId = m_MicrophoneDeviceIdToConnectionId[id];
    auto connected = m_ConnectionManager.IsConnected(connectionId);
    if(connected)
    {
        //Device is connected. Make sure it is linked to multiwait.
        if(running)
        {
            LinkMicrophone(id);
        }
        else
        {
            UnlinkMicrophone(id, false);
        }
    }
    else
    {
        //Device is disconnected. Unlink from multiwait.
        UnlinkMicrophone(id, false);
    }
}

void UacManager::UpdateSpeaker(UacSessionId id) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(id >=0 && id < OutSessionMax);
    auto& device = m_OutDevice[id];

    auto running = device.Update();

    auto connectionId = m_SpeakerDeviceIdToConnectId[id];
    auto connected = m_ConnectionManager.IsConnected(connectionId);
    if(connected)
    {
        //Device is connected. Make sure it is linked to multiwait.
        if(running)
        {
            LinkSpeaker(id, false);
        }
        else
        {
            UnlinkSpeaker(id, false);
        }
    }
    else
    {
        //Device is disconnected. Unlink from multiwait.
        UnlinkSpeaker(id, false);
    }
}

void UacManager::LinkMicrophone(UacSessionId id) NN_NOEXCEPT
{
    if(!m_IsMicrophoneLinked[id])
    {
        auto completionEvent = m_InDevice[id].GetBufferCompletionEvent();
        if(completionEvent)
        {
            nn::os::InitializeMultiWaitHolder(&m_MicrophoneUpdateHolder[id], completionEvent);
            auto data = m_ConnectionManager.CreateMultiWaitData(UacEventType_MicrophoneUpdate, id);
            nn::os::SetMultiWaitHolderUserData(&m_MicrophoneUpdateHolder[id], data);
            nn::os::LinkMultiWaitHolder(&m_UpdateMultiWait, &m_MicrophoneUpdateHolder[id]);
            m_IsMicrophoneLinked[id] = true;
        }
    }
}

void UacManager::UnlinkMicrophoneImpl(UacSessionId id) NN_NOEXCEPT
{
    nn::os::UnlinkMultiWaitHolder(&m_MicrophoneUpdateHolder[id]);
    nn::os::FinalizeMultiWaitHolder(&m_MicrophoneUpdateHolder[id]);
    m_IsMicrophoneLinked[id] = false;
    m_IsMicrophoneUnlinkDeferred[id] = false;
}

bool UacManager::UnlinkMicrophone(UacSessionId id, bool defer) NN_NOEXCEPT
{
    bool shouldWait = false;
    if(m_IsMicrophoneLinked[id] && !m_IsMicrophoneUnlinkDeferred[id])
    {
        if(defer)
        {
            m_IsMicrophoneUnlinkDeferred[id] = true;
            nn::os::SignalEvent(&m_CheckDeferredLinksEvent);
            shouldWait = true;
        }
        else
        {
            UnlinkMicrophoneImpl(id);
        }
    }
    return shouldWait;
}

void UacManager::LinkSpeakerImpl(UacSessionId id) NN_NOEXCEPT
{
    auto completionEvent = m_OutDevice[id].GetBufferCompletionEvent();
    if(completionEvent)
    {
        nn::os::ClearSystemEvent(completionEvent);
        nn::os::InitializeMultiWaitHolder(&m_SpeakerUpdateHolder[id], completionEvent);
        auto data = m_ConnectionManager.CreateMultiWaitData(UacEventType_SpeakerUpdate, id);
        nn::os::SetMultiWaitHolderUserData(&m_SpeakerUpdateHolder[id], data);
        nn::os::LinkMultiWaitHolder(&m_UpdateMultiWait, &m_SpeakerUpdateHolder[id]);
        m_IsSpeakerLinked[id] = true;
    }
    m_IsSpeakerLinkDeferred[id] = false;
}

bool UacManager::LinkSpeaker(UacSessionId id, bool defer) NN_NOEXCEPT
{
    bool shouldWait = false;
    if(!m_IsSpeakerLinked[id] && !m_IsSpeakerLinkDeferred[id] && !m_IsSpeakerUnlinkDeferred[id])
    {
        if(defer)
        {
            m_IsSpeakerLinkDeferred[id] = true;
            nn::os::SignalEvent(&m_CheckDeferredLinksEvent);
            shouldWait = true;
        }
        {
            LinkSpeakerImpl(id);
        }
    }
    return shouldWait;
}

void UacManager::UnlinkSpeakerImpl(UacSessionId id) NN_NOEXCEPT
{
    nn::os::UnlinkMultiWaitHolder(&m_SpeakerUpdateHolder[id]);
    nn::os::FinalizeMultiWaitHolder(&m_SpeakerUpdateHolder[id]);
    m_IsSpeakerLinked[id] = false;
    m_IsSpeakerUnlinkDeferred[id] = false;
}

bool UacManager::UnlinkSpeaker(UacSessionId id, bool defer) NN_NOEXCEPT
{
    auto shouldWait = false;
    if(m_IsSpeakerLinked[id] && !m_IsSpeakerLinkDeferred[id] && !m_IsSpeakerUnlinkDeferred[id])
    {
        if(defer)
        {
            m_IsSpeakerUnlinkDeferred[id] = true;
            nn::os::SignalEvent(&m_CheckDeferredLinksEvent);
            shouldWait = true;
        }
        else
        {
            UnlinkSpeakerImpl(id);
        }
    }
    return shouldWait;
}

void UacManager::LinkProcessSemaphore() NN_NOEXCEPT
{
    if(!m_IsGmixLinked && m_pProcessSemaphore)
    {
        nn::os::InitializeMultiWaitHolder(&m_GmixProcessHolder, m_pProcessSemaphore->GetBase());
        auto data = m_ConnectionManager.CreateMultiWaitData(UacEventType_ProcessGmix, 0);
        nn::os::SetMultiWaitHolderUserData(&m_GmixProcessHolder, data);
        nn::os::LinkMultiWaitHolder(&m_UpdateMultiWait, &m_GmixProcessHolder);
        m_IsGmixLinked = true;
    }
}

void UacManager::UnlinkProcessSemaphore() NN_NOEXCEPT
{
    if(m_IsGmixLinked)
    {
        nn::os::UnlinkMultiWaitHolder(&m_GmixProcessHolder);
        nn::os::FinalizeMultiWaitHolder(&m_GmixProcessHolder);
        m_IsGmixLinked = false;
    }
}

void UacManager::PrintStatus() NN_NOEXCEPT
{
    for(int i = 0; i < InSessionMax; ++i)
    {
        auto& device = m_InDevice[i];
        NN_UNUSED(device);
        NN_DETAIL_AUDIO_INFO("[%d] connected: %d, started: %d, linked: %d connectId: %d\n",
                    i, device.IsInitialized(), device.IsStarted(), m_IsMicrophoneLinked[i], m_MicrophoneDeviceIdToConnectionId[i]);
    }
    for(int i = 0; i < OutSessionMax; ++i)
    {
        auto& device = m_OutDevice[i];
        NN_UNUSED(device);
        NN_DETAIL_AUDIO_INFO("[%d] connected: %d, started: %d, linked: %d connectId: %d\n",
                    i, device.IsInitialized(), device.IsStarted(), m_IsSpeakerLinked[i], m_SpeakerDeviceIdToConnectId[i]);
    }
}

void UacManager::DeviceManagerUpdateThreadFunc(void* arg)
{
    auto manager = reinterpret_cast<UacManager*>(arg);
    manager->DeviceManagerUpdateThreadFuncImpl();
}

void UacManager::DeviceManagerAttachThreadFunc(void* arg)
{
    auto manager = reinterpret_cast<UacManager*>(arg);
    manager->DeviceManagerAttachThreadFuncImpl();
}

void UacManager::HandleMicrophoneUpdate(int id) NN_NOEXCEPT
{
    if(id != InvalidUacId)
    {
        m_InDevice[id].ClearBufferCompletionEvent();
        UpdateMicrophone(id);
    }
    else
    {
        UpdateUnlinkedMicrophones(id);
    }

}

void UacManager::HandleSpeakerUpdate(int id) NN_NOEXCEPT
{
    m_OutDevice[id].ClearBufferCompletionEvent();
    UpdateSpeaker(id);
}

bool UacManager::ConnectSpeaker(bool defer) NN_NOEXCEPT
{
    bool shouldWait = false;
    if(!m_CanConnectSpeaker)
    {
        return false;
    }
    for(int i = 0; i < OutSessionMax; ++i)
    {
        auto& device = m_OutDevice[i];
        if(!device.IsStarted())
        {
            auto connectionId = m_ConnectionManager.FindUnusedConnection(UacConnectionType_Speaker);
            if(connectionId != InvalidConnectionId)
            {
                device.InitializeDevice("usbOut",
                                        m_ConnectionManager.GetInterface(connectionId),
                                        m_ConnectionManager.GetHost(),
                                        m_ConnectionManager.GetParser());
                device.OpenSession(2 /*GetOutputChannelCount*/);
                device.StartDevice();
                m_SpeakerDeviceIdToConnectId[i] = connectionId;
                m_ConnectionManager.Open(connectionId);
                {
                    std::lock_guard<os::Mutex> lock(m_Mutex);
                    shouldWait = shouldWait || LinkSpeaker(i, defer);
                }
                ++m_NumSpeakers;
            }
        }
    }
    return shouldWait;
}

bool UacManager::DisconnectSpeaker(ConnectionId connectionId, bool defer) NN_NOEXCEPT
{
    auto shouldWait = false;
    for(int i = 0; i < OutSessionMax; ++i)
    {
        if(m_SpeakerDeviceIdToConnectId[i] == connectionId)
        {
            m_ConnectionManager.Close(connectionId);
            auto& device = m_OutDevice[i];
            device.StopDevice();
            device.CloseSession();
            device.FinalizeDevice();
            shouldWait = UnlinkSpeaker(i, defer);
            --m_NumSpeakers;

            m_SpeakerDeviceIdToConnectId[i] = InvalidConnectionId;

            if(m_ConnectionManager.GetNumSpeakers() == 1)
            {
                nn::os::SignalEvent(&m_SpeakerDetachEvent);
            }
        }
    }
    return shouldWait;
}

void UacManager::DisconnectAllSpeakers() NN_NOEXCEPT
{
    for(int i = 0; i < OutSessionMax; ++i)
    {
        auto& device = m_OutDevice[i];
        if(device.IsStarted())
        {
            auto connectionId = m_SpeakerDeviceIdToConnectId[i];
            DisconnectSpeaker(connectionId, false);
        }
    }
}

void UacManager::SetSpeakerSourceImpl(nne::audio::gmix::Session* session, nn::os::Semaphore* semaphore) NN_NOEXCEPT
{
    if(!nn::audio::server::IsUacEnabled())
    {
        return;
    }

    if(m_pProcessSemaphore)
    {
        UnlinkProcessSemaphore();
    }

    m_pProcessSemaphore = semaphore;
    m_pGmixSession = session;

    if(m_pProcessSemaphore && m_pGmixSession)
    {
        LinkProcessSemaphore();
        m_CanConnectSpeaker = true;
        ConnectSpeaker(false);
        auto bufSize = m_pGmixSession->GetBufferSize();

        for(int i = 0; i < OutSessionMax; ++i)
        {
            m_OutDevice[i].SetSourceBufferSize(bufSize);
        }
    }
    else
    {
        DisconnectAllSpeakers();
        UnlinkProcessSemaphore();
        m_CanConnectSpeaker = false;
    }
}

bool UacManager::HandleAttach() NN_NOEXCEPT
{
    nn::os::ClearSystemEvent(&m_AttachEvent);
    m_ConnectionManager.AttachDevice(&m_AttachMultiWait);
    auto shouldWait = ConnectSpeaker(true);

    nn::os::SignalEvent(&m_SpeakerAttachEvent);
    return shouldWait;
}

bool UacManager::HandleDetach(int id) NN_NOEXCEPT
{
    auto shouldWaitForDeferredLinks = false;
    if(m_ConnectionManager.IsSpeaker(id))
    {
        shouldWaitForDeferredLinks = shouldWaitForDeferredLinks || DisconnectSpeaker(id, true);
    }
    if(m_ConnectionManager.IsMicrophone(id))
    {
        for(int i = 0; i < InSessionMax; ++i)
        {
            if(id == m_MicrophoneDeviceIdToConnectionId[i])
            {
                shouldWaitForDeferredLinks = shouldWaitForDeferredLinks || UnlinkMicrophone(i, true);
                m_InDevice[i].FinalizeDevice();
            }
        }
    }
    m_ConnectionManager.DetachDevice(id);

    return shouldWaitForDeferredLinks;
}

void UacManager::HandleGmixProcess() NN_NOEXCEPT
{
    if(!m_pProcessSemaphore)
    {
        return;
    }

    m_pProcessSemaphore->TryAcquire();
    if(!m_pGmixSession)
    {
        return;
    }
    auto pos = m_pGmixSession->GetWritePosition();
    auto buffer = reinterpret_cast<int8_t*>(m_pGmixSession->GetBufferAddress());

    if(!buffer)
    {
        return;
    }
    auto totalSize = 0;
    if(pos > m_LastGmixPosition)
    {
        auto size = pos - m_LastGmixPosition;
        totalSize += size;
        for(int i = 0; i < OutSessionMax; ++i)
        {
            if(m_OutDevice[i].IsStarted())
            {
                m_OutDevice[i].AppendBytes(&buffer[m_LastGmixPosition], size);
            }
        }
    }
    else if(pos < m_LastGmixPosition)
    {
        auto sizeBeforeWrap = m_pGmixSession->GetBufferSize() - m_LastGmixPosition;
        totalSize += pos + sizeBeforeWrap;

        for(int i = 0; i < OutSessionMax; ++i)
        {
            if(m_OutDevice[i].IsStarted())
            {
                m_OutDevice[i].AppendBytes(&buffer[m_LastGmixPosition], sizeBeforeWrap);
                m_OutDevice[i].AppendBytes(&buffer[0], pos);
            }
        }
    }
    m_LastGmixPosition = pos;
}

void UacManager::HandleStateChange() NN_NOEXCEPT
{
    nn::os::ClearEvent(&m_StateChangeEvent);
    switch(m_StateChangeArg.reason)
    {
    case StateChangeReason_SetSpeakerSource:
    {
        auto arg = &m_StateChangeArg.parameters.setSource;
        SetSpeakerSourceImpl(arg->session, arg->semaphore);
    }
        break;
    case StateChangeReason_StartMicrophone:
        StartUacInSessionImpl(m_StateChangeArg.uacId);
        break;
    case StateChangeReason_StopMicrophone:
        StopUacInSessionImpl(m_StateChangeArg.uacId);
        break;
    case StateChangeReason_OpenMicrophone:
    {
        auto arg = &m_StateChangeArg.parameters.openMicrophone;
        arg->returnId = OpenUacInSessionImpl(arg->name, arg->format, arg->sessionId, *arg->aruid);
    }
        break;
    case StateChangeReason_CloseMicrophone:
        CloseUacInSessionImpl(m_StateChangeArg.uacId);
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
    nn::os::SignalEvent(&m_StateChangeCompleteEvent);
}

void UacManager::HandleDeferredLinks() NN_NOEXCEPT
{
    nn::os::ClearEvent(&m_CheckDeferredLinksEvent);
    for(int i = 0; i < InSessionMax; ++i)
    {
        if(m_IsMicrophoneUnlinkDeferred[i])
        {
            NN_SDK_ASSERT(m_IsMicrophoneLinked[i]);
            UnlinkMicrophoneImpl(i);
        }
    }
    for(int i = 0; i < OutSessionMax; ++i)
    {
        if(m_IsSpeakerLinkDeferred[i])
        {
            NN_SDK_ASSERT(!m_IsSpeakerLinked[i] && !m_IsSpeakerUnlinkDeferred[i]);
            LinkSpeakerImpl(i);
        }
    }
    for(int i = 0; i < OutSessionMax; ++i)
    {
        if(m_IsSpeakerUnlinkDeferred[i])
        {
            NN_SDK_ASSERT(m_IsSpeakerLinked[i] && !m_IsSpeakerLinkDeferred[i]);
            UnlinkSpeakerImpl(i);
        }
    }
    nn::os::SignalEvent(&m_DeferredLinksCompleteEvent);
}

void UacManager::DeviceManagerUpdateThreadFuncImpl() NN_NOEXCEPT
{
    static const nn::TimeSpan Timeout = nn::TimeSpan::FromMilliSeconds(5);
    nn::os::Tick nextIdleUpdate = nn::os::GetSystemTick() + nn::os::ConvertToTick(Timeout);

    while(m_IsInitialized)
    {
        nn::os::MultiWaitHolderType* holder = nullptr;
        holder = nn::os::TimedWaitAny(&m_UpdateMultiWait, Timeout);
        {
            std::lock_guard<os::Mutex> lock(m_Mutex);
            if(!holder || nn::os::GetSystemTick() > nextIdleUpdate)
            {
                HandleMicrophoneUpdate(InvalidUacId);
                nextIdleUpdate = nn::os::GetSystemTick() + nn::os::ConvertToTick(Timeout);
            }

            if(holder)
            {
                auto data = nn::os::GetMultiWaitHolderUserData(holder);
                auto id = m_ConnectionManager.GetIdFromEventData(data);
                auto type = m_ConnectionManager.GetTypeFromEventData(data);
                switch (type)
                {
                    case UacEventType_MicrophoneUpdate:
                        HandleMicrophoneUpdate(id);
                        break;
                    case UacEventType_SpeakerUpdate:
                        HandleSpeakerUpdate(id);
                        break;
                    case UacEventType_ProcessGmix:
                        //Should this be moved to a higher priority thread thread?
                        //Speakers will skip if this is delayed by more than 5ms.
                        HandleGmixProcess();
                        break;
                    case UacEventType_StateChange:
                        HandleStateChange();
                        break;
                    case UacEventType_CheckDeferredLinks:
                        HandleDeferredLinks();
                        break;
                    default:
                        NN_UNEXPECTED_DEFAULT;
                }
            }
        }
    }
}

void UacManager::DeviceManagerAttachThreadFuncImpl() NN_NOEXCEPT
{
    while(m_IsInitialized)
    {
        nn::os::MultiWaitHolderType* holder = nullptr;
        holder = nn::os::WaitAny(&m_AttachMultiWait);
        {
            if(holder)
            {
                auto data = nn::os::GetMultiWaitHolderUserData(holder);
                auto id = m_ConnectionManager.GetIdFromEventData(data);
                auto type = m_ConnectionManager.GetTypeFromEventData(data);
                bool shouldWait = false;
                switch (type)
                {
                    case UacEventType_Attach:
                        shouldWait = HandleAttach();
                        if(shouldWait)
                        {
                            nn::os::WaitEvent(&m_DeferredLinksCompleteEvent);
                        }
                        break;
                    case UacEventType_Detach:
                    {
                        shouldWait = false;
                        {
                            std::lock_guard<os::Mutex> lock(m_Mutex);
                            shouldWait = HandleDetach(id);
                        }
                        if(shouldWait)
                        {
                            nn::os::WaitEvent(&m_DeferredLinksCompleteEvent);
                        }
                        {
                            std::lock_guard<os::Mutex> lock(m_Mutex);
                            ConnectSpeaker(true);
                        }
                        break;
                    }
                    case UacEventType_CloseAttachThread:
                        break;
                    default:
                        NN_UNEXPECTED_DEFAULT;
                }
            }
        }
    }
}

void UacManager::SetSpeakerVolume(float volume) NN_NOEXCEPT
{
    m_SpeakerVolume = volume;

    for(int i = 0; i < OutSessionMax; ++i)
    {
        auto& device = m_OutDevice[i];
        device.SetMixVolume(m_SpeakerVolume);
    }
}

void UacManager::UnlinkAllDevices() NN_NOEXCEPT
{
    for(int i = 0; i < InSessionMax; ++i)
    {
        UnlinkMicrophone(i, false);
    }

    for(int i = 0; i < OutSessionMax; ++i)
    {
        UnlinkSpeaker(i, false);
    }
}

}  // namespace detail
}  // namespace audio
}  // namespace nn


