﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/os/os_Event.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_MultipleWait.h>
#include <nn/audio/audio_AudioInTypes.h>
#include <nn/cduac/cduac_Spec.h>
#include <nn/cduac/cduac_Api.h>
#include <nne/audio/audio.h>
#include "audio_UacTypes.h"

namespace nn {
namespace audio {
namespace detail {

class UacConnectionManager
{
public:
    static const int MaxConnections = nn::usb::HsLimitMaxInterfacesPerClientCount;
    static const int EventIdShift = 0;
    static const int EventTypeShift = 16;
    static const int EventIdMask = 0x0000FFFF;

public:
    UacConnectionManager() NN_NOEXCEPT;
    void Initialize() NN_NOEXCEPT;
    void Finalize() NN_NOEXCEPT;
    void Sleep() NN_NOEXCEPT;
    void Wake()  NN_NOEXCEPT;

    void SetAttachEvent(nn::os::SystemEventType* event) NN_NOEXCEPT;
    void ClearAttachEvent(nn::os::SystemEventType* event) NN_NOEXCEPT;
    void AttachDevice(nn::os::MultiWaitType* multiWait) NN_NOEXCEPT;
    void DetachDevice(int id) NN_NOEXCEPT;
    void DetachAllDevices() NN_NOEXCEPT;
    void SetUnsupportedSpeakerAttachEvent(nn::os::EventType* event) NN_NOEXCEPT;

    void Open(ConnectionId id) NN_NOEXCEPT;
    void Close(ConnectionId id) NN_NOEXCEPT;

    bool IsConnected(ConnectionId id) NN_NOEXCEPT;
    bool IsOpen(ConnectionId id) NN_NOEXCEPT;
    bool IsMicrophone(ConnectionId id) NN_NOEXCEPT;
    bool IsSpeaker(ConnectionId id) NN_NOEXCEPT;
    const char* GetName(ConnectionId id) NN_NOEXCEPT;
    int GetConnectionCount(UacConnectionType type) NN_NOEXCEPT;

    ConnectionId FindConnection(const char* name, UacConnectionType type) NN_NOEXCEPT;
    ConnectionId FindUnusedConnection(UacConnectionType type) NN_NOEXCEPT;
    nn::cduac::Host* GetHost() NN_NOEXCEPT;
    nn::cduac::Parser* GetParser() NN_NOEXCEPT;
    nn::cduac::Interface* GetInterface(ConnectionId id) NN_NOEXCEPT;

    void PrintStatus(const char* caller);

    uintptr_t CreateMultiWaitData(UacEventType type, int id) NN_NOEXCEPT;
    int GetIdFromEventData(uintptr_t data) NN_NOEXCEPT;
    int GetTypeFromEventData(uintptr_t data) NN_NOEXCEPT;
    bool IsInput(const nn::cduac::AudioControlInputTerminal* pInputTerminal, const nn::cduac::AudioControlOutputTerminal* pOutputTerminal) NN_NOEXCEPT;
    bool IsOutput(const nn::cduac::AudioControlInputTerminal* pInputTerminal, const nn::cduac::AudioControlOutputTerminal* pOutputTerminal) NN_NOEXCEPT;

    int GetNumSpeakers() NN_NOEXCEPT;
private:
    void SetConnectionName(ConnectionId interfaceId) NN_NOEXCEPT;

private:
    nn::cduac::Host                 m_Host;
    nn::cduac::Parser               m_Parser;
    nn::cduac::Interface            m_Interface[MaxConnections];
    nn::os::MultiWaitHolderType     m_DetachHolder[MaxConnections];
    char                            m_InterfaceName[MaxConnections][AudioIn::NameLength];
    int                             m_NextUacId;
    int                             m_SpeakerCount;
    bool                            m_IsInterfaceConnected[MaxConnections];
    bool                            m_IsMicrophone[MaxConnections];
    bool                            m_IsSpeaker[MaxConnections];
    bool                            m_IsOpen[MaxConnections];
    nn::os::EventType*              m_pUnsupportedSpeakerAttachEvent;
};

}  // namespace detail
}  // namespace audio
}  // namespace nn
