﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "audio_UacConnectionManager.h"
#include "audio_UacManager.h"
#include "../audio_AppletVolumeManager.h"
#include <nne/audio/audio.h>
#include <nn/audio/audio_Result.h>
#include <nn/audio/detail/audio_Log.h>
#include <nn/audio/server/audio_FirmwareDebugSettings.h>
#include <nn/nn_SystemThreadDefinition.h>
#include <cstdio>

namespace nn {
namespace audio {
namespace detail {

namespace
{
    const char g_UacPrefix[] = "Uac_";
    //NameLength should always be large enough to handle:
    //Uac_4294967296 (Prefix + int_max)
    NN_STATIC_ASSERT(AudioIn::NameLength >= sizeof(g_UacPrefix) + 11);
}

UacConnectionManager::UacConnectionManager() NN_NOEXCEPT
    : m_Host()
    , m_Parser()
    , m_Interface()
    , m_DetachHolder()
    , m_NextUacId(0)
    , m_SpeakerCount(0)
    , m_pUnsupportedSpeakerAttachEvent(nullptr)
{
    memset(m_InterfaceName, 0, sizeof(m_InterfaceName));
    for(int i = 0; i < nn::usb::HsLimitMaxInterfacesPerClientCount; ++i)
    {
        m_IsInterfaceConnected[i] = false;
        m_IsMicrophone[i] = false;
        m_IsSpeaker[i] = false;
        m_IsOpen[i] = false;
    }
}

void UacConnectionManager::Initialize() NN_NOEXCEPT
{
    for(int i = 0; i < nn::usb::HsLimitMaxInterfacesPerClientCount; ++i)
    {
        m_IsInterfaceConnected[i] = false;
    }
    m_Host.Initialize();
}

void UacConnectionManager::Finalize() NN_NOEXCEPT
{
    DetachAllDevices();
    m_Host.Finalize();
}

void UacConnectionManager::Sleep() NN_NOEXCEPT
{
    Finalize();
}

void UacConnectionManager::Wake() NN_NOEXCEPT
{
    Initialize();
}

void UacConnectionManager::SetAttachEvent(nn::os::SystemEventType* event) NN_NOEXCEPT
{
    m_Host.CreateInterfaceAvailableEvent(event);
}

void UacConnectionManager::ClearAttachEvent(nn::os::SystemEventType* event) NN_NOEXCEPT
{
    m_Host.DestroyInterfaceAvailableEvent(event);
}

void UacConnectionManager::SetUnsupportedSpeakerAttachEvent(nn::os::EventType* event) NN_NOEXCEPT
{
    m_pUnsupportedSpeakerAttachEvent = event;
}

void UacConnectionManager::AttachDevice(nn::os::MultiWaitType* multiWait) NN_NOEXCEPT
{
    int32_t ifCount;
    nn::usb::InterfaceQueryOutput interfaceQueryOutput[nn::usb::HsLimitMaxInterfacesPerClientCount];

    m_Host.QueryAvailableInterfaces(&ifCount, interfaceQueryOutput, sizeof(interfaceQueryOutput));

    auto& uacManager = UacManager::GetInstance();

    for (int i = 0; i < ifCount; i++)
    {
        nn::cduac::InterfaceProfile interfaceProfile;

        m_Parser.CreateInterfaceProfile(&interfaceProfile, &m_Host, &interfaceQueryOutput[i]);

        nn::cduac::AudioControlInputTerminal *pInputTerminal    = m_Parser.GetInputTerminal(&interfaceProfile);
        nn::cduac::AudioControlOutputTerminal *pOutputTerminal  = m_Parser.GetOutputTerminal(&interfaceProfile);

        if (pInputTerminal && pOutputTerminal)
        {
            for(int interfaceId = 0; interfaceId < nn::usb::HsLimitMaxInterfacesPerClientCount; ++interfaceId)
            {
                if(m_IsInterfaceConnected[interfaceId])
                {
                    continue;
                }
                auto result = m_Interface[interfaceId].Initialize(&m_Host, &interfaceProfile);

                if(!result.IsSuccess())
                {
                    continue;
                }

                auto detachEvent = m_Interface[interfaceId].GetStateChangeEvent();
                m_IsInterfaceConnected[interfaceId] = true;
                nn::os::InitializeMultiWaitHolder( &m_DetachHolder[interfaceId], detachEvent );
                auto data = CreateMultiWaitData(UacEventType_Detach, interfaceId);
                nn::os::SetMultiWaitHolderUserData( &m_DetachHolder[interfaceId], static_cast<uintptr_t>(data) );

                nn::os::LinkMultiWaitHolder( multiWait, &m_DetachHolder[interfaceId] );

                if(IsInput(pInputTerminal, pOutputTerminal))
                {
                    if(uacManager.IsUacInDeviceSupported(&m_Interface[interfaceId], &m_Parser))
                    {
                        m_IsMicrophone[interfaceId] = true;
                        SetConnectionName(interfaceId);
                        AppletVolumeManager::SignalInputNotification();
                    }
                }
                if(IsOutput(pInputTerminal, pOutputTerminal))
                {
                    if(uacManager.IsUacOutDeviceSupported(&m_Interface[interfaceId], &m_Parser))
                    {
                        m_IsSpeaker[interfaceId] = true;
                        SetConnectionName(interfaceId);
                        ++m_SpeakerCount;
                    }
                    else if(m_pUnsupportedSpeakerAttachEvent)
                    {
                        nn::os::SignalEvent(m_pUnsupportedSpeakerAttachEvent);
                    }
                }
                break;
            }
        }
    }
}

void UacConnectionManager::DetachDevice(int id) NN_NOEXCEPT
{
    if(m_IsInterfaceConnected[id])
    {
        m_IsInterfaceConnected[id] = false;
        if(m_IsMicrophone[id])
        {
            AppletVolumeManager::SignalInputNotification();
        }
        if(m_IsSpeaker[id])
        {
            AppletVolumeManager::SignalOutputNotification();
            --m_SpeakerCount;
        }
        m_IsMicrophone[id] = false;
        m_IsSpeaker[id] = false;
        nn::os::UnlinkMultiWaitHolder(&m_DetachHolder[id]);
        nn::os::FinalizeMultiWaitHolder(&m_DetachHolder[id]);
        NN_ABORT_UNLESS_RESULT_SUCCESS(m_Interface[id].Finalize());
    }
}

void UacConnectionManager::DetachAllDevices() NN_NOEXCEPT
{
    for(int interfaceId = 0; interfaceId < nn::usb::HsLimitMaxInterfacesPerClientCount; ++interfaceId)
    {
        DetachDevice(interfaceId);
    }
}

void UacConnectionManager::Open(ConnectionId id) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(id >= 0 && id < MaxConnections);
    NN_SDK_REQUIRES(!m_IsOpen[id]);
    m_IsOpen[id] = true;
}

void UacConnectionManager::Close(ConnectionId id) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(id >= 0 && id < MaxConnections);
    NN_SDK_REQUIRES(m_IsOpen[id]);
    m_IsOpen[id] = false;
}

bool UacConnectionManager::IsOpen(ConnectionId id) NN_NOEXCEPT
{
    if(id >= 0 && id < MaxConnections)
    {
        return m_IsOpen[id];
    }
    return false;
}

bool UacConnectionManager::IsMicrophone(ConnectionId id) NN_NOEXCEPT
{
    if(id >= 0 && id < MaxConnections)
    {
        return m_IsMicrophone[id];
    }
    return false;
}

bool UacConnectionManager::IsSpeaker(ConnectionId id) NN_NOEXCEPT
{
    if(id >= 0 && id < MaxConnections)
    {
        return m_IsSpeaker[id];
    }
    return false;
}

bool UacConnectionManager::IsConnected(ConnectionId id) NN_NOEXCEPT
{
    if(id >= 0 && id < MaxConnections)
    {
        return m_IsInterfaceConnected[id];
    }
    return false;
}

const char* UacConnectionManager::GetName(ConnectionId id) NN_NOEXCEPT
{
    if(IsConnected(id))
    {
        return m_InterfaceName[id];
    }
    return nullptr;
}

int UacConnectionManager::GetConnectionCount(UacConnectionType type) NN_NOEXCEPT
{
    int count = 0;
    bool isMicrophoneType = type == UacConnectionType_Microphone;
    for(int i = 0; i < MaxConnections; ++i)
    {
        if(isMicrophoneType && m_IsMicrophone[i])
        {
            ++count;
        }
    }
    return count;
}

ConnectionId UacConnectionManager::FindConnection(const char* name, UacConnectionType type) NN_NOEXCEPT
{
    bool isMicrophoneType = type == UacConnectionType_Microphone;
    bool isSpeakerType = type == UacConnectionType_Speaker;
    for(int i = 0; i < MaxConnections; ++i)
    {
        if((isMicrophoneType && m_IsMicrophone[i]) || (isSpeakerType && m_IsSpeaker[i]))
        {
            if(strcmp(name, GetName(i)) == 0)
            {
                return i;
            }
        }
    }
    return InvalidConnectionId;
}

ConnectionId UacConnectionManager::FindUnusedConnection(UacConnectionType type) NN_NOEXCEPT
{
    bool isMicrophoneType = type == UacConnectionType_Microphone;
    bool isSpeakerType = type == UacConnectionType_Speaker;
    for(int i = 0; i < MaxConnections; ++i)
    {
        if((isMicrophoneType && m_IsMicrophone[i]) || (isSpeakerType && m_IsSpeaker[i]))
        {
            if(!m_IsOpen[i])
            {
                return i;
            }
        }
    }
    return InvalidConnectionId;
}


nn::cduac::Host* UacConnectionManager::GetHost() NN_NOEXCEPT
{
    return &m_Host;
}

nn::cduac::Parser* UacConnectionManager::GetParser() NN_NOEXCEPT
{
    return &m_Parser;
}

nn::cduac::Interface* UacConnectionManager::GetInterface(ConnectionId id) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(id >= 0 && id < MaxConnections);
    return &m_Interface[id];
}

bool UacConnectionManager::IsInput(const nn::cduac::AudioControlInputTerminal* pInputTerminal, const nn::cduac::AudioControlOutputTerminal* pOutputTerminal) NN_NOEXCEPT
{
    bool ret = false;

    if( pInputTerminal == nullptr || pOutputTerminal == nullptr)
    {
        return ret;
    }

    if (pOutputTerminal->wTermainalType == static_cast<uint16_t>(nn::cduac::UsbTerminalType::UsbTerminalType_Streaming))
    {
        switch (pInputTerminal->wTermainalType)
        {
        case nn::cduac::InputTerminalType::InputTerminalType_Microphone:
            ret = true;
            break;
        default:
            break;
        }
    }

    return ret;
}

bool UacConnectionManager::IsOutput(const nn::cduac::AudioControlInputTerminal* pInputTerminal, const nn::cduac::AudioControlOutputTerminal* pOutputTerminal) NN_NOEXCEPT
{
    bool ret = false;

    if(!nn::audio::server::IsUacEnabled())
    {
        return false;
    }

    if( pInputTerminal == nullptr || pOutputTerminal == nullptr)
    {
        return ret;
    }

    if (pInputTerminal->wTermainalType == static_cast<uint16_t>(nn::cduac::UsbTerminalType::UsbTerminalType_Streaming))
    {
        switch (pOutputTerminal->wTermainalType)
        {
        case nn::cduac::OutputTerminalType::OutputTermainalType_Speaker:
        case nn::cduac::OutputTerminalType::OutputTermainalType_DesktopSpeaker:
        case nn::cduac::OutputTerminalType::OutputTermainalType_RoomSpeaker:
            ret = true;
            break;
        default:
            break;
        }
    }

    return ret;
}

int UacConnectionManager::GetNumSpeakers() NN_NOEXCEPT
{
    return m_SpeakerCount;
}

uintptr_t UacConnectionManager::CreateMultiWaitData(UacEventType type, int id) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(id <= std::numeric_limits<int16_t>::max());
    NN_STATIC_ASSERT(sizeof(UacEventType) == sizeof(int16_t));

    uintptr_t result = static_cast<uintptr_t>(type);
    result <<= EventTypeShift;
    result |= id;
    return result;
}

int UacConnectionManager::GetIdFromEventData(uintptr_t data) NN_NOEXCEPT
{
    return static_cast<int>((data >> EventIdShift) & EventIdMask);
}

int UacConnectionManager::GetTypeFromEventData(uintptr_t data) NN_NOEXCEPT
{
    return static_cast<int>(data >> EventTypeShift);
}

void UacConnectionManager::PrintStatus(const char* caller)
{
    NN_DETAIL_AUDIO_INFO("\n\n--------\n%s\n", caller);
    for(int interfaceId = 0; interfaceId < nn::usb::HsLimitMaxInterfacesPerClientCount; ++interfaceId)
    {
        NN_DETAIL_AUDIO_INFO("[%d] %s, connected: %d event: %p, IsMic: %d, IsSpeaker: %d\n",
                    interfaceId,
                    m_InterfaceName[interfaceId],
                    m_IsInterfaceConnected[interfaceId],
                    m_Interface[interfaceId].GetStateChangeEvent(),
                    m_IsMicrophone[interfaceId],
                    m_IsSpeaker[interfaceId]);
    }
    NN_DETAIL_AUDIO_INFO("\n--------\n\n");
}


void UacConnectionManager::SetConnectionName(ConnectionId id) NN_NOEXCEPT
{
    memset(m_InterfaceName[id], 0, sizeof(m_InterfaceName[id]));
    snprintf(m_InterfaceName[id], sizeof(m_InterfaceName[id]), "%s%u", g_UacPrefix, m_NextUacId);
    ++m_NextUacId;
}

}  // namespace detail
}  // namespace audio
}  // namespace nn


