﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

NN_PRAGMA_PUSH_WARNINGS
NN_DISABLE_WARNING_FROM_WINDOWS_SDK_HEADERS
#include <mmdeviceapi.h>
#include <Audioclient.h>
NN_PRAGMA_POP_WARNINGS

#include <stdint.h>
#include <algorithm>
#include <atomic>

#include <nn/os/os_ThreadCommon.h>
#include <nn/os/os_Semaphore.h>
#include <nn/audio/audio_AudioOutTypes.h>
#include <nn/audio/audio_AudioInTypes.h>
#include <nn/nn_Macro.h>
#include <nn/audio/audio_Result.h>

#include "audio_AudioDriver.h"
#include "audio_CharEncodingConverter-os.win32.h"

namespace nn {
namespace audio {
namespace detail {

class AudioWasapiDriver : public AudioDriver
{
public:
template <typename InfoType>
    static int ListDevices(server::SessionType sessionType, InfoType* outInfo, int count) NN_NOEXCEPT
    {
        memset(outInfo, 0, sizeof(InfoType) * count);
        EDataFlow type = SessionTypeToWasapiType(sessionType);
        NN_SDK_ASSERT(type == eRender || type == eCapture);

        const int DeviceNameLength = type == eRender ? AudioOut::NameLength : AudioIn::NameLength;
        IMMDeviceCollection* pDeviceCollection = nullptr;
        if (FAILED(s_pEnumerator->EnumAudioEndpoints(type, DEVICE_STATE_ACTIVE, &pDeviceCollection)))
        {
            return 0;
        }

        UINT deviceCount;
        if (FAILED(pDeviceCollection->GetCount(&deviceCount)))
        {
            pDeviceCollection->Release();
            return 0;
        }

        int listedCount = (std::min)(count, static_cast<int>(deviceCount));
        for (int i = 0; i < listedCount; ++i)
        {
            IMMDevice* pDevice;
            pDeviceCollection->Item(i, &pDevice);
            GetDeviceName(pDevice, outInfo[i].name, DeviceNameLength);
        }

        pDeviceCollection->Release();
        return listedCount;
    }

    AudioWasapiDriver() NN_NOEXCEPT;
    ~AudioWasapiDriver() NN_NOEXCEPT;

    static bool Initialize(server::SessionType type, nn::os::SemaphoreType* pSemaphore) NN_NOEXCEPT;
    static void Finalize(server::SessionType type) NN_NOEXCEPT;
    static Result IsSupported(server::SessionType sessionType, const char* name, const server::SessionFormat& format) NN_NOEXCEPT;
    static const char* GetDefaultName(server::SessionType type) NN_NOEXCEPT;

    Result Open(server::SessionType sessionType, AppletVolumeManager::SessionType avmType, const char* name, server::SessionFormat& format, int sessionId, nn::dd::ProcessHandle processHandle, nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;
    Result Close() NN_NOEXCEPT;
    Result Start() NN_NOEXCEPT;
    Result Stop() NN_NOEXCEPT;
    Result Sleep() NN_NOEXCEPT;
    Result Wake() NN_NOEXCEPT;
    uint64_t GetSamplesProcessed() const NN_NOEXCEPT;
    void* MapBuffer(server::SessionBuffer& buffer, nn::dd::ProcessHandle processHandle) NN_NOEXCEPT;
    void* MapBuffer(server::SessionBuffer& buffer, nn::sf::NativeHandle&& bufferHandle) NN_NOEXCEPT;
    bool UnmapBuffer(const server::SessionBuffer& buffer) NN_NOEXCEPT;
    bool TryCompleteBuffer(const server::SessionBuffer& buffer) NN_NOEXCEPT;
    Result AppendBuffer(const server::SessionBuffer& buffer) NN_NOEXCEPT;
    void* GetBufferAddress() const NN_NOEXCEPT;
    Result SetDeviceGain(float gain) NN_NOEXCEPT;
    float GetDeviceGain() const NN_NOEXCEPT;
    void* GetWindowsEventHandle() const NN_NOEXCEPT;

private:
    UINT32 GetCurrentPadding() NN_NOEXCEPT;
    UINT32 GetBufferSize() NN_NOEXCEPT;
    void* AcquireCaptureBuffer(int* numFrames) NN_NOEXCEPT;
    Result ReleaseCaptureBuffer(int numFrames) NN_NOEXCEPT;
    void* AcquireRenderBuffer(int numFrames) NN_NOEXCEPT;
    Result ReleaseRenderBuffer(int numFrames) NN_NOEXCEPT;
    void CopyAndApplyVolume(int8_t* dst, int8_t* src, size_t sizeBytes) const NN_NOEXCEPT;

    void* GetClientImpl() NN_NOEXCEPT;
    static void CreateWaveFormat(WAVEFORMATEXTENSIBLE* outFormat, const server::SessionFormat& format) NN_NOEXCEPT;
    static GUID SampleFormatToWindowsGuid(SampleFormat sampleFormat) NN_NOEXCEPT;
    static EDataFlow SessionTypeToWasapiType(server::SessionType sessionType) NN_NOEXCEPT;
    void UpdateSession() NN_NOEXCEPT;

    static void ThreadFunc(void* arg);
private:
    static const int NameLength = std::max(AudioOut::NameLength, AudioIn::NameLength);
    static const int BufferCountMax = 4;
    static const size_t StackSize = 4096;

    static NN_OS_ALIGNAS_THREAD_STACK char s_Stack[StackSize];
    static IMMDeviceEnumerator*  s_pEnumerator;
    static os::SemaphoreType*    s_pInputSemaphore;
    static os::SemaphoreType*    s_pOutputSemaphore;
    static os::ThreadType        s_Thread;
    static nn::os::MutexType     s_MutexType;
    static std::atomic<int>      s_InitializeReferenceCount;
    static std::atomic<bool>     s_IsThreadActive;

private:
    server::SessionBuffer m_AppendBufferList[BufferCountMax];
    IMMDevice* m_pDevice;
    IAudioClient* m_pClient;
    nn::os::SemaphoreType* m_pSemaphore;
    void* m_hEvent;
    union
    {
        IAudioRenderClient* m_pRenderClient;
        IAudioCaptureClient* m_pCaptureClient;
        void*               m_pGenericClient;
    };
    mutable nn::os::Mutex m_Mutex;
    server::SessionFormat m_Format;
    float m_Volume;
    int m_NumAppendedBuffers;
    size_t m_BufferReadPosition;
    UINT32 m_BufferSize;
    EDataFlow m_Type;
    bool m_IsOpen;
};

}  // namespace detail
}  // namespace audio
}  // namespace nn
