﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/Result/result_HandlingUtility.h>
#include "audio_AudioOutSession-os.horizon.h"

namespace nn { namespace audio {

namespace server {

namespace {
    const char                   g_DeviceOutName[] = "DeviceOut";
}

int ListAudioOuts(AudioOutInfo* outAudioOuts, int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outAudioOuts);
    NN_SDK_REQUIRES(count >= 0);

    memset(outAudioOuts, 0, count * sizeof(AudioOutInfo));

    if( count > 0 )
    {
        strcpy(outAudioOuts[0].name, g_DeviceOutName);
        count = 1;
    }

    return count;
}



void AudioOutInitialize(nn::os::SemaphoreType** semaphore, nn::os::EventType* pEventSemaphore) NN_NOEXCEPT
{
    *semaphore = nne::audio::gmix::GetProcessSemaphore(nne::audio::gmix::Session::Name::Default);
}

void AudioOutFinalize() NN_NOEXCEPT
{
}

} //namespace server

namespace detail {
AudioGmixDriver AudioOutSession::s_GmixDriver[server::AudioOutManagerImpl::NumberOfSessions];

AudioOutSession::AudioOutSession(int32_t sessionId, nn::os::SystemEvent& releaseClientBufferEvent) NN_NOEXCEPT
    : AudioSession(sessionId, GetDriver(sessionId), releaseClientBufferEvent, server::SessionType_Output)
{
}

AudioOutSession::~AudioOutSession() NN_NOEXCEPT
{
}

const char* AudioOutSession::GetDefaultName() const NN_NOEXCEPT
{
    return server::g_DeviceOutName;
}

Result AudioOutSession::IsSupported(const char* name, const server::SessionFormat& format) const NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(0 == strnlen(name, AudioOut::NameLength) || 0 == strncmp(name, GetDefaultName(), AudioOut::NameLength), nn::audio::ResultNotFound());
    // Fail if the sampleRate is not 0 or 48000
    NN_RESULT_THROW_UNLESS((0 == format.sampleRate || 48000 == format.sampleRate), ResultInvalidSampleRate());
    // Fail if the channelCount is not 0 or 2 or 6
    NN_RESULT_THROW_UNLESS(0 == format.channelCount || 2 == format.channelCount || 6 == format.channelCount, ResultInvalidChannelCount());

    NN_RESULT_SUCCESS;
}

AudioOutState AudioOutSession::GetState() const NN_NOEXCEPT
{
    switch(m_State)
    {
        case server::SessionState_Started:
            return AudioOutState_Started;
        case server::SessionState_Stopped:
            return AudioOutState_Stopped;
        default:
            NN_UNEXPECTED_DEFAULT;
    }
    return AudioOutState_Stopped;
}

AudioDriver* AudioOutSession::GetDriver(int sessionId) const NN_NOEXCEPT
{
    NN_ABORT_UNLESS_MINMAX(sessionId, 0, static_cast<int>(NN_ARRAY_SIZE(s_GmixDriver)) - 1);
    return &s_GmixDriver[sessionId];
}

void AudioOutSession::SetAudioOutVolume(float volume) NN_NOEXCEPT
{
    const auto avmType = SessionTypeToAppletVolumeManagerType(m_Type);
    const auto sessionId = GetSessionId();

    const auto result = AppletVolumeManager::SetVolume(avmType, sessionId, volume, m_AppletResourceUserId);
    NN_SDK_ASSERT(result.IsSuccess());
    NN_UNUSED(result);
}

float AudioOutSession::GetAudioOutVolume() const NN_NOEXCEPT
{
    const auto avmType = SessionTypeToAppletVolumeManagerType(m_Type);
    const auto sessionId = GetSessionId();

    float volume = 0.0f;
    const auto result = AppletVolumeManager::GetVolume(&volume, avmType, sessionId, m_AppletResourceUserId);
    NN_SDK_ASSERT(result.IsSuccess());
    NN_UNUSED(result);

    return volume;
}

}}} // namespace nn::audio::detail
