﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>

#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/audio/audio_Result.h>

#include "audio_AppletVolumeManager.h"
#include "audio_AudioDeviceImpl-os.horizon.h"
#include "detail/audioctrl_AudioControllerService.h"

namespace nn { namespace audio { namespace server { namespace detail {
namespace {

// TODO: extract device name definition to "audio/common" namespace.
const int g_DeviceNameCount = AppletVolumeManager::DeviceType_Count;
const char* g_DeviceNames[AppletVolumeManager::DeviceType_Count];
int g_DeviceCapableChannelCount[AppletVolumeManager::DeviceType_Count];

void InitializeDeviceInfo()
{
    static bool initialized = false;
    if (initialized == false)
    {
        g_DeviceNames[AppletVolumeManager::DeviceType_Hda]           = "AudioTvOutput";
        g_DeviceNames[AppletVolumeManager::DeviceType_AhubHeadphone] = "AudioStereoJackOutput";
        g_DeviceNames[AppletVolumeManager::DeviceType_AhubSpeaker]   = "AudioBuiltInSpeakerOutput";
        g_DeviceNames[AppletVolumeManager::DeviceType_UsbOutputDevice]   = "AudioUsbDeviceOutput";

        g_DeviceCapableChannelCount[AppletVolumeManager::DeviceType_Hda] = 6;
        g_DeviceCapableChannelCount[AppletVolumeManager::DeviceType_AhubHeadphone] = 2;
        g_DeviceCapableChannelCount[AppletVolumeManager::DeviceType_AhubSpeaker] = 2;
        g_DeviceCapableChannelCount[AppletVolumeManager::DeviceType_UsbOutputDevice] = 2;

        initialized = true;
    }
}

bool GetDeviceIndex(const char* deviceName, AppletVolumeManager::DeviceType* outIndex) NN_NOEXCEPT
{
    for (auto i = 0; i < g_DeviceNameCount; ++i)
    {
        if (strncmp(deviceName, g_DeviceNames[i], nn::audio::AudioDeviceName::NameLength) == 0)
        {
            *outIndex = static_cast<AppletVolumeManager::DeviceType>(i);
            return true;
        }
    }
    return false;
}

const char* GetDeviceName(AppletVolumeManager::DeviceType index) NN_NOEXCEPT
{
    return g_DeviceNames[index];
}

}

void InitializeAudioDevice(const nn::applet::AppletResourceUserId& id) NN_NOEXCEPT
{
    InitializeDeviceInfo();
    AppletVolumeManager::RegisterAppletResourceUserId(id);
}

void FinalizeAudioDevice(const nn::applet::AppletResourceUserId& id) NN_NOEXCEPT
{
    AppletVolumeManager::UnregisterAppletResourceUserId(id);
}

int ListAudioDeviceName(AudioDeviceName* outDeviceNames, int count, common::RevisionInfo revisionInfo) NN_NOEXCEPT
{
    const auto IsAudioUsbDeviceOutputSupported = common::CheckFeatureSupported(common::SupportTags::AudioUsbDeviceOutput, revisionInfo);

    int copyCount = 0;
    for (auto i = 0; i < count && i < g_DeviceNameCount; ++i)
    {
        if(!IsAudioUsbDeviceOutputSupported && i == AppletVolumeManager::DeviceType_UsbOutputDevice)
        {
            continue;
        }

        strncpy(outDeviceNames[i].name, g_DeviceNames[i], nn::audio::AudioDeviceName::NameLength);
        ++copyCount;
    }
    return copyCount;
}

void SetAudioDeviceOutputVolume(const AudioDeviceName* outDeviceName, float volume, const nn::applet::AppletResourceUserId& id, common::RevisionInfo revisionInfo) NN_NOEXCEPT
{
    const auto IsAudioUsbDeviceOutputSupported = common::CheckFeatureSupported(common::SupportTags::AudioUsbDeviceOutput, revisionInfo);

    AppletVolumeManager::DeviceType device = AppletVolumeManager::DeviceType_Count;
    if (GetDeviceIndex(outDeviceName->name, &device) && device < AppletVolumeManager::DeviceType_Count)
    {
        if(!IsAudioUsbDeviceOutputSupported && device == AppletVolumeManager::DeviceType_UsbOutputDevice)
        {
            return;
        }

        AppletVolumeManager::SetDeviceVolume(volume, device, AppletVolumeManager::SessionType_AudioRenderer, id);
        AppletVolumeManager::SetDeviceVolume(volume, device, AppletVolumeManager::SessionType_AudioOut, id);

        if(!IsAudioUsbDeviceOutputSupported && device == AppletVolumeManager::DeviceType_AhubHeadphone)
        {
            AppletVolumeManager::SetDeviceVolume(volume, AppletVolumeManager::DeviceType_UsbOutputDevice, AppletVolumeManager::SessionType_AudioRenderer, id);
            AppletVolumeManager::SetDeviceVolume(volume, AppletVolumeManager::DeviceType_UsbOutputDevice, AppletVolumeManager::SessionType_AudioOut, id);
        }
    }
}

float GetAudioDeviceOutputVolume(const AudioDeviceName* outDeviceName, const nn::applet::AppletResourceUserId& id, common::RevisionInfo revisionInfo) NN_NOEXCEPT
{
    AppletVolumeManager::DeviceType device = AppletVolumeManager::DeviceType_Count;
    const auto IsAudioUsbDeviceOutputSupported = common::CheckFeatureSupported(common::SupportTags::AudioUsbDeviceOutput, revisionInfo);

    if (GetDeviceIndex(outDeviceName->name, &device) && device < AppletVolumeManager::DeviceType_Count)
    {
        if(!IsAudioUsbDeviceOutputSupported && device == AppletVolumeManager::DeviceType_UsbOutputDevice)
        {
            return 0.0f;
        }

        float retVal = 0.f;
        AppletVolumeManager::GetDeviceVolume(&retVal, device, AppletVolumeManager::SessionType_AudioRenderer, id);
        float chkVal = 0.f;
        AppletVolumeManager::GetDeviceVolume(&chkVal, device, AppletVolumeManager::SessionType_AudioOut, id);
        NN_SDK_ASSERT_EQUAL(chkVal, retVal);
        return retVal;
    }
    return 0.f;
}

int GetAudioDeviceCapableChannelCount(nn::audio::AppletVolumeManager::DeviceType device) NN_NOEXCEPT
{
    InitializeDeviceInfo();
    return g_DeviceCapableChannelCount[device];
}

void GetActiveAudioDeviceName(AudioDeviceName* outDeviceName, common::RevisionInfo revisionInfo) NN_NOEXCEPT
{
    AppletVolumeManager::DeviceType type;
    AppletVolumeManager::GetActiveDevice(&type);

    const auto IsAudioUsbDeviceOutputSupported = common::CheckFeatureSupported(common::SupportTags::AudioUsbDeviceOutput, revisionInfo);

    if(!IsAudioUsbDeviceOutputSupported && type == AppletVolumeManager::DeviceType_UsbOutputDevice)
    {
        type = AppletVolumeManager::DeviceType_AhubHeadphone;
    }

    strncpy(outDeviceName->name, GetDeviceName(type), outDeviceName->NameLength - 1);
}

nn::Result QueryAudioDeviceSystemEvent(nn::os::NativeHandle* outHandle, const nn::applet::AppletResourceUserId& id) NN_NOEXCEPT
{
    return AppletVolumeManager::QueryDeviceEvent(outHandle, id);
}

nn::Result QueryAudioDeviceInputNotificationEvent(nn::os::NativeHandle* outHandle, const nn::applet::AppletResourceUserId& id) NN_NOEXCEPT
{
    return AppletVolumeManager::QueryInputNotificationEvent(outHandle, id);
}

nn::Result QueryAudioDeviceOutputNotificationEvent(nn::os::NativeHandle* outHandle, const nn::applet::AppletResourceUserId& id) NN_NOEXCEPT
{
    return AppletVolumeManager::QueryOutputNotificationEvent(outHandle, id);
}

namespace {

nn::audioctrl::AudioTarget ConvertToTarget(AppletVolumeManager::DeviceType device) NN_NOEXCEPT
{
    switch(device)
    {
    case AppletVolumeManager::DeviceType_AhubHeadphone:
        return nn::audioctrl::AudioTarget_Headphone;
    case AppletVolumeManager::DeviceType_AhubSpeaker:
        return nn::audioctrl::AudioTarget_Speaker;
    case AppletVolumeManager::DeviceType_UsbOutputDevice:
        return nn::audioctrl::AudioTarget_UsbOutputDevice;
    case AppletVolumeManager::DeviceType_Hda:
        return nn::audioctrl::AudioTarget_Tv;
    case AppletVolumeManager::DeviceType_Count:
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

int ConverToInt(nn::audioctrl::AudioOutputMode mode) NN_NOEXCEPT
{
    switch (mode)
    {
    case nn::audioctrl::AudioOutputMode_Pcm1ch:
        return 1;
        break;
    case nn::audioctrl::AudioOutputMode_Pcm2ch:
        return 2;
        break;
    case nn::audioctrl::AudioOutputMode_Pcm6ch:
        return 6;
        break;
    case nn::audioctrl::AudioOutputMode_PcmAuto:
        return 6;
        break;
    case nn::audioctrl::AudioOutputMode_Invalid: // 未設定時には invalid が使われている
        return -1;
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

}

int GetActiveChannelCount() NN_NOEXCEPT
{
    AppletVolumeManager::DeviceType type;
    AppletVolumeManager::GetActiveDevice(&type);
    // Headphone が指定されることはないため、Headphone は Speaker に丸める
    auto target = ConvertToTarget(type);
    if (target == nn::audioctrl::AudioTarget_Headphone)
    {
        target = nn::audioctrl::AudioTarget_Speaker;
    }
    auto count = ConverToInt(nn::audioctrl::server::detail::GetAudioOutputMode(target));
    return (count > 0) ?
        std::min(AppletVolumeManager::GetDeviceChannelCount(), count) :
        AppletVolumeManager::GetDeviceChannelCount();
}

float GetAudioSystemMasterVolumeSetting(const AudioDeviceName* name) NN_NOEXCEPT
{
    AppletVolumeManager::DeviceType device = AppletVolumeManager::DeviceType_Count;
    if (GetDeviceIndex(name->name, &device) && device < AppletVolumeManager::DeviceType_Count)
    {
        const auto target = ConvertToTarget(device);
        if (target == nn::audioctrl::AudioTarget_Tv)
        {
            return 1.0f;
        }
        if (nn::audioctrl::server::detail::IsTargetMute(target))
        {
            return 0.0f;
        }
        auto value = nn::audioctrl::server::detail::GetTargetVolume(target) - nn::audioctrl::server::detail::GetTargetVolumeMin();
        auto range = nn::audioctrl::server::detail::GetTargetVolumeMax() - nn::audioctrl::server::detail::GetTargetVolumeMin();
        return static_cast<float>(value) / range;
    }
    return 0.0f;
}

}}}} // namespace nn::audio::server::detail
