﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstddef>
#include <nn/nn_StaticAssert.h>
#include <nn/audio/audio_Result.h>

#include "../audio_SinkManager.h"
#include "../audio_AddrTypes.h"
#include "../common/audio_Util.h"
#include "../common/audio_BehaviorParameters.h"
#include "../dsp/audio_Upsampler.h" // TODO: remove "dsp" reference.
#include "audio_UpsamplerManager.h"
#include "audio_ServiceMemoryPoolInfo.h"

namespace nn { namespace audio { namespace server {

struct SinkState {};

struct CircularBufferSinkState : SinkState
{
    int32_t _currentPos;
    int32_t _commandPos;
    int32_t _previousCommandPos;
    AddressInfo _circularBuffer;

    void CleanUp() NN_NOEXCEPT;
};

struct DeviceSinkState : SinkState
{
    UpsamplerInfo* pUpsamplerInfo;
    int32_t _downMixMatrixCoeff[common::DeviceParameter::CoeffCount];
};

class NN_AUDIO_INFOTYPE_FILED_ALIGN SinkInfoBase
{
protected:
    common::SinkType m_Type;
    bool m_IsInUse;
    bool m_ShouldSkip;
    int8_t _padding[1];
    NodeId m_NodeId;
    int8_t m_Reserved[16];

    NN_AUDIO_INFOTYPE_FILED_ALIGN int8_t m_StateBuffer[sizeof(nn::audio::common::BiggestType<DeviceSinkState, CircularBufferSinkState>::Result)];
    NN_AUDIO_INFOTYPE_FILED_ALIGN int8_t m_ParameterBuffer[sizeof(nn::audio::common::BiggestType<common::DeviceParameter, common::CircularBufferParameter>::Result)];

public:
    SinkInfoBase() NN_NOEXCEPT;

    bool IsUsed() const NN_NOEXCEPT;
    bool ShouldSkip() const NN_NOEXCEPT;
    NodeId GetNodeId() const NN_NOEXCEPT;
    common::SinkType GetType() const NN_NOEXCEPT;
    const void* GetParameter() const NN_NOEXCEPT;
    const void* GetState() const NN_NOEXCEPT;

    void CleanUp() NN_NOEXCEPT;
    virtual void Update(common::BehaviorParameter::ErrorInfo* pOutErrorInfo, common::SinkOutStatus* pOutStatus, const common::SinkInParameter* pUserInParam, PoolMapper& poolMapper) NN_NOEXCEPT;
    virtual void UpdateForCommandGeneration() NN_NOEXCEPT;

    DeviceSinkState* GetDeviceState() NN_NOEXCEPT;
};
#if defined(NN_BUILD_CONFIG_SPEC_NX) && !defined(NN_BUILD_CONFIG_OS_WIN)
NN_AUDIO_INFOTYPE_CHECK(SinkInfoBase, 368);
#endif

template<typename parameter, typename state>
class SinkInfo : public SinkInfoBase
{
public:
    parameter* GetParameter() NN_NOEXCEPT
    {
        return reinterpret_cast<parameter*>(m_ParameterBuffer);
    }
    const parameter* GetParameter() const NN_NOEXCEPT
    {
        return reinterpret_cast<const parameter*>(m_ParameterBuffer);
    }

    state* GetState() NN_NOEXCEPT
    {
        return reinterpret_cast<state*>(m_StateBuffer);
    }
    const state* GetState() const NN_NOEXCEPT
    {
        return reinterpret_cast<const state*>(m_StateBuffer);
    }
};

class DeviceSinkInfo final : public SinkInfo<common::DeviceParameter, DeviceSinkState>
{
public:
    DeviceSinkInfo() NN_NOEXCEPT;

    void CleanUp() NN_NOEXCEPT;
    virtual void Update(common::BehaviorParameter::ErrorInfo* pOutErrorInfo, common::SinkOutStatus* pOutStatus, const common::SinkInParameter* pUserInParam, PoolMapper& poolMapper) NN_NOEXCEPT NN_OVERRIDE;
    virtual void UpdateForCommandGeneration() NN_NOEXCEPT NN_OVERRIDE;
};

class CircularBufferSinkInfo final : public SinkInfo<common::CircularBufferParameter, CircularBufferSinkState>
{
public:
    CircularBufferSinkInfo() NN_NOEXCEPT;

    void CleanUp() NN_NOEXCEPT;
    virtual void Update(common::BehaviorParameter::ErrorInfo* pOutErrorInfo, common::SinkOutStatus* pOutStatus, const common::SinkInParameter* pUserInParam, PoolMapper& poolMapper) NN_NOEXCEPT NN_OVERRIDE;
    virtual void UpdateForCommandGeneration() NN_NOEXCEPT NN_OVERRIDE;

    bool ShouldUpdateWorkBufferInfo(const common::SinkInParameter* pUserInParam) const NN_NOEXCEPT;
};

// Classes derived from server::SinkInfo should not have its own member fields to keep all server::SinkInfo type has same size.
NN_STATIC_ASSERT(sizeof(CircularBufferSinkInfo) == sizeof(SinkInfoBase));
NN_STATIC_ASSERT(sizeof(DeviceSinkInfo) == sizeof(SinkInfoBase));


// Data container & accessor for service side Sink related infos.
class SinkContext
{
    server::SinkInfoBase* m_Infos;
    int m_InfoCount;

public:
    SinkContext() NN_NOEXCEPT;
    void Initialize(server::SinkInfoBase* sinkInfos, int infoCount) NN_NOEXCEPT;
    server::SinkInfoBase& GetInfo(int id) const NN_NOEXCEPT;
    int GetCount() const NN_NOEXCEPT;
};

}}}  // namespace nn::audio::server
