﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/audio/audio_MemoryPool.h>
#include <nn/dd/dd_Types.h>
#include "../audio_AddrTypes.h"
#include "../audio_MemoryPoolInfo.h"
#include "../common/audio_BehaviorParameters.h"

//#define NN_AUDIO_MEMORYPOOL_ADDRESS_CHECK_ENABLED

namespace nn { namespace audio {

namespace server {
class AudioRenderSystem; // forward declaration
}

namespace server {

class PoolMapper; // forward declaration

class NN_AUDIO_INFOTYPE_FILED_ALIGN MemoryPoolInfo
{
NN_DISALLOW_COPY( MemoryPoolInfo );
NN_DISALLOW_MOVE( MemoryPoolInfo );

public:
    friend PoolMapper;
    typedef nn::audio::MemoryPoolInfo::State State;

    enum Location
    {
        Location_Invalid = 0,
        Location_Client,
        Location_Service,
    };

    explicit MemoryPoolInfo(Location location = Location_Client) NN_NOEXCEPT;
    bool Contains(CpuAddr addr, size_t size) const NN_NOEXCEPT;
    bool IsMapped() const NN_NOEXCEPT;
    DspAddr Translate(CpuAddr addr, size_t size) const NN_NOEXCEPT;

    void SetUsed(bool used) NN_NOEXCEPT;
    bool IsUsed() const NN_NOEXCEPT;

protected:
    CpuAddr GetCpuAddress() const NN_NOEXCEPT;
    DspAddr GetDspAddress() const NN_NOEXCEPT;
    size_t GetSize() const NN_NOEXCEPT;
    Location GetLocation() const NN_NOEXCEPT;

    void SetCpuAddress(CpuAddr cpuAddr, size_t size) NN_NOEXCEPT;
    void SetDspAddress(DspAddr dspAddr) NN_NOEXCEPT;

private:
    CpuAddr m_CpuAddress;
    DspAddr m_DspAddress;
#if !(defined(NN_BUILD_CONFIG_OS_WIN) && defined(NN_BUILD_CONFIG_ADDRESS_64))
    int8_t _padding0[4];
#endif
    size_t m_Size;
    Location m_Location;
    bool m_Used;
    int8_t _padding1[3];
};
#if defined(NN_BUILD_CONFIG_SPEC_NX) && !defined(NN_BUILD_CONFIG_OS_WIN)
NN_AUDIO_INFOTYPE_CHECK(MemoryPoolInfo, 32);
#endif


class AddressInfo
{
public:
    friend PoolMapper;

    explicit AddressInfo(CpuAddr cpuAddr = nullptr, size_t size = 0) NN_NOEXCEPT;
    ~AddressInfo() NN_NOEXCEPT;

    size_t GetSize() const NN_NOEXCEPT;
    DspAddr GetReference(bool countReference = true) const NN_NOEXCEPT;
    CpuAddr GetCpuAddr() const NN_NOEXCEPT;
    void Setup(CpuAddr addr, size_t size) NN_NOEXCEPT;
    bool IsMapped() const NN_NOEXCEPT;

private:
    void SetPool(MemoryPoolInfo* pool) NN_NOEXCEPT;
    bool HasMappedMemoryPool() const NN_NOEXCEPT;

    CpuAddr m_CpuAddr;
    size_t m_Size;
    MemoryPoolInfo* m_Pool;

public:
    DspAddr GetForceMappedDspAddr() const NN_NOEXCEPT;
    void SetForceMappedDspAddr(DspAddr addr) NN_NOEXCEPT;
private:
    DspAddr m_ForceMappedDspAddr;

};

class PoolMapper
{
public:
    friend nn::audio::server::AudioRenderSystem;
    PoolMapper(nn::dd::ProcessHandle processHandle, bool forceMapping = false) NN_NOEXCEPT;
    PoolMapper(nn::dd::ProcessHandle processHandle, MemoryPoolInfo* pInfos, int poolCount, bool forceMapping) NN_NOEXCEPT;

private:
    bool Map(server::MemoryPoolInfo* pOutMemoryPool) const NN_NOEXCEPT;
    bool Unmap(server::MemoryPoolInfo* pOutMemoryPool) const NN_NOEXCEPT;

    static void ClearUseState(server::MemoryPoolInfo* pPoolInfos, int poolCount) NN_NOEXCEPT;
    server::MemoryPoolInfo* FindMemoryPool(server::MemoryPoolInfo* pInfos, int poolCount, CpuAddr addr, size_t size) const NN_NOEXCEPT;
    server::MemoryPoolInfo* FindMemoryPool(CpuAddr addr, size_t size) const NN_NOEXCEPT;
    bool FillDspAddr(AddressInfo* pOutAddr, server::MemoryPoolInfo* pInfos, int poolCount) const NN_NOEXCEPT;
    bool FillDspAddr(AddressInfo* pOutAddr) const NN_NOEXCEPT;

public:
    bool TryAttachBuffer(common::BehaviorParameter::ErrorInfo* pOutErroInfo, AddressInfo* pOutAddr, const CpuAddr cpuAddr, size_t size) NN_NOEXCEPT;
    bool IsForceMapEnabled() const NN_NOEXCEPT;
    void ForceUnmapPointer(AddressInfo* pOutAddr) const NN_NOEXCEPT;

    enum Error
    {
        Error_NoError = 0,
        Error_InvalidAddressOrSize,
        Error_FailedToMap,
        Error_FailedToUnmap,
    };
    Error Update(MemoryPoolInfo* pOutPoolInfo, const nn::audio::MemoryPoolInfo::InParameter* pInParameter, nn::audio::MemoryPoolInfo::OutStatus* pOutOutStatus) const NN_NOEXCEPT;

protected:
    bool InitializeSystemPool(server::MemoryPoolInfo* pOutInfo, void* address, size_t size) const NN_NOEXCEPT;

private:
    DspAddr Map(nn::dd::ProcessHandle processHandle, CpuAddr addr, size_t size) const NN_NOEXCEPT;
    bool Unmap(nn::dd::ProcessHandle processHandle, CpuAddr addr, size_t size) const NN_NOEXCEPT;
    nn::dd::ProcessHandle GetProcessHandle(const server::MemoryPoolInfo* pInfo) const NN_NOEXCEPT;

    nn::dd::ProcessHandle m_ProcessHandle;
    MemoryPoolInfo* m_Infos;
    int m_PoolCount;
    bool m_IsForceMapEnabled;
};

}}} // namespace nn::audio::service
