﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring> // memset, memcpy
#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_BytePtr.h>

#include "audio_ServiceEffectInfo.h"
#include "../dsp/audio_Dsp.h" // ::nn::audio::dsp::InvalidateDspCache()

namespace nn {
namespace audio {
namespace server {

NN_DEFINE_STATIC_CONSTANT(const size_t EffectInfoBase::ParameterBufferSize);
NN_DEFINE_STATIC_CONSTANT(const size_t EffectInfoBase::StateBufferSize);

// ================================================================================
// EffectInfoBase
// ================================================================================

EffectInfoBase::EffectInfoBase() NN_NOEXCEPT
{
    CleanUp();
}

void EffectInfoBase::CleanUp() NN_NOEXCEPT
{
    m_Type = ::nn::audio::EffectType_Invalid;
    m_AttachMixId = nn::audio::Invalid_MixId;
    m_ProcessingOrder = -1;
    m_Enabled = false;
    m_ShouldSkip = false;

    memset(m_ParameterBuffer, 0, sizeof(m_ParameterBuffer));
    memset(m_StateBuffer, 0, sizeof(m_StateBuffer));

    for (auto i = 0; i < BufferCount; ++i)
    {
        this->m_WorkBuffer[i].Setup(nullptr, 0);
    }
}

void EffectInfoBase::ForceUnmapBuffers(PoolMapper& poolMapper) NN_NOEXCEPT
{
    for (auto j = 0; j < server::EffectInfoBase::BufferCount; ++j)
    {
        if (this->m_WorkBuffer[j].GetReference(false) != 0)
        {
            poolMapper.ForceUnmapPointer(&this->m_WorkBuffer[j]);
        }
    }
}

bool EffectInfoBase::IsEnabled() const NN_NOEXCEPT
{
    return m_Enabled;
}

bool EffectInfoBase::ShouldSkip() const NN_NOEXCEPT
{
    return m_ShouldSkip;
}

nn::audio::EffectType EffectInfoBase::GetType() const NN_NOEXCEPT
{
    return m_Type;
}

nn::audio::MixId EffectInfoBase::GetMixId() const NN_NOEXCEPT
{
    return m_AttachMixId;
}

int32_t EffectInfoBase::GetProcessingOrder() const NN_NOEXCEPT
{
    return m_ProcessingOrder;
}

const void* EffectInfoBase::GetParameter() const NN_NOEXCEPT
{
    return reinterpret_cast<const void*>(m_ParameterBuffer);
}

void* EffectInfoBase::GetStateBuffer() NN_NOEXCEPT
{
    return reinterpret_cast<void*>(m_StateBuffer);
}

void EffectInfoBase::SetUsage(UsageState state) NN_NOEXCEPT
{
    m_UsageState = state;
}

bool EffectInfoBase::ShouldUpdateWorkBufferInfo(const nn::audio::EffectInfo::InParameter* pUserInParam) const NN_NOEXCEPT
{
    auto newBuffer = (pUserInParam->_isNew == true);
    auto shouldRetry = m_ShouldSkip;
    return newBuffer || shouldRetry;
}

EffectInfoBase::UsageState EffectInfoBase::GetUsage() const NN_NOEXCEPT
{
    return m_UsageState;
}

void EffectInfoBase::StoreStatus(nn::audio::EffectInfo::OutStatus* pOutStatus, bool isRendererActive) const NN_NOEXCEPT
{
    if ((isRendererActive == false && GetUsage() != UsageState::Initialized) || // レンダラが停止中かつ、 Effect が初期化直後でない場合(Start 前に Add された Effect が削除されないために)
        (GetUsage() == UsageState::Stopped)) // Effect が無効化され、ADSP にコマンドが発行されていないことが保証されている
    {
        NN_SDK_ASSERT(GetUsage() != UsageState::Initialized || (GetUsage() == UsageState::Stopped && IsEnabled() == false));
        pOutStatus->_usageStatus = nn::audio::EffectInfo::OutStatus::UsageStatus_Removable;
    }
    else
    {
        pOutStatus->_usageStatus = nn::audio::EffectInfo::OutStatus::UsageStatus_Used;
    }
}

void EffectInfoBase::Update(common::BehaviorParameter::ErrorInfo* pOutErroInfo, const nn::audio::EffectInfo::InParameter* pUserInParam, PoolMapper& poolMapper) NN_NOEXCEPT
{
    NN_UNUSED(pUserInParam);
    NN_UNUSED(poolMapper);
    NN_SDK_ASSERT_EQUAL(m_Type, nn::audio::EffectType_Invalid);
    pOutErroInfo->_result = ResultSuccess();
    pOutErroInfo->_info = 0;
}

void EffectInfoBase::UpdateForCommandGeneration() NN_NOEXCEPT
{
    NN_SDK_ASSERT_EQUAL(m_Type, nn::audio::EffectType_Invalid);
}

DspAddr EffectInfoBase::GetWorkBuffer(Index index) const NN_NOEXCEPT
{
    NN_UNUSED(index);
    NN_SDK_ASSERT(false, "Should not be called.");
    return 0u;
}

nn::audio::DspAddr EffectInfoBase::GetSingleBuffer(server::EffectInfoBase::Index index) const NN_NOEXCEPT
{
    if (this->IsEnabled())
    {
        // touch memory pool and get dsp address.
        return this->m_WorkBuffer[index].GetReference();
    }

    if (GetUsage() != UsageState::Stopped)
    {
        // get dsp address.
        const auto bufferAddr = this->m_WorkBuffer[index].GetReference(false);
        const auto bufferSize = this->m_WorkBuffer[index].GetSize();

        if (bufferAddr != 0u && bufferSize > 0)
        {
            ::nn::audio::dsp::InvalidateDspCache(bufferAddr, bufferSize);
        }
    }

    return 0u;
}

// ================================================================================
// BufferMixer
// ================================================================================
void BufferMixerInfo::Update(common::BehaviorParameter::ErrorInfo* pOutErroInfo, const nn::audio::EffectInfo::InParameter* pUserInParam, PoolMapper& poolMapper) NN_NOEXCEPT
{
    NN_UNUSED(poolMapper);
    m_AttachMixId = pUserInParam->_attachMixId;
    m_ProcessingOrder = pUserInParam->_processingOrder;
    *GetParameter() = pUserInParam->_bufferMixerParameter;

    // TODO : _parameter の enabled の整理
    m_Enabled = pUserInParam->_enabled;

    pOutErroInfo->_result = ResultSuccess();
    pOutErroInfo->_info = 0;
}

void BufferMixerInfo::UpdateForCommandGeneration() NN_NOEXCEPT
{
    auto state = IsEnabled() ? EffectInfoBase::UsageState::Running : EffectInfoBase::UsageState::Stopped;
    SetUsage(state);
}

// ================================================================================
// Delay
// ================================================================================
void DelayInfo::Update(common::BehaviorParameter::ErrorInfo* pOutErroInfo, const nn::audio::EffectInfo::InParameter* pUserInParam, PoolMapper& poolMapper) NN_NOEXCEPT
{
    NN_SDK_ASSERT_EQUAL(GetType(), EffectType_Delay);
    NN_SDK_ASSERT_NOT_NULL(GetParameter());

    // Check channel count
    if (IsSupportedDelayChannelCount(pUserInParam->_delayParameter._numChannelCountMax) == false)
    {
        return;
    }

    auto prevStatus = GetParameter()->_parameterStatus;

    m_AttachMixId = pUserInParam->_attachMixId;
    m_ProcessingOrder = pUserInParam->_processingOrder;
    *GetParameter() = pUserInParam->_delayParameter;
    if (IsSupportedDelayChannelCount(pUserInParam->_delayParameter._numChannels) == false)  // workaround for SIGLO-46782
    {
        GetParameter()->_numChannels = GetParameter()->_numChannelCountMax;
    }
    m_Enabled = pUserInParam->_enabled;

    if (prevStatus != EffectParameterStatus_Updated)
    {
        GetParameter()->_parameterStatus = prevStatus;
    }

    if (ShouldUpdateWorkBufferInfo(pUserInParam))
    {
        NN_SDK_ASSERT(nn::util::is_aligned(pUserInParam->_bufferBase.GetAddress(), nn::audio::BufferAlignSize)); // TODO: should return error, not assert.
        NN_SDK_ASSERT_EQUAL(nn::util::align_up(pUserInParam->_bufferSize, nn::audio::BufferAlignSize), pUserInParam->_bufferSize);

        SetUsage(UsageState::Initialized);
        GetParameter()->_parameterStatus = EffectParameterStatus_Init;

        m_ShouldSkip =
            poolMapper.TryAttachBuffer(
                pOutErroInfo,
                &m_WorkBuffer[Index_Delay],
                pUserInParam->_bufferBase,
                static_cast<size_t>(pUserInParam->_bufferSize)) ? false : true;
    }
    else
    {
        pOutErroInfo->SetResult(ResultSuccess(), 0);
    }
}

void DelayInfo::UpdateForCommandGeneration() NN_NOEXCEPT
{
    auto state = IsEnabled() ? EffectInfoBase::UsageState::Running : EffectInfoBase::UsageState::Stopped;
    SetUsage(state);
    GetParameter()->_parameterStatus = EffectParameterStatus_Updated;
}

nn::audio::DspAddr DelayInfo::GetWorkBuffer(Index index) const NN_NOEXCEPT
{
    NN_UNUSED(index);
    return GetSingleBuffer(server::EffectInfoBase::Index::Index_Delay);
}

// ================================================================================
// Reverb
// ================================================================================
void ReverbInfo::Update(common::BehaviorParameter::ErrorInfo* pOutErroInfo, const nn::audio::EffectInfo::InParameter* pUserInParam, PoolMapper& poolMapper) NN_NOEXCEPT
{
    NN_SDK_ASSERT_EQUAL(GetType(), EffectType_Reverb);
    NN_SDK_ASSERT_NOT_NULL(GetParameter());

    // Check channel count
    if (IsSupportedReverbChannelCount(pUserInParam->_reverbParameter._numChannelCountMax) == false)
    {
        return;
    }

    auto prevStatus = GetParameter()->_parameterStatus;
    m_AttachMixId = pUserInParam->_attachMixId;
    m_ProcessingOrder = pUserInParam->_processingOrder;
    *GetParameter() = pUserInParam->_reverbParameter;
    if (IsSupportedReverbChannelCount(pUserInParam->_reverbParameter._numChannels) == false)  // workaround for SIGLO-46782
    {
        GetParameter()->_numChannels = GetParameter()->_numChannelCountMax;
    }
    m_Enabled = pUserInParam->_enabled;

    if (prevStatus != EffectParameterStatus_Updated)
    {
        GetParameter()->_parameterStatus = prevStatus;
    }

    if (ShouldUpdateWorkBufferInfo(pUserInParam))
    {
        NN_SDK_ASSERT(nn::util::is_aligned(pUserInParam->_bufferBase.GetAddress(), nn::audio::BufferAlignSize)); // TODO: should return error, not assert.
        NN_SDK_ASSERT_EQUAL(nn::util::align_up(pUserInParam->_bufferSize, nn::audio::BufferAlignSize), pUserInParam->_bufferSize);

        SetUsage(UsageState::Initialized);
        GetParameter()->_parameterStatus = EffectParameterStatus_Init;

        m_ShouldSkip = poolMapper.TryAttachBuffer(
            pOutErroInfo,
            &m_WorkBuffer[Index_Reverb],
            pUserInParam->_bufferBase,
            static_cast<size_t>(pUserInParam->_bufferSize)) ? false : true;
    }
    else
    {
        pOutErroInfo->SetResult(ResultSuccess(), 0);
    }
}

void ReverbInfo::UpdateForCommandGeneration() NN_NOEXCEPT
{
    auto state = IsEnabled() ? EffectInfoBase::UsageState::Running : EffectInfoBase::UsageState::Stopped;
    SetUsage(state);
    GetParameter()->_parameterStatus = EffectParameterStatus_Updated;
}

nn::audio::DspAddr ReverbInfo::GetWorkBuffer(Index index) const NN_NOEXCEPT
{
    NN_UNUSED(index);
    return GetSingleBuffer(server::EffectInfoBase::Index::Index_Reverb);
}

// ================================================================================
// I3dl2Reverb
// ================================================================================
void I3dl2ReverbInfo::Update(common::BehaviorParameter::ErrorInfo* pOutErroInfo, const nn::audio::EffectInfo::InParameter* pUserInParam, PoolMapper& poolMapper) NN_NOEXCEPT
{
    NN_SDK_ASSERT_EQUAL(GetType(), EffectType_I3dl2Reverb);
    NN_SDK_ASSERT_NOT_NULL(GetParameter());

    // Check channel count
    if (IsSupportedI3dl2ReverbChannelCount(pUserInParam->_i3dl2ReverbParameter._numChannelCountMax) == false)
    {
        return;
    }

    auto prevStatus = GetParameter()->_parameterStatus;

    m_AttachMixId = pUserInParam->_attachMixId;
    m_ProcessingOrder = pUserInParam->_processingOrder;
    *GetParameter() = pUserInParam->_i3dl2ReverbParameter;
    if (IsSupportedI3dl2ReverbChannelCount(pUserInParam->_i3dl2ReverbParameter._numChannels) == false)  // workaround for SIGLO-46782
    {
        GetParameter()->_numChannels = GetParameter()->_numChannelCountMax;
    }
    m_Enabled = pUserInParam->_enabled;

    if (prevStatus != EffectParameterStatus_Updated)
    {
        GetParameter()->_parameterStatus = prevStatus;
    }

    if (ShouldUpdateWorkBufferInfo(pUserInParam))
    {
        NN_SDK_ASSERT(nn::util::is_aligned(pUserInParam->_bufferBase.GetAddress(), nn::audio::BufferAlignSize)); // TODO: should return error, not assert.
        NN_SDK_ASSERT_EQUAL(nn::util::align_up(pUserInParam->_bufferSize, nn::audio::BufferAlignSize), pUserInParam->_bufferSize);

        SetUsage(UsageState::Initialized);
        GetParameter()->_parameterStatus = EffectParameterStatus_Init;

        m_ShouldSkip = poolMapper.TryAttachBuffer(
            pOutErroInfo,
            &m_WorkBuffer[Index_I3dl2Reverb],
            pUserInParam->_bufferBase,
            static_cast<size_t>(pUserInParam->_bufferSize)) ? false : true;
    }
    else
    {
        pOutErroInfo->SetResult(ResultSuccess(), 0);
    }
}

void I3dl2ReverbInfo::UpdateForCommandGeneration() NN_NOEXCEPT
{
    auto state = IsEnabled() ? EffectInfoBase::UsageState::Running : EffectInfoBase::UsageState::Stopped;
    SetUsage(state);
    GetParameter()->_parameterStatus = EffectParameterStatus_Updated;
}

nn::audio::DspAddr I3dl2ReverbInfo::GetWorkBuffer(Index index) const NN_NOEXCEPT
{
    NN_UNUSED(index);
    return GetSingleBuffer(server::EffectInfoBase::Index::Index_I3dl2Reverb);
}

// ================================================================================
// Aux
// ================================================================================
void AuxInfo::Update(common::BehaviorParameter::ErrorInfo* pOutErroInfo, const nn::audio::EffectInfo::InParameter* pUserInParam, PoolMapper& poolMapper) NN_NOEXCEPT
{
    NN_SDK_ASSERT_EQUAL(GetType(), EffectType_Aux);
    NN_SDK_ASSERT_NOT_NULL(GetParameter());

    m_AttachMixId = pUserInParam->_attachMixId;
    m_ProcessingOrder = pUserInParam->_processingOrder;
    *GetParameter() = pUserInParam->_auxParameter;
    m_Enabled = pUserInParam->_enabled;

    if (ShouldUpdateWorkBufferInfo(pUserInParam))
    {
        auto addresses = &m_AuxAddresses;

        size_t infoSize = nn::util::align_up(sizeof(::nn::audio::AuxBufferInfo), nn::audio::BufferAlignSize);
        size_t auxCircularBufferSize = GetParameter()->_sampleCount * sizeof(int32_t);
        size_t auxBufferSize = infoSize + auxCircularBufferSize;

        auto sendMapped = poolMapper.TryAttachBuffer(
            pOutErroInfo,
            &m_WorkBuffer[Index_AuxSend],
            GetParameter()->_sendAuxBufferInfo,
            auxBufferSize);
        auto returnMapped = poolMapper.TryAttachBuffer(
            pOutErroInfo,
            &m_WorkBuffer[Index_AuxReturn],
            GetParameter()->_returnAuxBufferInfo,
            auxBufferSize);

        m_ShouldSkip = (sendMapped && returnMapped) ? false : true;

        // Fill addresses for AuxBufferDsp.
        if (sendMapped && returnMapped) // MemoryPool を用いて正しくマップされた場合
        {
            DspAddr dspAddrBase;
            dspAddrBase = m_WorkBuffer[Index_AuxSend].GetReference(false);
            addresses->_sendDspInfo = dspAddrBase + offsetof(AuxBufferInfo, _dsp);
            addresses->_sendBufferBase = dspAddrBase + sizeof(AuxBufferInfo);
            NN_SDK_ASSERT(addresses->_sendDspInfo);
            NN_SDK_ASSERT(addresses->_sendBufferBase);

            dspAddrBase = m_WorkBuffer[Index_AuxReturn].GetReference(false);
            addresses->_returnDspInfo = dspAddrBase + offsetof(AuxBufferInfo, _dsp);
            addresses->_returnBufferBase = dspAddrBase + sizeof(AuxBufferInfo);
            NN_SDK_ASSERT(addresses->_returnDspInfo);
            NN_SDK_ASSERT(addresses->_returnBufferBase);
        }
    }
    else
    {
        pOutErroInfo->SetResult(ResultSuccess(), 0);
    }
}

void AuxInfo::UpdateForCommandGeneration() NN_NOEXCEPT
{
    auto state = IsEnabled() ? EffectInfoBase::UsageState::Running : EffectInfoBase::UsageState::Stopped;
    SetUsage(state);
}

nn::audio::DspAddr AuxInfo::GetWorkBuffer(Index index) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_EQUAL(index, Index_Invalid);
    return m_WorkBuffer[index].GetReference();
}

// ================================================================================
// BiquadFilter
// ================================================================================
void BiquadFilterInfo::Update(common::BehaviorParameter::ErrorInfo* pOutErroInfo, const nn::audio::EffectInfo::InParameter* pUserInParam, PoolMapper& poolMapper) NN_NOEXCEPT
{
    NN_UNUSED(poolMapper);
    m_AttachMixId = pUserInParam->_attachMixId;
    m_ProcessingOrder = pUserInParam->_processingOrder;
    *GetParameter() = pUserInParam->_biquadFilterParameter;

    m_Enabled = pUserInParam->_enabled;

    const auto prevStatus = GetParameter()->_parameterStatus;
    if (prevStatus != EffectParameterStatus_Updated)
    {
        GetParameter()->_parameterStatus = prevStatus;
    }

    pOutErroInfo->_result = ResultSuccess();
    pOutErroInfo->_info = 0;
}

void BiquadFilterInfo::UpdateForCommandGeneration() NN_NOEXCEPT
{
    const auto state = IsEnabled() ? EffectInfoBase::UsageState::Running : EffectInfoBase::UsageState::Stopped;
    SetUsage(state);
    GetParameter()->_parameterStatus = EffectParameterStatus_Updated;
}

EffectContext::EffectContext() NN_NOEXCEPT
    : m_Infos(nullptr)
    , m_InfoCount(0)
{}

void EffectContext::Initialize(server::EffectInfoBase* infos, int infoCount) NN_NOEXCEPT
{
    m_Infos = infos;
    m_InfoCount = infoCount;
}

nn::audio::server::EffectInfoBase& EffectContext::GetInfo(int id) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_RANGE(id, 0, m_InfoCount);
    return m_Infos[id];
}

int EffectContext::GetCount() const NN_NOEXCEPT
{
    return m_InfoCount;
}

}  // namespace server
}  // namespace audio
}  // namespace nn
