﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/lmem/lmem_ExpHeap.h>

#include <nn/applet/applet_Types.h>
#include <nn/audio/audio_FinalOutputRecorderTypes.h>
#include <nn/audio/audio_AudioInTypes.h>
#include <nn/sf/impl/sf_ExpHeapAllocator.h>

#include <nn/audio/detail/audio_IFinalOutputRecorderManager.h>
#include <nn/audio/detail/audio_IFinalOutputRecorder.h>

#include <nn/audio/audio_Result.h>
#include <nn/os/os_Thread.h>

namespace nn{ namespace audio { namespace server {

class FinalOutputRecorderManagerImpl : public nn::sf::ISharedObject
{
public:
    FinalOutputRecorderManagerImpl() NN_NOEXCEPT;
    virtual ~FinalOutputRecorderManagerImpl() NN_NOEXCEPT;
    nn::Result OpenFinalOutputRecorder(nn::sf::Out<nn::sf::SharedPointer<detail::IFinalOutputRecorder>> outFinalOutputRecorder, nn::audio::FinalOutputRecorderParameter param, nn::sf::NativeHandle processHandle, nn::sf::Out<nn::audio::detail::FinalOutputRecorderParameterInternal> finalOutputRecorderInter, nn::applet::AppletResourceUserId appletId) NN_NOEXCEPT;
    void ReportCloseSession(int sessionId) NN_NOEXCEPT;
    nn::Result Sleep() NN_NOEXCEPT;
    nn::Result Wake() NN_NOEXCEPT;

    nn::os::Mutex* GetMutex() NN_NOEXCEPT;

    nn::Result GetAdspLoad(nn::sf::Out<int32_t> value) NN_NOEXCEPT;

public:
    static const int NumberOfFinalOutputRecorderSessions = 2;

private:
    static void ThreadFunc(void* arg) NN_NOEXCEPT;
    void ThreadFuncImpl() NN_NOEXCEPT;
    void FirstTimeInitialize();
    class FinalOutputRecorderImpl;
    typedef nn::sf::ExpHeapAllocator MyAllocator;
    MyAllocator m_Allocator;
    static const int SizeOfSession = 3 * 8 * 1024;
    static const int HeapBufferSize = NumberOfFinalOutputRecorderSessions * SizeOfSession;

    std::aligned_storage<HeapBufferSize>::type m_HeapBuffer;
    nn::lmem::HeapHandle m_HeapHandle;

    bool m_IsInitialized;
    os::Mutex m_Mutex;

    class SessionsCircularBuffer
    {
    public:
        SessionsCircularBuffer()
            : m_NextAvailableIndex(0)
            , m_NextFreeIndex(0)
            , m_CurrentAmount(NumberOfFinalOutputRecorderSessions)
        {
            for( int i = 0; i < NumberOfFinalOutputRecorderSessions; ++i )
            {
                m_AvailableSessions[i] = i;
            }
        }

        nn::Result GetNextSessions(int* sessions)
        {
            NN_RESULT_THROW_UNLESS(m_CurrentAmount > 0, ResultOutOfResource());
            *sessions = m_AvailableSessions[m_NextAvailableIndex];
            m_NextAvailableIndex = ( m_NextAvailableIndex + 1 ) % NumberOfFinalOutputRecorderSessions;
            --m_CurrentAmount;
            NN_RESULT_SUCCESS;
        }

        void InsertSession(int sessionId)
        {
            NN_SDK_ASSERT(sessionId >= 0);
            NN_SDK_ASSERT(sessionId < NumberOfFinalOutputRecorderSessions);
            NN_SDK_ASSERT(m_CurrentAmount < NumberOfFinalOutputRecorderSessions, "Can't hold more sessions (max %d)", NumberOfFinalOutputRecorderSessions);

            m_AvailableSessions[m_NextFreeIndex] = sessionId;
            m_NextFreeIndex = ( m_NextFreeIndex + 1 ) % NumberOfFinalOutputRecorderSessions;
            ++m_CurrentAmount;
        }
    private:

        int m_AvailableSessions[NumberOfFinalOutputRecorderSessions];
        int m_NextAvailableIndex;
        int m_NextFreeIndex;
        int m_CurrentAmount;
    };

    SessionsCircularBuffer m_Sessions;

    static const int SessionStackSize = 12 * 1024;
    char*            m_Stack;

    nn::os::SemaphoreType* m_pSemaphore;
    FinalOutputRecorderImpl* m_Session[NumberOfFinalOutputRecorderSessions];
    nn::os::EventType      m_Event[NumberOfFinalOutputRecorderSessions];
    nn::applet::AppletResourceUserId m_AppletId[NumberOfFinalOutputRecorderSessions];
    nn::os::EventType      m_EventSemaphoreSignal;

    std::atomic<bool> m_IsAwake;
    std::atomic<bool> m_IsThreadActive;
    os::ThreadType    m_Thread;
};
}}} //namespace nn / audio / server

