﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "../common/audio_Command.h"
#include "audio_PerformanceMetricsManager.h"
#include "audio_ServiceVoiceInfo.h"
#include "audio_ServiceMemoryPoolInfo.h"
#include "audio_ServiceMixInfo.h"
#include "audio_ServiceSinkInfo.h"
#include "audio_ServiceSplitterInfo.h"
#include "audio_UpsamplerManager.h"
#include "audio_CommandBuffer.h"
#include "audio_BaseAspect.h"

namespace nn { namespace audio { namespace server {

struct RendererSystemContext
{
    uint32_t sessionId;
    int32_t channelCount;
    int32_t mixBufferCount;
    int32_t* depopBuffer;
    UpsamplerManager* pUpsamplerManager;
    MemoryPoolInfo* pServicePool;
};

class CommandGenerator
{
public :
    CommandGenerator(
        ICommandBuffer* pCommandBuffer,
        const CommandListHeader* pListHeader,
        RendererSystemContext* sysContext,
        VoiceContext* voiceData, MixContext* mixData, EffectContext* effectData, SinkContext* sinkData,
        SplitterContext* splitterDestinationData,
        PerformanceManager* pPerformanceManager) NN_NOEXCEPT;

    void GenerateVoiceCommands() NN_NOEXCEPT;
    void GenerateSubMixCommands() NN_NOEXCEPT;
    void GenerateFinalMixCommands() NN_NOEXCEPT;
    void GenerateSinkCommands() NN_NOEXCEPT;
    static size_t CalculateCommandBufferSize(const nn::audio::detail::AudioRendererParameterInternal& parameter) NN_NOEXCEPT;

private:
    void GenerateDataSourceCommand(server::VoiceInfo& voice, VoiceState& state, int channel) NN_NOEXCEPT;
    void GenerateVoiceMixCommand(const float* pMixVolume, const float* pMixVolumePrev, VoiceState* state, int mixBufferOffset, int mixBufferCount, int voiceBufferIndex, NodeId nodeId) NN_NOEXCEPT;
    void GenerateVoiceCommand(server::VoiceInfo& voice) NN_NOEXCEPT;
    void GenerateBiquadFilterCommandForVoice(const server::VoiceInfo* pVoice, VoiceState* state, int mixBufferCount, int channel, NodeId nodeId) NN_NOEXCEPT;
    void GenerateMixCommands(server::MixInfo& pTargetMix) NN_NOEXCEPT;
    void GenerateSubMixCommand(server::MixInfo& pSubMixInfo) NN_NOEXCEPT;
    void GenerateFinalMixCommand() NN_NOEXCEPT;
    void GenerateSinkCommand(int mixBufferOffset, server::SinkInfoBase* pSinkInfo) NN_NOEXCEPT;
    void GenerateEffectCommand(const server::MixInfo& mixInfo) NN_NOEXCEPT;
    void GenerateCircularBufferSinkCommand(int mixBufferOffset, server::SinkInfoBase* pSinkInfo) NN_NOEXCEPT;
    void GenerateDeviceSinkCommand(int mixBufferOffset, server::SinkInfoBase* pSinkInfo) NN_NOEXCEPT;
    void GenerateBufferMixerCommand(int mixBufferOffset, EffectInfoBase& info, NodeId nodeId) NN_NOEXCEPT;
    void GenerateDelayCommand(int mixBufferOffset, EffectInfoBase& info, NodeId nodeId) NN_NOEXCEPT;
    void GenerateReverbCommand(int mixBufferOffset, EffectInfoBase& info, NodeId nodeId, bool longSizePreDelaySupported) NN_NOEXCEPT;
    void GenerateI3dl2ReverbEffectCommand(int mixBufferOffset, EffectInfoBase& info, NodeId nodeId) NN_NOEXCEPT;
    void GenerateAuxCommand(int mixBufferOffset, EffectInfoBase& info, NodeId nodeId) NN_NOEXCEPT;
    void GenerateBiquadFilterEffectCommand(int mixBufferOffset, EffectInfoBase& info, NodeId nodeId) NN_NOEXCEPT;

    server::SplitterDestinationData* GetDestinationData(common::SplitterInfoId infoId, int dataIndex) NN_NOEXCEPT;

    ICommandBuffer* m_pCommandBuffer;
    const CommandListHeader* m_pCommandListHeader;

    RendererSystemContext* m_SysContext;
    VoiceContext* m_VoiceContext;
    MixContext* m_MixContext;
    EffectContext* m_EffectContext;
    SinkContext* m_SinkContext;
    SplitterContext* m_SendContext;

public: // PerformanceMetrics related
    PerformanceManager* m_PerformanceManager; // TODO: can be set to private, need to refactor.
    void GeneratePerformanceCommand(const PerformanceEntryAddresses* entry, const PerformanceCommandType type, NodeId nodeId) NN_NOEXCEPT;
};

template <typename Aspect, typename Publisher>
class PerfAspect
{
public:
    PerfAspect(Publisher* publisher, PerformanceEntryType entry, NodeId id, PerformanceDetailType detail = PerformanceDetailType_Unknown)
        : m_Publisher(publisher)
        , m_Type(entry)
        , m_DetailType(detail)
        , m_NodeId(id)
    {
    }

    Aspect operator-> () NN_NOEXCEPT
    {
    }

private:
    Publisher* m_Publisher;
    PerformanceEntryType m_Type;
    PerformanceDetailType m_DetailType;
    NodeId m_NodeId;
};

class EntryAspect final
    : public BaseAspect<CommandGenerator>
{
private:
    PerformanceEntryAddresses m_Addresses;
    bool m_Published;
    NodeId m_NodeId;
public:
    EntryAspect(CommandGenerator* publisher, PerformanceEntryType type, NodeId id) NN_NOEXCEPT;
    ~EntryAspect() NN_NOEXCEPT;
};

class DetailAspect final
    : public BaseAspect<CommandGenerator>
{
private:
    PerformanceEntryAddresses m_Addresses;
    NodeId m_ParentId;
    bool m_Published;
public:
    DetailAspect(CommandGenerator* publisher, PerformanceEntryType parentType, NodeId parentId, PerformanceDetailType target) NN_NOEXCEPT;
    ~DetailAspect() NN_NOEXCEPT;
};

class NullAspect final
    : public BaseAspect<CommandBuffer>
{
public:
    NN_IMPLICIT NullAspect(CommandBuffer* publisher) NN_NOEXCEPT;
    ~NullAspect() NN_NOEXCEPT;
};

}}}  // namespace nn::audio::server
