﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstring>
#include <cstdint>
#include <algorithm>
#include <nn/dd/dd_Types.h>
#include <nn/applet/applet_Types.h>
#include <nn/os/os_TransferMemory.h>
#include <nn/os/os_Tick.h>
#include <nn/sf/sf_NativeHandle.h>
#include <nn/nn_Macro.h>
#include <nn/audio/audio_SampleFormat.h>
#include <nn/nn_TimeSpan.h>
#include <nn/audio/detail/audio_Log.h>


namespace nn { namespace audio { namespace server {

enum SessionType
{
    SessionType_Input,
    SessionType_Output,
    SessionType_Record,
    SessionType_Invalid,
};

struct SessionBuffer
{
    SessionBuffer() NN_NOEXCEPT
        : completionTime(0)
        , addr(nullptr)
        , mappedAddr(nullptr)
        , userAddr(nullptr)
        , size(0)
        {}
    SessionBuffer(void* _addr, const void* _userAddr, size_t _size) NN_NOEXCEPT
        : completionTime(0)
        , addr(_addr)
        , mappedAddr(nullptr)
        , userAddr(_userAddr)
        , size(_size)
        {}

    nn::TimeSpan completionTime;
    void* addr;
    void* mappedAddr;
    const void* userAddr;
    size_t size;
};

struct SessionFormat
{
    SampleFormat format;
    int32_t sampleRate;
    uint16_t channelCount;
};

enum SessionState
{
    SessionState_Started = 0,
    SessionState_Stopped,
};

template <int Capacity>
class SessionBufferList
{
public:
    explicit SessionBufferList(int deviceBufferMax) NN_NOEXCEPT
        : m_ReleasedBufferIndex(0)
        , m_ReleasedBufferCount(0)
        , m_DeviceAppendIndex(0)
        , m_DeviceAppendCount(0)
        , m_ServerAppendIndex(0)
        , m_ServerAppendCount(0)
        , m_MaxDeviceBufferCount(deviceBufferMax)
    {}

    void Reset() NN_NOEXCEPT
    {
        m_ReleasedBufferIndex = 0;
        m_ReleasedBufferCount = 0;
        m_DeviceAppendIndex = 0;
        m_DeviceAppendCount = 0;
        m_ServerAppendIndex = 0;
        m_ServerAppendCount = 0;
    }

    bool AppendServerBuffer(const SessionBuffer& buffer) NN_NOEXCEPT
    {
        auto availableBufferCount = Capacity - (m_ReleasedBufferCount + m_DeviceAppendCount + m_ServerAppendCount);
        if(availableBufferCount)
        {
            m_BufferList[m_ServerAppendIndex] = buffer;
            m_ServerAppendIndex = m_ServerAppendIndex == (Capacity - 1) ? 0 : m_ServerAppendIndex + 1;
            ++m_ServerAppendCount;
            return true;
        }
        return false;
    }

    int GetDeviceBuffersToAppend(SessionBuffer* buffers, int numBuffers) NN_NOEXCEPT
    {
        if(!buffers)
        {
            return 0;
        }
        int usableBuffers = std::min(numBuffers, m_ServerAppendCount);
        usableBuffers = std::min(usableBuffers, m_MaxDeviceBufferCount - m_DeviceAppendCount);
        auto index = m_DeviceAppendIndex;
        for(int i = 0; i < usableBuffers; ++i)
        {
            buffers[i] = m_BufferList[index];
            ++m_DeviceAppendCount;
            --m_ServerAppendCount;
            index = (index == Capacity - 1) ? 0 : index + 1;
        }
        m_DeviceAppendIndex = index;
        return usableBuffers;
    }

    const SessionBuffer* GetReleasedBuffer() NN_NOEXCEPT
    {
        if(m_ReleasedBufferCount > 0)
        {
            auto index = m_ReleasedBufferIndex - m_ReleasedBufferCount;
            if(index < 0)
            {
                index += Capacity;
            }
            --m_ReleasedBufferCount;
            return &m_BufferList[index];
        }
        return nullptr;
    }

    bool GetPlaybackBuffer(SessionBuffer* buffer) const NN_NOEXCEPT
    {
        if(m_DeviceAppendCount > 0)
        {
            auto index = m_DeviceAppendIndex - m_DeviceAppendCount;
            if(index < 0)
            {
                index += Capacity;
            }
            *buffer = m_BufferList[index];
            return true;
        }
        return false;
    }

    int GetMappedBuffers(SessionBuffer* buffers, int count) NN_NOEXCEPT
    {
        if(m_DeviceAppendCount + m_ServerAppendCount == 0)
        {
            return 0;
        }

        int writeIndex = 0;
        int endIndex = std::min(count, m_DeviceAppendCount + m_ServerAppendCount);

        auto listIndex = m_DeviceAppendIndex - m_DeviceAppendCount;
        if(listIndex < 0)
        {
            listIndex += Capacity;
        }
        for(int i = 0; i < m_DeviceAppendCount && writeIndex != endIndex; ++i)
        {
            if(listIndex < 0)
            {
                listIndex += Capacity;
            }
            buffers[writeIndex] = m_BufferList[listIndex];
            ++writeIndex;
            if(++listIndex == Capacity)
            {
                listIndex = 0;
            }
        }

        listIndex = m_ServerAppendIndex - m_ServerAppendCount;
        if(listIndex < 0)
        {
            listIndex += Capacity;
        }
        for(int i = 0; i < m_ServerAppendCount && writeIndex != endIndex; ++i)
        {

            buffers[writeIndex] = m_BufferList[listIndex];
            ++writeIndex;
            if(++listIndex == Capacity)
            {
                listIndex = 0;
            }
        }

        m_ReleasedBufferCount += m_ServerAppendCount + m_DeviceAppendCount;
        m_ReleasedBufferIndex += m_ServerAppendCount + m_DeviceAppendCount;

        m_ServerAppendCount = 0;
        m_DeviceAppendCount = 0;
        if(m_ReleasedBufferIndex >= Capacity)
        {
            m_ReleasedBufferIndex -= Capacity;
        }
        return writeIndex;
    }

    void* GetBufferAddress() const NN_NOEXCEPT
    {
        if(m_DeviceAppendCount > 0)
        {
            auto index = m_DeviceAppendIndex - m_DeviceAppendCount;
            if(index < 0)
            {
                index += Capacity;
            }
            return m_BufferList[index].addr;
        }
        return nullptr;
    }

    int GetAppendedBufferCount() const NN_NOEXCEPT
    {
        return m_DeviceAppendCount + m_ServerAppendCount;
    }

    int GetDeviceAppendedBufferCount() const NN_NOEXCEPT
    {
        return m_DeviceAppendCount;
    }

    int GetReleasedBufferCount() const NN_NOEXCEPT
    {
        return m_ReleasedBufferCount;
    }

    void NotifyBufferComplete() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_DeviceAppendCount > 0);
        --m_DeviceAppendCount;
        ++m_ReleasedBufferCount;
        m_BufferList[m_ReleasedBufferIndex].completionTime = nn::os::GetSystemTick().ToTimeSpan();
        m_ReleasedBufferIndex = m_ReleasedBufferIndex == (Capacity - 1) ? 0 : m_ReleasedBufferIndex + 1;
    }

    bool ContainsUserAddr(const void* userAddr) const NN_NOEXCEPT
    {
        if(m_ReleasedBufferCount + m_DeviceAppendCount + m_ServerAppendCount == 0)
        {
            return false;
        }

        auto startIndex = m_ReleasedBufferIndex - m_ReleasedBufferCount;
        if(startIndex < 0)
        {
            startIndex += Capacity;
        }
        auto endIndex = m_ServerAppendIndex;

        if(startIndex > endIndex)
        {
            for(int i = startIndex; i < endIndex; ++i)
            {
                if(m_BufferList[i].userAddr == userAddr)
                {
                    return true;
                }
            }
            return false;
        }
        else
        {
            for(int i = startIndex; i < Capacity; ++i)
            {
                if(m_BufferList[i].userAddr == userAddr)
                {
                    return true;
                }
            }
            for(int i = 0; i < endIndex; ++i)
            {
                if(m_BufferList[i].userAddr == userAddr)
                {
                    return true;
                }
            }
            return false;
        }
    }

    bool FlushBuffers() NN_NOEXCEPT
    {
        if(m_DeviceAppendCount != 0)
        {
            NN_DETAIL_AUDIO_ERROR("Flush failed (m_DeviceAppendCount == %d)\n", m_DeviceAppendCount);
            return false;
        }

        if(m_ServerAppendCount + m_ReleasedBufferCount > Capacity)
        {
            NN_DETAIL_AUDIO_ERROR("Flush failed (m_ServerAppendCount(%d) + m_ReleasedBufferCount(%d) > %d)\n", m_ServerAppendCount, m_ReleasedBufferCount, Capacity);
            return false;
        }

        m_ReleasedBufferCount += m_ServerAppendCount;
        m_ReleasedBufferIndex += m_ServerAppendCount;

        m_ServerAppendCount = 0;
        if(m_ReleasedBufferIndex >= Capacity)
        {
            m_ReleasedBufferIndex -= Capacity;
        }
        return true;
    }
private:
    SessionBuffer m_BufferList[Capacity];
    int m_ReleasedBufferIndex;
    int m_ReleasedBufferCount;
    int m_DeviceAppendIndex;
    int m_DeviceAppendCount;
    int m_ServerAppendIndex;
    int m_ServerAppendCount;
    const int m_MaxDeviceBufferCount;
};

template <int Capacity>
class TransferMemoryHelper
{
public:
    TransferMemoryHelper()
    {
        for(int i = 0; i < Capacity; ++i)
        {
            m_InUse[i] = false;
        }
    }
    void* Map(void* addr, size_t size, nn::sf::NativeHandle&& bufferHandle) NN_NOEXCEPT
    {
        for(int i = 0; i < Capacity; ++i)
        {
            if(!m_InUse[i])
            {
                m_TransferMemory[i].Attach(nn::util::align_up(size, nn::os::MemoryPageSize), bufferHandle.GetOsHandle(), bufferHandle.IsManaged());
                void* mappedAddr;
                m_TransferMemory[i].Map(&mappedAddr, nn::os::MemoryPermission_ReadWrite);
                m_InUse[i] = true;
                m_MappedAddr[i] = mappedAddr;
                bufferHandle.Detach();
                return mappedAddr;
            }
        }
        return nullptr;
    }

    bool Unmap(void* addr) NN_NOEXCEPT
    {
        for(int i = 0; i < Capacity; ++i)
        {
            if(m_InUse[i] && m_MappedAddr[i] == addr)
            {
                m_InUse[i] = false;
                m_TransferMemory[i].Unmap();
                nn::os::DestroyTransferMemory(m_TransferMemory[i].GetBase());
                return true;
            }
        }
        return false;
    }

    void UnmapAll() NN_NOEXCEPT
    {
        for(int i = 0; i < Capacity; ++i)
        {
            if(m_InUse[i])
            {
                m_InUse[i] = false;
                m_TransferMemory[i].Unmap();
                nn::os::DestroyTransferMemory(m_TransferMemory[i].GetBase());
            }
        }
    }

private:
    nn::os::TransferMemory m_TransferMemory[Capacity];
    void* m_MappedAddr[Capacity];
    bool m_InUse[Capacity];
};

}}} // nn::audio::server
