﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>

#include <nn/nn_Common.h>

#include <nn/os/os_TransferMemory.h>
#include <nn/sf/sf_ObjectFactory.h>

#include <nn/audio/audio_AudioOut.h>
#include <nn/audio/audio_AudioRenderer.h>

#include "../common/audio_AudioRendererSession.h"
#include "../dsp/audio_Dsp.h"
#include "../dsp/audio_DspProfiler.h"
#include "../dsp/audio_CommandProcessor.h"
#include "../common/audio_AudioRendererParameterInternal.h"

#include "audio_ServiceDiagnostics.h"
#include "audio_AudioRendererManagerImpl.h"
#include "audio_AudioRenderSystemManager.h"
#include "audio_SystemEventHolderPool.h"

namespace nn { namespace audio { namespace server {

namespace {
SystemEventHolderPool<nn::audio::common::AudioRendererSessionCountMax> g_SystemEventHolderPool;
} // anonymous namespace

class AudioRendererManagerImpl::AudioRendererImpl
{
public:
    explicit AudioRendererImpl(AudioRendererManagerImpl* pParent) NN_NOEXCEPT
        : m_Parent(pParent, true)
        , m_TransferMemory()
        , m_IsTransferMemoryMapped(false)
        , m_SystemEventHolder(g_SystemEventHolderPool.Acquire())
        , m_System(m_SystemEventHolder.GetSystemEvent())
    {
    }

    ~AudioRendererImpl() NN_NOEXCEPT
    {
        m_System.Finalize();
        g_SystemEventHolderPool.Release(m_SystemEventHolder);
        dsp::WaitForUnmap();

#if defined(NN_AUDIO_ENABLE_AUDIO_RENDER_SYSTEM_MANAGER)
        if(m_System.GetExecutionMode() == nn::audio::AudioRendererExecutionMode_AutoExecution)
        {
            AudioRenderSystemManager::GetInstance().Remove(m_System);
        }
#endif
        if(m_IsTransferMemoryMapped)
        {
            m_TransferMemory.Unmap();
        }
        m_Parent->ReleaseSessionId(m_System.GetSessionId());

        NN_AUDIO_SERVICE_LOG("[AudioRenderer] Close sessionId(%d)\n", m_System.GetSessionId());
    }

    nn::Result GetSampleRate(nn::sf::Out<std::int32_t> outSampleRate) NN_NOEXCEPT;
    nn::Result GetSampleCount(nn::sf::Out<std::int32_t> outSampleCount) NN_NOEXCEPT;
    nn::Result GetMixBufferCount(nn::sf::Out<std::int32_t> outMixBufferCount) NN_NOEXCEPT;
    nn::Result GetState(nn::sf::Out<std::int32_t> outState) NN_NOEXCEPT;
    nn::Result Start() NN_NOEXCEPT;
    nn::Result Stop() NN_NOEXCEPT;
    nn::Result SetRenderingTimeLimit(std::int32_t limitPercent) NN_NOEXCEPT;
    nn::Result GetRenderingTimeLimit(nn::sf::Out<std::int32_t> limitPercent) NN_NOEXCEPT;
    nn::Result RequestUpdate(const nn::sf::OutBuffer& outBuffer, const nn::sf::OutBuffer& performanceBuffer, const nn::sf::InBuffer& inBuffer) NN_NOEXCEPT;
    nn::Result RequestUpdateAuto(const nn::sf::OutBuffer& outBuffer, const nn::sf::OutBuffer& performanceBuffer, const nn::sf::InBuffer& inBuffer) NN_NOEXCEPT;
    nn::Result QuerySystemEvent(nn::sf::Out<nn::sf::NativeHandle> pOutHandle) NN_NOEXCEPT;
    nn::Result Initialize(const nn::audio::detail::AudioRendererParameterInternal& parameter, nn::sf::NativeHandle& workBufferHandle, nn::sf::NativeHandle& processHandle, uint64_t workBufferSize, const nn::applet::AppletResourceUserId& appletResourceUserId, int sessionId) NN_NOEXCEPT;
    nn::Result Initialize(const nn::audio::detail::AudioRendererParameterInternal& parameter, uint64_t workBufferAddress, nn::sf::NativeHandle& processHandle, uint64_t workBufferSize, const nn::applet::AppletResourceUserId& appletResourceUserId, int sessionId) NN_NOEXCEPT;
    nn::Result ExecuteAudioRendererRendering() NN_NOEXCEPT;

private:
    nn::Result RequestUpdateInternal(const nn::sf::OutBuffer& outBuffer, const nn::sf::OutBuffer& performanceBuffer, const nn::sf::InBuffer& inBuffer) NN_NOEXCEPT;

private:
    nn::sf::SharedPointer<AudioRendererManagerImpl> m_Parent;
    nn::os::TransferMemory m_TransferMemory;
    bool m_IsTransferMemoryMapped;
    SystemEventHolder& m_SystemEventHolder;
    AudioRenderSystem m_System;
};

nn::Result AudioRendererManagerImpl::AudioRendererImpl::GetSampleRate(nn::sf::Out<std::int32_t> outSampleRate) NN_NOEXCEPT
{
    *outSampleRate = m_System.GetSampleRate();
    NN_RESULT_SUCCESS;
}

nn::Result AudioRendererManagerImpl::AudioRendererImpl::GetSampleCount(nn::sf::Out<std::int32_t> outSampleCount) NN_NOEXCEPT
{
    *outSampleCount = m_System.GetSampleCount();
    NN_RESULT_SUCCESS;
}

nn::Result AudioRendererManagerImpl::AudioRendererImpl::GetMixBufferCount(nn::sf::Out<std::int32_t> outMixBufferCount) NN_NOEXCEPT
{
    *outMixBufferCount = m_System.GetMixBufferCount();
    NN_RESULT_SUCCESS;
}

nn::Result AudioRendererManagerImpl::AudioRendererImpl::GetState(nn::sf::Out<std::int32_t> outState) NN_NOEXCEPT
{
    *outState = m_System.IsActive() ? AudioRendererState_Started : AudioRendererState_Stopped;
    NN_RESULT_SUCCESS;
}

nn::Result AudioRendererManagerImpl::AudioRendererImpl::Start() NN_NOEXCEPT
{
    NN_AUDIO_SERVICE_LOG("[AudioRenderer] Start sessionId(%d)\n", m_System.GetSessionId());

    return m_System.Start();
}

nn::Result AudioRendererManagerImpl::AudioRendererImpl::Stop() NN_NOEXCEPT
{
    NN_AUDIO_SERVICE_LOG("[AudioRenderer] Stop sessionId(%d)\n", m_System.GetSessionId());

    m_System.Stop();
    NN_RESULT_SUCCESS;
}

nn::Result AudioRendererManagerImpl::AudioRendererImpl::SetRenderingTimeLimit(std::int32_t limitPercent) NN_NOEXCEPT
{
    m_System.SetRenderingTimeLimit(limitPercent);
    NN_RESULT_SUCCESS;
}

nn::Result AudioRendererManagerImpl::AudioRendererImpl::GetRenderingTimeLimit(nn::sf::Out<std::int32_t>limitPercent) NN_NOEXCEPT
{
    *limitPercent = m_System.GetRenderingTimeLimit();
    NN_RESULT_SUCCESS;
}

nn::Result AudioRendererManagerImpl::AudioRendererImpl::RequestUpdateInternal(const nn::sf::OutBuffer& outBuffer, const nn::sf::OutBuffer& performanceBuffer, const nn::sf::InBuffer& inBuffer) NN_NOEXCEPT
{
    return m_System.Update(outBuffer.GetPointerUnsafe(), outBuffer.GetSize(), performanceBuffer.GetPointerUnsafe(), performanceBuffer.GetSize(), inBuffer.GetPointerUnsafe(), inBuffer.GetSize());
}

nn::Result AudioRendererManagerImpl::AudioRendererImpl::RequestUpdate(const nn::sf::OutBuffer& outBuffer, const nn::sf::OutBuffer& performanceBuffer, const nn::sf::InBuffer& inBuffer) NN_NOEXCEPT
{
    return RequestUpdateInternal(outBuffer, performanceBuffer, inBuffer);
}

nn::Result AudioRendererManagerImpl::AudioRendererImpl::RequestUpdateAuto(const nn::sf::OutBuffer& outBuffer, const nn::sf::OutBuffer& performanceBuffer, const nn::sf::InBuffer& inBuffer) NN_NOEXCEPT
{
    return RequestUpdateInternal(outBuffer, performanceBuffer, inBuffer);
}

nn::Result AudioRendererManagerImpl::AudioRendererImpl::QuerySystemEvent(nn::sf::Out<nn::sf::NativeHandle> pOutHandle) NN_NOEXCEPT
{
    nn::os::NativeHandle handle;
    NN_RESULT_DO(m_System.QuerySystemEvent(&handle));
    pOutHandle.Set(nn::sf::NativeHandle(handle, false));
    NN_RESULT_SUCCESS;
}

nn::Result AudioRendererManagerImpl::AudioRendererImpl::ExecuteAudioRendererRendering() NN_NOEXCEPT
{
    const auto IsSupportedMethod = (m_System.GetExecutionMode() == nn::audio::AudioRendererExecutionMode_ManualExecution)
                              && (m_System.GetRenderingDevice() == nn::audio::AudioRendererRenderingDevice_Cpu);

    if(!IsSupportedMethod)
    {
        NN_RESULT_THROW(nn::audio::ResultNotSupported());
    }
    else
    {
#if defined(NN_AUDIO_ENABLE_CPU_RENDERER)
        dsp::CommandListProcessor commandListProcessor;

        const auto writtenSize = m_System.GenerateCommand(m_System.GetCommandBufferWorkBuffer(), m_System.GetCommandBufferWorkBufferSize());
        CommandListHeader* pHeader = reinterpret_cast<CommandListHeader*>(m_System.GetCommandBufferWorkBuffer());
        commandListProcessor.SetProcessTimeMax(std::numeric_limits<uint32_t>::max());
        commandListProcessor.Setup(pHeader, writtenSize);
        commandListProcessor.Process();
#else
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::audio::ResultNotImplemented());
#endif
    }

    NN_RESULT_SUCCESS;
}

nn::Result AudioRendererManagerImpl::AudioRendererImpl::Initialize(const nn::audio::detail::AudioRendererParameterInternal& parameter, nn::sf::NativeHandle& workBufferHandle, nn::sf::NativeHandle& processHandle, uint64_t workBufferSize, const nn::applet::AppletResourceUserId& appletResourceUserId, int sessionId) NN_NOEXCEPT
{
#if defined(NN_AUDIO_ENABLE_AUDIO_RENDER_SYSTEM_MANAGER)
    // TODO: 関数末尾に移動する
    if(static_cast<nn::audio::AudioRendererExecutionMode>(parameter.executionMode) == nn::audio::AudioRendererExecutionMode_AutoExecution)
    {
        // Remove() will be called in ~AudioRendererManagerImpl().
        NN_RESULT_THROW_UNLESS(AudioRenderSystemManager::GetInstance().Add(m_System), ResultOutOfResource());
    }
#endif // defined(NN_AUDIO_ENABLE_AUDIO_RENDER_SYSTEM_MANAGER)

    void* workBuffer = nullptr;
    size_t size = workBufferSize & 0xffffffff;

    m_TransferMemory.Attach(size, workBufferHandle.GetOsHandle(), workBufferHandle.IsManaged());
    workBufferHandle.Detach();
    NN_RESULT_DO(m_TransferMemory.Map(&workBuffer, nn::os::MemoryPermission_None));
    m_IsTransferMemoryMapped = true;

    NN_RESULT_DO(m_System.Initialize(parameter, processHandle.GetOsHandle(), workBuffer, workBufferSize & 0xffffffff, appletResourceUserId, sessionId));

    processHandle.Detach();


    NN_RESULT_SUCCESS;
}

nn::Result AudioRendererManagerImpl::AudioRendererImpl::Initialize(const nn::audio::detail::AudioRendererParameterInternal& parameter, uint64_t workBufferAddress, nn::sf::NativeHandle& processHandle, uint64_t workBufferSize, const nn::applet::AppletResourceUserId& appletResourceUserId, int sessionId) NN_NOEXCEPT
{
#if defined(NN_AUDIO_ENABLE_AUDIO_RENDER_SYSTEM_MANAGER)
    // TODO: 関数末尾に移動する
    if(static_cast<nn::audio::AudioRendererExecutionMode>(parameter.executionMode) == nn::audio::AudioRendererExecutionMode_AutoExecution)
    {
        // Remove() will be called in ~AudioRendererManagerImpl().
        NN_RESULT_THROW_UNLESS(AudioRenderSystemManager::GetInstance().Add(m_System), ResultOutOfResource());
    }
#endif // defined(NN_AUDIO_ENABLE_AUDIO_RENDER_SYSTEM_MANAGER)

    NN_RESULT_DO(m_System.Initialize(parameter, processHandle.GetOsHandle(), reinterpret_cast<void*>(workBufferAddress), workBufferSize & 0xffffffff, appletResourceUserId, sessionId));
    processHandle.Detach();

    NN_RESULT_SUCCESS;
}

AudioRendererManagerImpl::AudioRendererManagerImpl() NN_NOEXCEPT
    : m_ActiveSessionCount(0)
    , m_ActiveSessionCountMutex(false)
{
    // 厳密にはメモリ管理構造体のサイズなどを考慮する必要がある
    NN_STATIC_ASSERT(sizeof(AudioRenderSystem) * common::AudioRendererSessionCountMax < sizeof(m_HeapBuffer));

    m_HeapHandle = nn::lmem::CreateExpHeap(&m_HeapBuffer, sizeof(m_HeapBuffer), nn::lmem::CreationOption_NoOption);
    m_Allocator.Attach(m_HeapHandle);

    // Generate session IDs
    for(int i = 0; i < common::AudioRendererSessionCountMax; ++i)
    {
        m_SessionIdPool[i] = i;
    }

    AudioRenderSystemManager::CreateInstance(&m_AudioRendererSystemManagerBuffer, sizeof(m_AudioRendererSystemManagerBuffer));
}

AudioRendererManagerImpl::~AudioRendererManagerImpl() NN_NOEXCEPT
{
    AudioRenderSystemManager::DeleteInstance();
    nn::lmem::DestroyExpHeap(m_HeapHandle);
}

nn::Result AudioRendererManagerImpl::OpenAudioRenderer(nn::sf::Out<nn::sf::SharedPointer<detail::IAudioRenderer>> outAudioRenderer, const nn::audio::detail::AudioRendererParameterInternal& parameter, nn::sf::NativeHandle workBufferHandle, nn::sf::NativeHandle processHandle, uint64_t workBufferSize, nn::applet::AppletResourceUserId appletResourceUserId) NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_OS_WIN)  // TODO: remove this condition after implementing ListAudioOut for Horizon
    AudioOutInfo audioOutInfo;
    if (ListAudioOuts(&audioOutInfo, 1) == 0)
    {
        NN_RESULT_THROW(ResultNoAudioDeviceFound());
    }
#endif // defined(NN_BUILD_CONFIG_OS_WIN)

    // Acquire session ID
    const auto sessionId = AcquireSessionId();

    typedef nn::sf::ObjectFactory<MyAllocator::Policy> Factory;
    auto p = Factory::CreateSharedEmplaced<detail::IAudioRenderer, AudioRendererManagerImpl::AudioRendererImpl>(&m_Allocator, this);
    NN_RESULT_DO(Factory::GetEmplacedImplPointer<AudioRendererManagerImpl::AudioRendererImpl>(p)->Initialize(parameter, workBufferHandle, processHandle, workBufferSize, appletResourceUserId, sessionId));

    *outAudioRenderer = std::move(p);

    NN_AUDIO_SERVICE_LOG(
            "[AudioRenderer] Open sessionId(%d) aruid.lower(%llu) workBufferSize(%llu) parameter(%d %d %d %d %d %d %d %d %d %d)\n",
            sessionId, appletResourceUserId.lower, workBufferSize,
            parameter.sampleRate,
            parameter.sampleCount,
            parameter.mixBufferCount,
            parameter.subMixCount,
            parameter.voiceCount,
            parameter.sinkCount,
            parameter.effectCount,
            parameter.performanceFrameCount,
            parameter.isVoiceDropEnabled,
            parameter.splitterCount,
            parameter.splitterSendChannelCount,
            parameter.renderingDevice,
            parameter.executionMode);

    NN_RESULT_SUCCESS;
}

nn::Result AudioRendererManagerImpl::OpenAudioRendererForManualExecution(nn::sf::Out<nn::sf::SharedPointer<nn::audio::detail::IAudioRenderer>> outAudioRenderer,
        const nn::audio::detail::AudioRendererParameterInternal& parameter, std::uint64_t workBufferAddress, nn::sf::NativeHandle&& processHandle, std::uint64_t size, nn::applet::AppletResourceUserId id) NN_NOEXCEPT
{
    // Acquire session ID
    const auto sessionId = AcquireSessionId();

    typedef nn::sf::ObjectFactory<MyAllocator::Policy> Factory;
    auto p = Factory::CreateSharedEmplaced<detail::IAudioRenderer, AudioRendererManagerImpl::AudioRendererImpl>(&m_Allocator, this);
    NN_RESULT_DO(Factory::GetEmplacedImplPointer<AudioRendererManagerImpl::AudioRendererImpl>(p)->Initialize(parameter, workBufferAddress, processHandle, size, id, sessionId));

    *outAudioRenderer = std::move(p);

    NN_RESULT_SUCCESS;
}

int AudioRendererManagerImpl::AcquireSessionId() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_ActiveSessionCountMutex);

    NN_SDK_ASSERT_LESS(m_ActiveSessionCount, common::AudioRendererSessionCountMax);
    const auto sessionId = m_SessionIdPool[m_ActiveSessionCount];
    m_SessionIdPool[m_ActiveSessionCount] = common::AudioRendererInvalidSessionId;
    ++m_ActiveSessionCount;

    return sessionId;
}

void AudioRendererManagerImpl::ReleaseSessionId(int sessionId) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_ActiveSessionCountMutex);

    NN_SDK_ASSERT_GREATER(m_ActiveSessionCount, 0);
    --m_ActiveSessionCount;
    m_SessionIdPool[m_ActiveSessionCount] = sessionId;
}

nn::Result AudioRendererManagerImpl::GetAudioDeviceService(nn::sf::Out<nn::sf::SharedPointer<::nn::audio::detail::IAudioDevice>> outAudioDevice,
                                                           nn::applet::AppletResourceUserId& appletResourceUserId) NN_NOEXCEPT
{
    typedef nn::sf::ObjectFactory<MyAllocator::Policy> Factory;
    auto p = Factory::CreateSharedEmplaced<::nn::audio::detail::IAudioDevice, AudioRendererManagerImpl::AudioDeviceImpl>(&m_Allocator, this, appletResourceUserId, common::GetInitialReleaseRevision());
    NN_SDK_ASSERT_NOT_NULL(p);
    *outAudioDevice = std::move(p);
    NN_RESULT_SUCCESS;
}

nn::Result AudioRendererManagerImpl::GetAudioDeviceServiceWithRevisionInfo(nn::sf::Out<nn::sf::SharedPointer<::nn::audio::detail::IAudioDevice>> outAudioDevice,
                                                           nn::applet::AppletResourceUserId& appletResourceUserId, uint32_t revisionInfo) NN_NOEXCEPT
{
    typedef nn::sf::ObjectFactory<MyAllocator::Policy> Factory;
    auto p = Factory::CreateSharedEmplaced<::nn::audio::detail::IAudioDevice, AudioRendererManagerImpl::AudioDeviceImpl>(&m_Allocator, this, appletResourceUserId, revisionInfo);
    NN_SDK_ASSERT_NOT_NULL(p);
    *outAudioDevice = std::move(p);
    NN_RESULT_SUCCESS;
}

nn::Result AudioRendererManagerImpl::GetWorkBufferSize(nn::sf::Out<int64_t> outSize, const nn::audio::detail::AudioRendererParameterInternal& parameter) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(common::CheckValidRevision(parameter._magic), ResultOperationFailed());
    *outSize = AudioRenderSystem::GetWorkBufferSize(parameter);
    NN_RESULT_SUCCESS;
}

void AudioRendererManagerImpl::Sleep() NN_NOEXCEPT
{
#if defined(NN_AUDIO_ENABLE_AUDIO_RENDER_SYSTEM_MANAGER)
    AudioRenderSystemManager::GetInstance().Sleep();
#endif // defined(NN_AUDIO_ENABLE_AUDIO_RENDER_SYSTEM_MANAGER)
}

void AudioRendererManagerImpl::Wake() NN_NOEXCEPT
{
#if defined(NN_AUDIO_ENABLE_AUDIO_RENDER_SYSTEM_MANAGER)
    AudioRenderSystemManager::GetInstance().Wake();
#endif // defined(NN_AUDIO_ENABLE_AUDIO_RENDER_SYSTEM_MANAGER)
}

}}}  // namespace nn::audio::server
