﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <nn/nn_Common.h>
#include <nn/audio/audio_Applet.h>
#include "audio_AudioRendererManagerForAppletImpl.h"
#include "audio_AudioRenderSystemManager.h"
#include "audio_AppletVolumeManager.h"
#include "audio_AudioManagerForAppletImplCommon.h"

namespace nn { namespace audio { namespace server {

nn::Result AudioRendererManagerForAppletImpl::RequestSuspend(const nn::applet::AppletResourceUserId& id, std::int64_t fadeTimeNanoSeconds) NN_NOEXCEPT
{
    NN_AUDIO_APPLET_API_LOG("[AudioRenderer] id.lower(%llu) fadeTimeNanoSeconds(%lld)\n", id.lower, fadeTimeNanoSeconds);

    const auto fadeTime = nn::TimeSpan::FromNanoSeconds(fadeTimeNanoSeconds);
    NN_RESULT_THROW_UNLESS(nn::TimeSpan::FromMilliSeconds(0) <= fadeTime && fadeTime <= nn::TimeSpan::FromMilliSeconds(10), ResultOperationFailed());

    return AppletVolumeManager::Suspend(AppletVolumeManager::SessionType_AudioRenderer, fadeTimeNanoSeconds, id);
}

nn::Result AudioRendererManagerForAppletImpl::RequestResume(const nn::applet::AppletResourceUserId& id, std::int64_t fadeTimeNanoSeconds) NN_NOEXCEPT
{
    NN_AUDIO_APPLET_API_LOG("[AudioRenderer] id.lower(%llu) fadeTimeNanoSeconds(%lld)\n", id.lower, fadeTimeNanoSeconds);

    const auto fadeTime = nn::TimeSpan::FromNanoSeconds(fadeTimeNanoSeconds);
    NN_RESULT_THROW_UNLESS(nn::TimeSpan::FromMilliSeconds(0) <= fadeTime && fadeTime <= nn::TimeSpan::FromMilliSeconds(10), ResultOperationFailed());

    return AppletVolumeManager::Resume(AppletVolumeManager::SessionType_AudioRenderer, fadeTimeNanoSeconds, id);
}

nn::Result AudioRendererManagerForAppletImpl::GetProcessMasterVolume(nn::sf::Out<float> pOutVolume, const nn::applet::AppletResourceUserId& id) NN_NOEXCEPT
{
    NN_AUDIO_APPLET_API_LOG("[AudioRenderer] id.lower(%llu)\n", id.lower);

    return AppletVolumeManager::GetAppletVolume(pOutVolume.GetPointer(), AppletVolumeManager::SessionType_AudioRenderer, id);
}

nn::Result AudioRendererManagerForAppletImpl::SetProcessMasterVolume(const nn::applet::AppletResourceUserId& id, float volume, std::int64_t fadeTimeNanoSeconds) NN_NOEXCEPT
{
    NN_AUDIO_APPLET_API_LOG("[AudioRenderer] id.lower(%llu) volume(%.2f) fadeTimeNanoSeconds(%lld)\n", id.lower, volume, fadeTimeNanoSeconds);

    const auto fadeTime = nn::TimeSpan::FromNanoSeconds(fadeTimeNanoSeconds);
    NN_RESULT_THROW_UNLESS(nn::audio::ProcessMasterVolumeMin <= volume && volume <= nn::audio::ProcessMasterVolumeMax, ResultOperationFailed());
    NN_RESULT_THROW_UNLESS(nn::TimeSpan::FromMilliSeconds(0) <= fadeTime && fadeTime <= nn::TimeSpan::FromMilliSeconds(1000), ResultOperationFailed());

    return AppletVolumeManager::SetAppletVolume(AppletVolumeManager::SessionType_AudioRenderer, volume, fadeTimeNanoSeconds, id);
}

nn::Result AudioRendererManagerForAppletImpl::RegisterAppletResourceUserId(nn::applet::AppletResourceUserId id) NN_NOEXCEPT
{
    NN_AUDIO_APPLET_API_LOG("[AudioRenderer] id.lower(%llu)\n", id.lower);

    return AppletVolumeManager::RegisterAppletResourceUserId(id);
}

nn::Result AudioRendererManagerForAppletImpl::UnregisterAppletResourceUserId(nn::applet::AppletResourceUserId id) NN_NOEXCEPT
{
    NN_AUDIO_APPLET_API_LOG("[AudioRenderer] id.lower(%llu)\n", id.lower);

    return AppletVolumeManager::UnregisterAppletResourceUserId(id);
}

nn::Result AudioRendererManagerForAppletImpl::GetProcessRecordVolume(nn::sf::Out<float> pOutVolume, const nn::applet::AppletResourceUserId& id) NN_NOEXCEPT
{
    NN_AUDIO_APPLET_API_LOG("[AudioRenderer] id.lower(%llu)\n", id.lower);

    return AppletVolumeManager::GetRecordVolume(pOutVolume.GetPointer(), AppletVolumeManager::SessionType_AudioRenderer, id);
}

nn::Result AudioRendererManagerForAppletImpl::SetProcessRecordVolume(const nn::applet::AppletResourceUserId& id, float volume, std::int64_t fadeTimeNanoSeconds) NN_NOEXCEPT
{
    NN_AUDIO_APPLET_API_LOG("[AudioRenderer] id.lower(%llu) volume(%.2f) fadeTimeNanoSeconds(%lld)\n", id.lower, volume, fadeTimeNanoSeconds);

    const auto fadeTime = nn::TimeSpan::FromNanoSeconds(fadeTimeNanoSeconds);
    NN_RESULT_THROW_UNLESS(nn::audio::ProcessRecordVolumeMin <= volume && volume <= nn::audio::ProcessRecordVolumeMax, ResultOperationFailed());
    NN_RESULT_THROW_UNLESS(nn::TimeSpan::FromMilliSeconds(0) <= fadeTime && fadeTime <= nn::TimeSpan::FromMilliSeconds(1000), ResultOperationFailed());

    return AppletVolumeManager::SetRecordVolume(AppletVolumeManager::SessionType_AudioRenderer, volume, fadeTimeNanoSeconds, id);
}

}}}  // namespace nn::audio::server
