﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os/os_TransferMemory.h>
#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/dd/dd_DeviceAddressSpace.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/audio/audio_ResultForPrivate.h>
#include "../common/audio_AudioInSession.h"
#include "audio_SystemEventHolderPool.h"

#include "audio_ServiceDiagnostics.h"
#include "audio_AudioInSession.h"
#include "audio_AudioInManagerImpl.h"
#include "../common/audio_AudioInPrivate.h"

namespace nn { namespace audio { namespace server {

namespace {
nn::audio::server::SystemEventHolderPool<nn::audio::common::AudioInSessionCountMax> g_SystemEventHolderPool;
} // anonymous namespace

class AudioInManagerImpl::AudioInImpl
{
public:
    explicit AudioInImpl(AudioInManagerImpl* pParent, uint32_t sessionId, nn::os::EventType* pEvent) NN_NOEXCEPT
        : m_SystemEventHolder(g_SystemEventHolderPool.Acquire())
        , m_AudioInClass(sessionId, m_SystemEventHolder.GetSystemEvent())
        , m_Parent(pParent, true)
        , m_IsUacDevice(false)
    {
        NN_UNUSED(pEvent);
        m_pParentMutex = m_Parent->GetMutex();
    }

    ~AudioInImpl() NN_NOEXCEPT
    {
        m_AudioInClass.Close();
        m_Parent->ReportCloseSession(m_AudioInClass.GetSessionId());
        g_SystemEventHolderPool.Release(m_SystemEventHolder);

        NN_AUDIO_SERVICE_LOG("[AudioIn] Close sessionId(%d)\n", m_AudioInClass.GetSessionId());
    }

    nn::Result GetAudioInState(nn::sf::Out<uint32_t> outAudioIn) NN_NOEXCEPT;
    nn::Result Start() NN_NOEXCEPT;
    nn::Result Stop() NN_NOEXCEPT;
    nn::Result Sleep() NN_NOEXCEPT;
    nn::Result Wake() NN_NOEXCEPT;
    nn::Result AppendAudioInBuffer(nn::sf::InBuffer inAudioBuffer, uint64_t outBufferClientPtr) NN_NOEXCEPT;
    nn::Result AppendAudioInBufferAuto(nn::sf::InBuffer inAudioBuffer, uint64_t outBufferClientPtr) NN_NOEXCEPT;
    nn::Result AppendUacInBuffer(nn::sf::InBuffer inAudioBuffer, uint64_t outBufferClientPtr, nn::sf::NativeHandle bufferHandle) NN_NOEXCEPT;
    nn::Result AppendUacInBufferAuto(nn::sf::InBuffer inAudioBuffer, uint64_t outBufferClientPtr, nn::sf::NativeHandle bufferHandle) NN_NOEXCEPT;
    nn::Result RegisterBufferEvent(nn::sf::Out<nn::sf::NativeHandle> bufferEvent) NN_NOEXCEPT;
    nn::Result GetReleasedAudioInBuffers(nn::sf::OutBuffer outAudioBuffer, nn::sf::Out<int> count) NN_NOEXCEPT;
    nn::Result GetReleasedAudioInBuffersAuto(nn::sf::OutBuffer outAudioBuffer, nn::sf::Out<int> count) NN_NOEXCEPT;
    nn::Result GetAudioInBufferCount(nn::sf::Out<int> outAudioInBufferCount);
    nn::Result ContainsAudioInBuffer(uint64_t audioBufferPointer, nn::sf::Out<bool> contains) NN_NOEXCEPT;
    nn::Result Initialize(nn::sf::InBuffer name,  nn::audio::AudioInParameter param, nn::sf::NativeHandle& processHandle, nn::sf::Out<nn::audio::detail::AudioInParameterInternal> audioInInter, nn::sf::OutBuffer nameOut, const nn::applet::AppletResourceUserId appletId) NN_NOEXCEPT;
    nn::Result SetDeviceGain(float gain) NN_NOEXCEPT;
    nn::Result GetDeviceGain(nn::sf::Out<float> gain) NN_NOEXCEPT;
    nn::Result Update() NN_NOEXCEPT;
    bool IsUacDevice() NN_NOEXCEPT;

    static bool IsUacDeviceName(const char* name) NN_NOEXCEPT
    {
        return strncmp(name, nn::audio::common::UacDevicePrefix, nn::audio::common::UacDevicePrefixSize) == 0;
    }

    // W/A Utility | only for bug fix purpose, do not use in other case.
    void RollBackAudioInName(nn::sf::OutBuffer nameOut) NN_NOEXCEPT; // See SIGLO-70562

    uint32_t GetSessionId() NN_NOEXCEPT
    {
        return m_AudioInClass.GetSessionId();
    }

private:
    nn::Result AppendAudioInBufferInternal(nn::sf::InBuffer inAudioBuffer, uint64_t outBufferClientPtr) NN_NOEXCEPT;
    nn::Result AppendUacInBufferInternal(nn::sf::InBuffer inAudioBuffer, uint64_t outBufferClientPtr, nn::sf::NativeHandle& bufferHandle) NN_NOEXCEPT;
    nn::Result GetReleasedAudioInBuffersInternal(nn::sf::OutBuffer outAudioBuffer, nn::sf::Out<int> count) NN_NOEXCEPT;

private:
    SystemEventHolder& m_SystemEventHolder;
    nn::audio::detail::AudioInSession m_AudioInClass;
    nn::os::Mutex* m_pParentMutex;

    nn::sf::SharedPointer<AudioInManagerImpl> m_Parent;
    bool m_IsUacDevice;
};

nn::Result AudioInManagerImpl::AudioInImpl::GetAudioInState(nn::sf::Out<uint32_t> outAudioIn) NN_NOEXCEPT
{
    outAudioIn.Set(m_AudioInClass.GetState());
    NN_RESULT_SUCCESS;
}

nn::Result AudioInManagerImpl::AudioInImpl::Start() NN_NOEXCEPT
{
    NN_AUDIO_SERVICE_LOG("[AudioIn] Start sessionId(%d)\n", GetSessionId());

    std::lock_guard<os::Mutex> lock(*m_pParentMutex);
    return m_AudioInClass.Start();
}

nn::Result AudioInManagerImpl::AudioInImpl::Stop() NN_NOEXCEPT
{
    NN_AUDIO_SERVICE_LOG("[AudioIn] Stop sessionId(%d)\n", GetSessionId());

    std::lock_guard<os::Mutex> lock(*m_pParentMutex);
    m_AudioInClass.Stop();
    NN_RESULT_SUCCESS;
}

nn::Result AudioInManagerImpl::AudioInImpl::Sleep() NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(*m_pParentMutex);
    m_AudioInClass.Sleep();
    NN_RESULT_SUCCESS;
}

nn::Result AudioInManagerImpl::AudioInImpl::Wake() NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(*m_pParentMutex);
    m_AudioInClass.Wake();
    NN_RESULT_SUCCESS;
}

nn::Result AudioInManagerImpl::AudioInImpl::AppendAudioInBufferInternal(nn::sf::InBuffer inAudioBuffer, uint64_t inBufferClientPtr) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(*m_pParentMutex);
    const AudioInBuffer* pTempBuffer = reinterpret_cast<const AudioInBuffer*>(inAudioBuffer.GetPointerUnsafe());
    auto couldAppend = m_AudioInClass.AppendBuffer(pTempBuffer->buffer, pTempBuffer->size, reinterpret_cast<const void*>(inBufferClientPtr));
    NN_RESULT_THROW_UNLESS(couldAppend, nn::audio::ResultBufferCountMax());
    NN_RESULT_SUCCESS;
}

nn::Result AudioInManagerImpl::AudioInImpl::AppendAudioInBuffer(nn::sf::InBuffer inAudioBuffer, uint64_t inBufferClientPtr) NN_NOEXCEPT
{
    return AppendAudioInBufferInternal(inAudioBuffer, inBufferClientPtr);
}

nn::Result AudioInManagerImpl::AudioInImpl::AppendAudioInBufferAuto(nn::sf::InBuffer inAudioBuffer, uint64_t inBufferClientPtr) NN_NOEXCEPT
{
    return AppendAudioInBufferInternal(inAudioBuffer, inBufferClientPtr);
}

nn::Result AudioInManagerImpl::AudioInImpl::AppendUacInBufferInternal(nn::sf::InBuffer inAudioBuffer, uint64_t inBufferClientPtr, nn::sf::NativeHandle& bufferHandle) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(*m_pParentMutex);
    const AudioInBuffer* pTempBuffer = reinterpret_cast<const AudioInBuffer*>(inAudioBuffer.GetPointerUnsafe());
    auto couldAppend = m_AudioInClass.AppendBuffer(pTempBuffer->buffer, pTempBuffer->size, reinterpret_cast<const void*>(inBufferClientPtr), std::move(bufferHandle));
    NN_RESULT_THROW_UNLESS(couldAppend, nn::audio::ResultBufferCountMax());
    NN_RESULT_SUCCESS;
}

nn::Result AudioInManagerImpl::AudioInImpl::AppendUacInBuffer(nn::sf::InBuffer inAudioBuffer, uint64_t inBufferClientPtr, nn::sf::NativeHandle bufferHandle) NN_NOEXCEPT
{
    return AppendUacInBufferInternal(inAudioBuffer, inBufferClientPtr, bufferHandle);
}

nn::Result AudioInManagerImpl::AudioInImpl::AppendUacInBufferAuto(nn::sf::InBuffer inAudioBuffer, uint64_t inBufferClientPtr, nn::sf::NativeHandle bufferHandle) NN_NOEXCEPT
{
    return AppendUacInBufferInternal(inAudioBuffer, inBufferClientPtr, bufferHandle);
}

nn::Result AudioInManagerImpl::AudioInImpl::RegisterBufferEvent(nn::sf::Out<nn::sf::NativeHandle>  bufferEvent) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(*m_pParentMutex);
    nn::os::NativeHandle handle;
    m_AudioInClass.RegisterBufferEvent(&handle);
    bufferEvent.Set(nn::sf::NativeHandle(handle, false));
    NN_RESULT_SUCCESS;
}

nn::Result AudioInManagerImpl::AudioInImpl::GetReleasedAudioInBuffersInternal(nn::sf::OutBuffer outAudioBuffer, nn::sf::Out<int> count) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(*m_pParentMutex);
    uintptr_t* pAudioArray = reinterpret_cast<uintptr_t*>(outAudioBuffer.GetPointerUnsafe());
    pAudioArray[0] = 0;
    size_t length = outAudioBuffer.GetSize() / sizeof(*pAudioArray);

    int loops = 0;
    for( unsigned int i = 0; i < length; ++i )
    {
        auto sessionBuffer = m_AudioInClass.GetReleasedBuffer();

        if(!sessionBuffer)
        {
            break;
        }

        void* userAddr = const_cast<void*>(sessionBuffer->userAddr);
        nn::audio::AudioInBuffer* pAudioBuffer = reinterpret_cast<nn::audio::AudioInBuffer*>(userAddr);

        if( pAudioBuffer != nullptr )
        {
            pAudioArray[i] = reinterpret_cast<uintptr_t>(pAudioBuffer);
            ++loops;
        }
        else
        {
            break;
        }
    }
    count.Set(loops);
    NN_RESULT_SUCCESS;
}

nn::Result AudioInManagerImpl::AudioInImpl::GetReleasedAudioInBuffers(nn::sf::OutBuffer outAudioBuffer, nn::sf::Out<int> count) NN_NOEXCEPT
{
    return GetReleasedAudioInBuffersInternal(outAudioBuffer, count);
}

nn::Result AudioInManagerImpl::AudioInImpl::GetReleasedAudioInBuffersAuto(nn::sf::OutBuffer outAudioBuffer, nn::sf::Out<int> count) NN_NOEXCEPT
{
    return GetReleasedAudioInBuffersInternal(outAudioBuffer, count);
}

nn::Result AudioInManagerImpl::AudioInImpl::ContainsAudioInBuffer(uint64_t audioBufferPointer, nn::sf::Out<bool> contains) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(*m_pParentMutex);
    bool found = m_AudioInClass.ContainsBuffer(reinterpret_cast<void*>(audioBufferPointer));
    contains.Set(found);
    NN_RESULT_SUCCESS;
}

nn::Result AudioInManagerImpl::AudioInImpl::GetAudioInBufferCount(nn::sf::Out<int> outAudioInBufferCount)
{
    std::lock_guard<os::Mutex> lock(*m_pParentMutex);
    int count = m_AudioInClass.GetAppendedBufferCount();
    outAudioInBufferCount.Set(count);
    NN_RESULT_SUCCESS;
}

nn::Result AudioInManagerImpl::AudioInImpl::Initialize(nn::sf::InBuffer name,  nn::audio::AudioInParameter param, nn::sf::NativeHandle& processHandle, nn::sf::Out<nn::audio::detail::AudioInParameterInternal> outAudioInInter, nn::sf::OutBuffer nameOut, const nn::applet::AppletResourceUserId appletId) NN_NOEXCEPT
{
    const char* audioName = name.GetPointerUnsafe();
    std::lock_guard<os::Mutex> lock(*m_pParentMutex);
    SessionFormat format;
    format.sampleRate = param.sampleRate;
    format.channelCount = param._channelCount;
    format.format = SampleFormat_PcmInt16;
    NN_RESULT_DO(m_AudioInClass.Open(audioName, format, processHandle.GetOsHandle(), appletId));
    processHandle.Detach();
    outAudioInInter->sampleRate = m_AudioInClass.GetSampleRate();
    outAudioInInter->channelCount = m_AudioInClass.GetChannelCount();
    outAudioInInter->state = m_AudioInClass.GetState();
    outAudioInInter->sampleFormat = m_AudioInClass.GetSampleFormat();

    m_IsUacDevice = IsUacDeviceName(audioName);
    strcpy(nameOut.GetPointerUnsafe(), m_AudioInClass.GetName());
    NN_RESULT_SUCCESS;
}

bool AudioInManagerImpl::AudioInImpl::IsUacDevice() NN_NOEXCEPT
{
    return m_IsUacDevice;
}

nn::Result AudioInManagerImpl::AudioInImpl::SetDeviceGain(float gain) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(*m_pParentMutex);
    return m_AudioInClass.SetDeviceGain(gain);
}

nn::Result AudioInManagerImpl::AudioInImpl::GetDeviceGain(nn::sf::Out<float> gain) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(*m_pParentMutex);
    *gain = m_AudioInClass.GetDeviceGain();

    NN_RESULT_SUCCESS;
}

nn::Result AudioInManagerImpl::AudioInImpl::Update() NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(*m_pParentMutex);
    return m_AudioInClass.Update();
}

void AudioInManagerImpl::AudioInImpl::RollBackAudioInName(nn::sf::OutBuffer nameOut) NN_NOEXCEPT
{
    const char g_DeviceInName[] = "DeviceIn";
    const char g_UacInName[] = "UacIn";

    if (m_IsUacDevice)
    {
        strcpy(nameOut.GetPointerUnsafe(), g_UacInName);
    }
    else
    {
        strcpy(nameOut.GetPointerUnsafe(), g_DeviceInName);
    }

    NN_DETAIL_AUDIO_ERROR("nameOut: %s\n", nameOut.GetPointerUnsafe());
}

AudioInManagerImpl::AudioInManagerImpl() NN_NOEXCEPT
    : m_IsInitialized(false)
    , m_Mutex(true)
    , m_pSemaphore(nullptr)
    , m_IsThreadActive(false)
    , m_Thread()
    , m_IsAwake(true)
{
    m_HeapHandle = nn::lmem::CreateExpHeap( &m_HeapBuffer, sizeof(m_HeapBuffer), nn::lmem::CreationOption_ThreadSafe );
    m_Allocator.Attach(m_HeapHandle);

    m_pSemaphore = static_cast<os::SemaphoreType*>(m_Allocator.Allocate( sizeof( *m_pSemaphore) ));
    os::InitializeSemaphore( m_pSemaphore, 0, 1 );
    os::InitializeEvent(&m_EventSemaphoreSignal, false, nn::os::EventClearMode_AutoClear);
    for( int i = 0; i < NumberOfAudioInSessions; ++i)
    {
        m_Session[i] = nullptr;
        m_AppletId[i] = nn::applet::AppletResourceUserId::GetInvalidId();
    }

}

AudioInManagerImpl::~AudioInManagerImpl() NN_NOEXCEPT
{
    if( m_IsInitialized )
    {
        m_IsThreadActive = false;
        nn::os::WaitThread( &m_Thread );
        nn::os::DestroyThread( &m_Thread );

        m_Allocator.Deallocate(m_Stack, SessionStackSize);
        nn::audio::server::AudioInFinalize();
        m_IsInitialized = false;
    }
    os::FinalizeEvent(&m_EventSemaphoreSignal);
    os::FinalizeSemaphore(m_pSemaphore);
    m_Allocator.Deallocate( m_pSemaphore, sizeof( *m_pSemaphore) );
    nn::lmem::DestroyExpHeap(m_HeapHandle);
}

void AudioInManagerImpl::FirstTimeInitialize()
{
    if( !m_IsInitialized )
    {
        nn::audio::server::AudioInInitialize(m_pSemaphore, &m_EventSemaphoreSignal);
        m_Stack = static_cast<char*>(nn::lmem::AllocateFromExpHeap(m_HeapHandle, SessionStackSize, nn::os::ThreadStackAlignment));
        m_IsThreadActive = true;

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(&m_Thread, ThreadFunc, this, m_Stack, SessionStackSize, NN_SYSTEM_THREAD_PRIORITY(audio, AudioInManager)));
        nn::os::SetThreadNamePointer(&m_Thread, NN_SYSTEM_THREAD_NAME(audio, AudioInManager));
        nn::os::StartThread(&m_Thread);
        m_IsInitialized = true;
    }
}

nn::Result AudioInManagerImpl::OpenAudioInInternal(uint64_t protocolVersion, nn::sf::Out<nn::sf::SharedPointer<detail::IAudioIn>> outAudioIn, nn::sf::InBuffer name,  nn::audio::AudioInParameter param, nn::sf::NativeHandle& processHandle, nn::sf::Out<nn::audio::detail::AudioInParameterInternal> audioInInter, nn::sf::OutBuffer nameOut, nn::applet::AppletResourceUserId appletId) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(m_Mutex);
    FirstTimeInitialize();

    int session=-1;
    const char* audioName = name.GetPointerUnsafe();
    bool isUac = AudioInImpl::IsUacDeviceName(audioName);
    nn::Result res = m_Sessions.GetNextSessions(&session, isUac);

    if( res.IsFailure() )
    {
        return res;
    }

    typedef nn::sf::ObjectFactory<MyAllocator::Policy> Factory;

    auto p = Factory::CreateSharedEmplaced<detail::IAudioIn, AudioInManagerImpl::AudioInImpl>(&m_Allocator, this, static_cast<uint32_t>(session), &(m_Event[session]));

    os::InitializeEvent( &(m_Event[session]), false, nn::os::EventClearMode_AutoClear );
    m_Session[session] = Factory::GetEmplacedImplPointer<AudioInManagerImpl::AudioInImpl>(p);
    m_AppletId[session] = appletId;
    res = m_Session[session]->Initialize(name, param, processHandle, audioInInter, nameOut, appletId);

    // W/A for SIGLO-70562
    // ~NXAddon 4.x.x : "DeviceIn" or "UacIn"
    // NXAddon 5.0.0~ : "Uac_XXX" or "BuiltInHeadset"
    if (res.IsSuccess() && !(protocolVersion & common::AudioInProtocolFlags::GetAudioInNameFix))
    {
        m_Session[session]->RollBackAudioInName(nameOut);
    }

    *outAudioIn = std::move(p);

    NN_AUDIO_SERVICE_LOG("[AudioIn] Open sessionId(%d) appletResourceUserId.lower(%llu) name(%s) sampleRate(%d)\n",
            session, appletId.lower, audioName, param.sampleRate);

    return res;
}

nn::Result AudioInManagerImpl::OpenAudioInProtocolSpecified(uint64_t protocolVersion, nn::sf::Out<nn::sf::SharedPointer<detail::IAudioIn>> outAudioIn, nn::sf::InBuffer name, nn::audio::AudioInParameter param, nn::sf::NativeHandle processHandle, nn::sf::Out<nn::audio::detail::AudioInParameterInternal> audioInInter, nn::sf::OutBuffer nameOut, nn::applet::AppletResourceUserId appletId) NN_NOEXCEPT
{
    return OpenAudioInInternal(protocolVersion, outAudioIn, name, param, processHandle, audioInInter, nameOut, appletId);
}

// old shim
nn::Result AudioInManagerImpl::OpenAudioIn(nn::sf::Out<nn::sf::SharedPointer<detail::IAudioIn>> outAudioIn, nn::sf::InBuffer name,  nn::audio::AudioInParameter param, nn::sf::NativeHandle processHandle, nn::sf::Out<nn::audio::detail::AudioInParameterInternal> audioInInter, nn::sf::OutBuffer nameOut, nn::applet::AppletResourceUserId appletId) NN_NOEXCEPT
{
    return OpenAudioInInternal(0, outAudioIn, name, param, processHandle, audioInInter, nameOut, appletId);
}

// old shim
nn::Result AudioInManagerImpl::OpenAudioInAuto(nn::sf::Out<nn::sf::SharedPointer<detail::IAudioIn>> outAudioIn, nn::sf::InBuffer name,  nn::audio::AudioInParameter param, nn::sf::NativeHandle processHandle, nn::sf::Out<nn::audio::detail::AudioInParameterInternal> audioInInter, nn::sf::OutBuffer nameOut, nn::applet::AppletResourceUserId appletId) NN_NOEXCEPT
{
    return OpenAudioInInternal(0, outAudioIn, name, param, processHandle, audioInInter, nameOut, appletId);
}

nn::Result AudioInManagerImpl::ListAudioInsInternal(nn::sf::OutBuffer outAudioInfo, nn::sf::Out<int> amountRet, bool filtered) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(m_Mutex);
    FirstTimeInitialize();

    nn::audio::AudioInInfo* pInfo = reinterpret_cast<nn::audio::AudioInInfo*>(outAudioInfo.GetPointerUnsafe());
    size_t length = outAudioInfo.GetSize() / sizeof(*pInfo);

    int count = nn::audio::server::ListAudioIns(pInfo, static_cast<int>(length), filtered);

    for(int i = 0; i< count; ++i)
    {
        NN_AUDIO_SERVICE_LOG("[AudioIn] ListAudioIns(%d) name: %s\n", i, pInfo[i].name);
    }
    amountRet.Set( count );
    NN_RESULT_SUCCESS;
}

nn::Result AudioInManagerImpl::ListAudioIns(nn::sf::OutBuffer outAudioInfo, nn::sf::Out<int> amountRet) NN_NOEXCEPT
{
    return ListAudioInsInternal(outAudioInfo, amountRet, false);
}

nn::Result AudioInManagerImpl::ListAudioInsAuto(nn::sf::OutBuffer outAudioInfo, nn::sf::Out<int> amountRet) NN_NOEXCEPT
{
    return ListAudioInsInternal(outAudioInfo, amountRet, false);
}

nn::Result AudioInManagerImpl::ListAudioInsAutoFiltered(nn::sf::OutBuffer outAudioInfo, nn::sf::Out<int> amountRet) NN_NOEXCEPT
{
    return ListAudioInsInternal(outAudioInfo, amountRet, true);
}

void AudioInManagerImpl::ReportCloseSession(int sessionId) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(m_Mutex);
    m_Sessions.InsertSession(sessionId);
    os::FinalizeEvent(&(m_Event[sessionId]));
    m_Session[sessionId] = nullptr;
    m_AppletId[sessionId] = nn::applet::AppletResourceUserId::GetInvalidId();
}

nn::Result AudioInManagerImpl::Sleep() NN_NOEXCEPT
{
    if( m_IsAwake )
    {
        std::lock_guard<os::Mutex> lock(m_Mutex);
        for( int i = 0; i < NumberOfAudioInSessions; ++i )
        {
            if( m_Session[i] != nullptr )
            {
                m_Session[i]->Sleep();
            }
        }
        m_IsAwake = false;
        while(nn::os::GetCurrentSemaphoreCount(m_pSemaphore) > 0)
        {
            nn::os::TryAcquireSemaphore(m_pSemaphore);
        }
    }
    NN_RESULT_SUCCESS;
}

nn::Result AudioInManagerImpl::Wake() NN_NOEXCEPT
{
    if( !m_IsAwake )
    {
        std::lock_guard<os::Mutex> lock(m_Mutex);
        nn::audio::server::AudioInInitialize(m_pSemaphore, &m_EventSemaphoreSignal);
        for( int i = 0; i < NumberOfAudioInSessions; ++i )
        {
            if( m_Session[i] != nullptr )
            {
                m_Session[i]->Wake();
            }
        }
        m_IsAwake = true;
    }
    NN_RESULT_SUCCESS;
}

void AudioInManagerImpl::ThreadFunc(void* arg) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(arg);
    reinterpret_cast<AudioInManagerImpl*>(arg)->ThreadFuncImpl();
}

void AudioInManagerImpl::ThreadFuncImpl() NN_NOEXCEPT
{
    while( m_IsThreadActive )
    {
        bool semaphoreAcquired = nn::os::TimedAcquireSemaphore( m_pSemaphore, nn::TimeSpan::FromMilliSeconds(5));
        {
            std::lock_guard<os::Mutex> lock(m_Mutex);
            for( int i =0; i < NumberOfAudioInSessions; ++i )
            {
                if( m_Session[i] != nullptr )
                {
                    if( semaphoreAcquired || m_Session[i]->IsUacDevice() )
                    {
                        m_Session[i]->Update();
                    }
                }
            }
        }
    }
}

nn::os::Mutex* AudioInManagerImpl::GetMutex() NN_NOEXCEPT
{
    return &m_Mutex;
}

}}}
