﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/sf/sf_HipcServer.h>

namespace nn { namespace audio { namespace server {

template <size_t SessionCountMax, size_t PortCountMax, typename ManagerOption  = nn::sf::DefaultHipcSimpleAllInOneServerManagerOption>
class AudioHipcServerManager : public nn::sf::HipcSimpleAllInOneServerManager<SessionCountMax, PortCountMax, ManagerOption>
{
private:
    typedef nn::sf::HipcSimpleAllInOneServerManager<SessionCountMax, PortCountMax, ManagerOption> Base;
    nn::os::Event m_RequestSleepEvent;
    nn::os::Event m_RequestWakeEvent;
    nn::os::Event m_RequestDoneEvent;
    bool m_IsAwake;
public:
    AudioHipcServerManager() NN_NOEXCEPT
        : m_RequestSleepEvent(nn::os::EventClearMode_AutoClear),
          m_RequestWakeEvent(nn::os::EventClearMode_AutoClear),
          m_RequestDoneEvent(nn::os::EventClearMode_AutoClear)
    {

    }

    void Start() NN_NOEXCEPT
    {
        m_IsAwake = true;
        Base::Start();
    }

    void Sleep() NN_NOEXCEPT
    {
        if(m_IsAwake == false)
        {
            return;
        }

        m_RequestSleepEvent.Signal();
        m_RequestDoneEvent.Wait();
        m_IsAwake = false;
    }

    void Wake() NN_NOEXCEPT
    {
        if(m_IsAwake)
        {
            return;
        }

        m_RequestWakeEvent.Signal();
        m_RequestDoneEvent.Wait();
        m_IsAwake = true;
    }

    void LoopAuto() NN_NOEXCEPT
    {
        // AddUserWaitHolder の制約により AcceptTag と InvokeTag と異なる値を指定する必要がある
        const uintptr_t RequestSleepEventId = 100;
        NN_STATIC_ASSERT(RequestSleepEventId != AudioHipcServerManager::HipcSimpleAllInOneServerManager::AcceptTag);
        NN_STATIC_ASSERT(RequestSleepEventId != AudioHipcServerManager::HipcSimpleAllInOneServerManager::InvokeTag);

        // スリープ要求を受け取るイベントの準備
        nn::os::MultiWaitHolderType sleepRequestEventHolder = {};
        nn::os::InitializeMultiWaitHolder(&sleepRequestEventHolder,
                                            m_RequestSleepEvent.GetBase());
        nn::os::SetMultiWaitHolderUserData(&sleepRequestEventHolder, RequestSleepEventId);
        Base::AddUserWaitHolder(&sleepRequestEventHolder);

        for(;;)
        {
            auto waitId = Base::Wait();
            switch (::nn::os::GetMultiWaitHolderUserData(waitId))
            {
            case AudioHipcServerManager::HipcSimpleAllInOneServerManager::AcceptTag:
            case AudioHipcServerManager::HipcSimpleAllInOneServerManager::InvokeTag:
                Base::ProcessAuto(waitId);
                break;

            case RequestSleepEventId:
                m_RequestSleepEvent.Clear();
                Base::AddUserWaitHolder(&sleepRequestEventHolder);

                // スリープ要求が完了したことを通知
                m_RequestDoneEvent.Signal();

                // ウェイク要求を待機
                m_RequestWakeEvent.Wait();

                // ウェイク要求が完了したことを通知
                m_RequestDoneEvent.Signal();
                break;
            default:
                NN_UNEXPECTED_DEFAULT;
            }
        }
    }
};

}}}  // namespace nn::audio::server
