﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/audio/audio_Result.h>

#include <nn/audio/audio_Device.h>
#include <nn/audio/detail/audio_IAudioDevice.h>
#include "audio_AudioRendererManagerImpl.h"
#include "audio_AudioDeviceImpl.h"

namespace nn { namespace audio { namespace server {

AudioRendererManagerImpl::AudioDeviceImpl::AudioDeviceImpl(AudioRendererManagerImpl* pParent, nn::applet::AppletResourceUserId& id, common::RevisionInfo revisionInfo) NN_NOEXCEPT
    : m_Parent(pParent, true)
    , m_Id(id)
    , m_RevisionInfo(revisionInfo)
{
    detail::InitializeAudioDevice(m_Id);
}

AudioRendererManagerImpl::AudioDeviceImpl::~AudioDeviceImpl() NN_NOEXCEPT
{
    detail::FinalizeAudioDevice(m_Id);
}

nn::Result AudioRendererManagerImpl::AudioDeviceImpl::ListAudioDeviceNameInternal(const nn::sf::OutBuffer& outDeviceNames, nn::sf::Out<std::int32_t> amountRet) NN_NOEXCEPT
{
    auto pNames = reinterpret_cast<nn::audio::AudioDeviceName*>(outDeviceNames.GetPointerUnsafe());
    auto count = static_cast<int>(outDeviceNames.GetSize() / sizeof(*pNames));
    if (pNames == nullptr)
    {
        return ResultInsufficientBuffer();
    }
    *amountRet = detail::ListAudioDeviceName(pNames, count, m_RevisionInfo);

    NN_RESULT_SUCCESS;
}

nn::Result AudioRendererManagerImpl::AudioDeviceImpl::ListAudioDeviceName(const nn::sf::OutBuffer& outDeviceNames, nn::sf::Out<std::int32_t> amountRet) NN_NOEXCEPT
{
    return ListAudioDeviceNameInternal(outDeviceNames, amountRet);
}


nn::Result AudioRendererManagerImpl::AudioDeviceImpl::ListAudioDeviceNameAuto(const nn::sf::OutBuffer& outDeviceNames, nn::sf::Out<std::int32_t> amountRet) NN_NOEXCEPT
{
    return ListAudioDeviceNameInternal(outDeviceNames, amountRet);
}

nn::Result AudioRendererManagerImpl::AudioDeviceImpl::SetAudioDeviceOutputVolumeInternal(const nn::sf::InBuffer& nameIn, float volume) NN_NOEXCEPT
{
    NN_SDK_ASSERT_MINMAX(volume, nn::audio::AudioDeviceOutputVolumeMin, nn::audio::AudioDeviceOutputVolumeMax);
    auto clampedVolume = std::min(std::max(volume, nn::audio::AudioDeviceOutputVolumeMin), nn::audio::AudioDeviceOutputVolumeMax);
    auto pName = reinterpret_cast<const nn::audio::AudioDeviceName*>(nameIn.GetPointerUnsafe());
    if (pName == nullptr)
    {
        return ResultInsufficientBuffer();
    }

    detail::SetAudioDeviceOutputVolume(pName, clampedVolume, m_Id, m_RevisionInfo);
    NN_RESULT_SUCCESS;
}

nn::Result AudioRendererManagerImpl::AudioDeviceImpl::SetAudioDeviceOutputVolume(const nn::sf::InBuffer& nameIn, float volume) NN_NOEXCEPT
{
    return SetAudioDeviceOutputVolumeInternal(nameIn, volume);
}

nn::Result AudioRendererManagerImpl::AudioDeviceImpl::SetAudioDeviceOutputVolumeAuto(const nn::sf::InBuffer& nameIn, float volume) NN_NOEXCEPT
{
    return SetAudioDeviceOutputVolumeInternal(nameIn, volume);
}

nn::Result AudioRendererManagerImpl::AudioDeviceImpl::GetAudioDeviceOutputVolumeInternal(const nn::sf::InBuffer& nameIn, nn::sf::Out<float> pOutVolume) NN_NOEXCEPT
{
    auto pName = reinterpret_cast<const nn::audio::AudioDeviceName*>(nameIn.GetPointerUnsafe());
    if (pName == nullptr)
    {
        return ResultInsufficientBuffer();
    }

    *pOutVolume = detail::GetAudioDeviceOutputVolume(pName, m_Id, m_RevisionInfo);
    NN_RESULT_SUCCESS;
}

nn::Result AudioRendererManagerImpl::AudioDeviceImpl::GetAudioDeviceOutputVolume(const nn::sf::InBuffer& nameIn, nn::sf::Out<float> pOutVolume) NN_NOEXCEPT
{
    return GetAudioDeviceOutputVolumeInternal(nameIn, pOutVolume);
}

nn::Result AudioRendererManagerImpl::AudioDeviceImpl::GetAudioDeviceOutputVolumeAuto(const nn::sf::InBuffer& nameIn, nn::sf::Out<float> pOutVolume) NN_NOEXCEPT
{
    return GetAudioDeviceOutputVolumeInternal(nameIn, pOutVolume);
}

nn::Result AudioRendererManagerImpl::AudioDeviceImpl::GetActiveAudioDeviceNameInternal(const nn::sf::OutBuffer& outDeviceName) NN_NOEXCEPT
{
    auto pName = reinterpret_cast<nn::audio::AudioDeviceName*>(outDeviceName.GetPointerUnsafe());
    if (pName == nullptr)
    {
        return ResultInsufficientBuffer();
    }

    detail::GetActiveAudioDeviceName(pName, m_RevisionInfo);
    NN_RESULT_SUCCESS;
}

nn::Result AudioRendererManagerImpl::AudioDeviceImpl::GetActiveAudioDeviceName(const nn::sf::OutBuffer& outDeviceName) NN_NOEXCEPT
{
    return GetActiveAudioDeviceNameInternal(outDeviceName);
}

nn::Result AudioRendererManagerImpl::AudioDeviceImpl::GetActiveAudioDeviceNameAuto(const nn::sf::OutBuffer& outDeviceName) NN_NOEXCEPT
{
    return GetActiveAudioDeviceNameInternal(outDeviceName);
}

nn::Result AudioRendererManagerImpl::AudioDeviceImpl::QueryAudioDeviceSystemEvent(nn::sf::Out<nn::sf::NativeHandle> pOutHandle) NN_NOEXCEPT
{
    nn::os::NativeHandle handle;
    NN_RESULT_DO(detail::QueryAudioDeviceSystemEvent(&handle, m_Id));
    pOutHandle.Set(nn::sf::NativeHandle(handle, false));
    NN_RESULT_SUCCESS;
}

nn::Result AudioRendererManagerImpl::AudioDeviceImpl::QueryAudioDeviceInputEvent(nn::sf::Out<nn::sf::NativeHandle> pOutHandle) NN_NOEXCEPT
{
    nn::os::NativeHandle handle;
    NN_RESULT_DO(detail::QueryAudioDeviceInputNotificationEvent(&handle, m_Id));
    pOutHandle.Set(nn::sf::NativeHandle(handle, false));
    NN_RESULT_SUCCESS;
}

nn::Result AudioRendererManagerImpl::AudioDeviceImpl::QueryAudioDeviceOutputEvent(nn::sf::Out<nn::sf::NativeHandle> pOutHandle) NN_NOEXCEPT
{
    nn::os::NativeHandle handle;
    NN_RESULT_DO(detail::QueryAudioDeviceOutputNotificationEvent(&handle, m_Id));
    pOutHandle.Set(nn::sf::NativeHandle(handle, false));
    NN_RESULT_SUCCESS;
}

nn::Result AudioRendererManagerImpl::AudioDeviceImpl::GetActiveChannelCount(nn::sf::Out<std::int32_t> channelCount) NN_NOEXCEPT
{
    *channelCount = detail::GetActiveChannelCount();
    NN_RESULT_SUCCESS;
}

nn::Result AudioRendererManagerImpl::AudioDeviceImpl::GetAudioSystemMasterVolumeSetting(nn::sf::Out<float> volume, const nn::sf::InBuffer& name) NN_NOEXCEPT
{
    auto pName = reinterpret_cast<const nn::audio::AudioDeviceName*>(name.GetPointerUnsafe());
    *volume = detail::GetAudioSystemMasterVolumeSetting(pName);
    NN_RESULT_SUCCESS;
}

}}}  // namespace nn::audio::server
