﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstdint>
#include <nn/nn_Macro.h>

namespace nn { namespace audio { namespace dsp { namespace detail {

NN_FORCEINLINE void ApplyUniformGain2(int32_t* output, const int32_t* input, int32_t gain, int sampleCount) NN_NOEXCEPT
{
    const int Q = 15;
    for (auto i = 0; i < sampleCount; i += 2)
    {
        output[i + 0] = (static_cast<int64_t>(input[i + 0]) * gain + (1 << (Q - 1))) >> Q;
        output[i + 1] = (static_cast<int64_t>(input[i + 1]) * gain + (1 << (Q - 1))) >> Q;
    }
}

NN_FORCEINLINE void ApplyUniformGain4(int32_t* output, const int32_t* input, int32_t gain, int sampleCount) NN_NOEXCEPT
{
    const int Q = 15;
    for (auto i = 0; i < sampleCount; i += 4)
    {
        output[i + 0] = (static_cast<int64_t>(input[i + 0]) * gain + (1 << (Q - 1))) >> Q;
        output[i + 1] = (static_cast<int64_t>(input[i + 1]) * gain + (1 << (Q - 1))) >> Q;
        output[i + 2] = (static_cast<int64_t>(input[i + 2]) * gain + (1 << (Q - 1))) >> Q;
        output[i + 3] = (static_cast<int64_t>(input[i + 3]) * gain + (1 << (Q - 1))) >> Q;
    }
}

NN_FORCEINLINE void ApplyUniformGain4(float* output, const float* input, float gain, int sampleCount) NN_NOEXCEPT
{
    for (auto i = 0; i < sampleCount; i += 4)
    {
        output[i + 0] = input[i + 0] * gain;
        output[i + 1] = input[i + 1] * gain;
        output[i + 2] = input[i + 2] * gain;
        output[i + 3] = input[i + 3] * gain;
    }
}

NN_FORCEINLINE void ApplyLinearEnvelopeGain2(int32_t* output, const int32_t* input, int32_t gain, int32_t delta, int sampleCount) NN_NOEXCEPT
{
    const int Q = 15;
    auto gain0 = gain;
    auto gain1 = gain + delta;
    for (auto i = 0; i < sampleCount; i += 2)
    {
        output[i + 0] = (static_cast<int64_t>(input[i + 0]) * gain0 + (1 << (Q - 1))) >> Q;
        output[i + 1] = (static_cast<int64_t>(input[i + 1]) * gain1 + (1 << (Q - 1))) >> Q;
        gain0 += delta * 2;
        gain1 += delta * 2;
    }
}

NN_FORCEINLINE void ApplyLinearEnvelopeGain4(int32_t* output, const int32_t* input, int32_t gain, int32_t delta, int sampleCount) NN_NOEXCEPT
{
    const int Q = 15;
    auto gain0 = gain + delta * 0;
    auto gain1 = gain + delta * 1;
    auto gain2 = gain + delta * 2;
    auto gain3 = gain + delta * 3;
    for (auto i = 0; i < sampleCount; i += 4)
    {
        output[i + 0] = (static_cast<int64_t>(input[i + 0]) * gain0 + (1 << (Q - 1))) >> Q;
        output[i + 1] = (static_cast<int64_t>(input[i + 1]) * gain1 + (1 << (Q - 1))) >> Q;
        output[i + 2] = (static_cast<int64_t>(input[i + 2]) * gain2 + (1 << (Q - 1))) >> Q;
        output[i + 3] = (static_cast<int64_t>(input[i + 3]) * gain3 + (1 << (Q - 1))) >> Q;
        gain0 += delta * 4;
        gain1 += delta * 4;
        gain2 += delta * 4;
        gain3 += delta * 4;
    }
}

NN_FORCEINLINE void ApplyLinearEnvelopeGain4(float* output, const float* input, float gain, float delta, int sampleCount) NN_NOEXCEPT
{
    auto gain0 = gain + delta * 0.0f;
    auto gain1 = gain + delta * 1.0f;
    auto gain2 = gain + delta * 2.0f;
    auto gain3 = gain + delta * 3.0f;
    for (auto i = 0; i < sampleCount; i += 4)
    {
        output[i + 0] = input[i + 0] * gain0;
        output[i + 1] = input[i + 1] * gain1;
        output[i + 2] = input[i + 2] * gain2;
        output[i + 3] = input[i + 3] * gain3;
        gain0 += delta * 4.0f;
        gain1 += delta * 4.0f;
        gain2 += delta * 4.0f;
        gain3 += delta * 4.0f;
    }
}

}}}}  // namespace nn::audio::dsp::detail
