﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "../audio_EffectI3dl2Reverb.h"
#include "../audio_Qf.h"
#include "../audio_Axfx2DelayLine.h"
#include "../audio_DspUtility.h"

namespace nn { namespace audio { namespace detail {

namespace {
enum NUM_CHANS { NUM_1CHANS = 1, NUM_2CHANS = 2, NUM_4CHANS = 4, NUM_6CHANS = 6 };
enum Channel_Order { ch_L = 0, ch_R = 1, ch_LS = 2, ch_RS = 3, ch_C = 4, ch_LFE = 5 };

const int32_t ErBuses4Channel[I3dl2ReverbParameter::EarlyReflectionTaps] =
{
    ch_L, ch_L, ch_L, ch_R, ch_R, ch_R, ch_R, ch_LS, ch_LS, ch_LS,
    ch_R, ch_R, ch_R, ch_L, ch_L, ch_L, ch_L, ch_RS, ch_RS, ch_RS
};
const int32_t ErBuses6Channel[I3dl2ReverbParameter::EarlyReflectionTaps] =
{
    ch_C, ch_L, ch_L, ch_R, ch_R, ch_R, ch_R, ch_LS, ch_LS, ch_LS,
    ch_R, ch_R, ch_R, ch_L, ch_L, ch_L, ch_L, ch_RS, ch_RS, ch_RS
};
} // namespace

NN_FORCEINLINE void ApplyI3dl2ReverbEffectBypass(const int32_t **ppInData, int32_t **ppOutData, int channelCount, int sampleCount)
{
    for(int i = 0; i < channelCount; ++i)
    {
        if (ppOutData[i] != ppInData[i])
        {
            memcpy(ppOutData[i], ppInData[i], sampleCount* sizeof(int32_t));
        }
    }
}

const f32 ErGainsI3dl2[I3dl2ReverbParameter::EarlyReflectionTaps] =
{ 0.670958f, 0.610273f, 1.000000f, 0.356801f, 0.683607f, 0.659781f, 0.519391f, 0.247120f, 0.459452f, 0.450214f, 0.641963f, 0.548790f, 0.929252f, 0.382704f, 0.728672f, 0.697942f, 0.546396f, 0.245628f, 0.452136f, 0.440424f };

inline f32 Int32ToF32(int32_t value)
{
    return static_cast<f32>(value) / 65536.f;
}

inline int32_t F32ToInt32(f32 value)
{
    const f32 int32Tmin = -8388608.f; // 24bits
    const f32 int32Tmax =  8388607.f;
    f32 temp = value * 65536.f;

    if (temp < int32Tmin)
    {
        temp = int32Tmin;  //-(2^31)
    }

    if (temp > int32Tmax)
    {
        temp = int32Tmax;  //(2^31) -1
    }

    return static_cast<int32_t>(temp);
}

NN_FORCEINLINE void ApplyI3dl2ReverbEffect1ch(I3dl2ReverbState* state, const int32_t** ppInData, int32_t** ppOutData, int sampleCount)
{
    int samp;
    uint32_t i;
    uint32_t tap;
    f32 temp_samp;
    f32 out_ptr[NUM_1CHANS]; // 2
    f32 tempin;
    f32 temps[I3dl2ReverbParameter::FeedbackDelayNetworkCount];
    f32 temps2[I3dl2ReverbParameter::FeedbackDelayNetworkCount];
    f32 filt_in;
    f32 filt_out;
    f32 late_in_i;
    f32 fdn_in;
    f32 fdn_out[I3dl2ReverbParameter::FeedbackDelayNetworkCount];
    f32 dryGain = state->_dryGain;
    const int32_t* Mono = ppInData[0];
    int32_t* MonoOut = ppOutData[0];

    Axfx2DelayLine *earlyDelayLine = &(state->_earlyDelayLine);
    uint32_t erLateTap = state->_erLateTap;

    for (samp = 0; (samp < sampleCount); ++samp)
    {
        // zero the output
        out_ptr[ch_L] = 0.0;

        // mix LFE into all inputs
        tempin = Int32ToF32(Mono[samp]);

        // get last er delay output for input to late reverb, mixing center into left and right
        fdn_in = Axfx2DelayTapOut(earlyDelayLine, erLateTap);

        // early reflections
        // get early reflections from delay line and add to output bus:
        for (tap = 0; (tap < I3dl2ReverbParameter::EarlyReflectionTaps); ++tap) // 20
        {
            temp_samp = Axfx2DelayTapOut(earlyDelayLine, state->_erTapTimes[tap]) * ErGainsI3dl2[tap];
            out_ptr[ch_L] += temp_samp;
        }

        // apply early reflections level
        for (i = 0; i < NUM_1CHANS; i++)
        {
            out_ptr[i] *= state->_earlyLevel;
        }

        // low-pass filter inputs to er delay lines and tick er delays:
        state->_lpfZ = tempin * state->_lpfB0 + state->_lpfZ * state->_lpfA1; // LPF
        Axfx2DelayTick(earlyDelayLine, state->_lpfZ);                              // tick delay

        for (i = 0; i < I3dl2ReverbParameter::FeedbackDelayNetworkCount; i++)
        {
            // perform attenuation and high freq filtering with 1st-order shelf filter (transposed direct form II)
            filt_in = Axfx2DelayNextOut(&state->_fdnDelayLines[i]);
            filt_out = filt_in * state->_fdnFilterCoef[0][i] + state->_fdnFilterZ[i];
            state->_fdnFilterZ[i] = filt_in * state->_fdnFilterCoef[1][i] + filt_out * state->_fdnFilterCoef[2][i];
            temps[i] = filt_out;
        }

        // Jot/Stautner/Puckette mixing matrix
        temps2[0] = temps[1] + temps[2];
        temps2[1] = -temps[0] - temps[3];
        temps2[2] = temps[0] - temps[3];
        temps2[3] = temps[1] - temps[2];

        // add inputs, then all-pass filter, then feedback to delays
        for (i = 0; i < I3dl2ReverbParameter::FeedbackDelayNetworkCount; ++i)
        {
            late_in_i = (fdn_in * state->_lateLevel);
            fdn_out[i] = Axfx2AllPassTick(&state->_apDelayLine[i], (late_in_i + temps2[i]));
            fdn_out[i] = Axfx2AllPassTick(&state->_ap2DelayLine[i], fdn_out[i]);
            Axfx2DelayTick(&state->_fdnDelayLines[i], fdn_out[i]);
        }

        // mix in output from late reverb
        MonoOut[samp] = F32ToInt32(dryGain * Int32ToF32(Mono[samp]) + (out_ptr[ch_L] + fdn_out[0] + fdn_out[1])); // Left
    }
}

NN_FORCEINLINE void ApplyI3dl2ReverbEffect2ch(I3dl2ReverbState* state, const int32_t** ppInData, int32_t** ppOutData, int sampleCount)
{
    int samp;
    uint32_t i;
    f32 out_ptr[NUM_2CHANS]; // 2
    f32 tempin;
    f32 temps[I3dl2ReverbParameter::FeedbackDelayNetworkCount];
    f32 temps2[I3dl2ReverbParameter::FeedbackDelayNetworkCount];
    f32 filt_in;
    f32 filt_out;
    f32 fdn_in;
    f32 fdn_out[NUM_2CHANS];
    f32 dryGain = state->_dryGain;
    const int32_t* L = ppInData[0];
    const int32_t* R = ppInData[1];
    int32_t* Lout = ppOutData[0];
    int32_t* Rout = ppOutData[1];

    Axfx2DelayLine *earlyDelayLine = &(state->_earlyDelayLine);
    const uint32_t ErLateTap = state->_erLateTap + 1;
    const uint32_t MaxDelayPlusOne = earlyDelayLine->maxDelay + 1;

    const f32 *CurBasePtr = earlyDelayLine->buff;
    const uint32_t ErTapTimes[] =
    {
        state->_erTapTimes[0] + 1, state->_erTapTimes[1] + 1, state->_erTapTimes[2] + 1, state->_erTapTimes[3] + 1,
        state->_erTapTimes[4] + 1, state->_erTapTimes[5] + 1, state->_erTapTimes[6] + 1, state->_erTapTimes[7] + 1,
        state->_erTapTimes[8] + 1, state->_erTapTimes[9] + 1, state->_erTapTimes[10] + 1, state->_erTapTimes[11] + 1,
        state->_erTapTimes[12] + 1, state->_erTapTimes[13] + 1, state->_erTapTimes[14] + 1, state->_erTapTimes[15] + 1,
        state->_erTapTimes[16] + 1, state->_erTapTimes[17] + 1, state->_erTapTimes[18] + 1, state->_erTapTimes[19] + 1,
    };

    for (samp = 0; (samp < sampleCount); ++samp)
    {
        // zero the output
        out_ptr[ch_L] = out_ptr[ch_R] = 0.0;

        // mix LFE into all inputs
        tempin = Int32ToF32(L[samp] + R[samp]);

        const f32 *CurInPtr = earlyDelayLine->in;

        // get last er delay output for input to late reverb, mixing center into left and right
        fdn_in = Axfx2DelayTapOut(CurInPtr, CurBasePtr, ErLateTap, MaxDelayPlusOne);

        // early reflections
        // get early reflections from delay line and add to output bus:

        // Unrolled by I3dl2ReverbParameter::EarlyReflectionTaps times (20 times)
        // This eliminates the left/right channel array lookup and allows the
        // compiler to better mix all the cpu's load, integer, and float instruction pipelines.
        {
            uint32_t tap = 0;
            out_ptr[ch_L] += Axfx2DelayTapOut(CurInPtr, CurBasePtr, ErTapTimes[tap], MaxDelayPlusOne) * ErGainsI3dl2[tap]; tap++;
            out_ptr[ch_L] += Axfx2DelayTapOut(CurInPtr, CurBasePtr, ErTapTimes[tap], MaxDelayPlusOne) * ErGainsI3dl2[tap]; tap++;
            out_ptr[ch_L] += Axfx2DelayTapOut(CurInPtr, CurBasePtr, ErTapTimes[tap], MaxDelayPlusOne) * ErGainsI3dl2[tap]; tap++;

            out_ptr[ch_R] += Axfx2DelayTapOut(CurInPtr, CurBasePtr, ErTapTimes[tap], MaxDelayPlusOne) * ErGainsI3dl2[tap]; tap++;
            out_ptr[ch_R] += Axfx2DelayTapOut(CurInPtr, CurBasePtr, ErTapTimes[tap], MaxDelayPlusOne) * ErGainsI3dl2[tap]; tap++;
            out_ptr[ch_R] += Axfx2DelayTapOut(CurInPtr, CurBasePtr, ErTapTimes[tap], MaxDelayPlusOne) * ErGainsI3dl2[tap]; tap++;
            out_ptr[ch_R] += Axfx2DelayTapOut(CurInPtr, CurBasePtr, ErTapTimes[tap], MaxDelayPlusOne) * ErGainsI3dl2[tap]; tap++;

            out_ptr[ch_L] += Axfx2DelayTapOut(CurInPtr, CurBasePtr, ErTapTimes[tap], MaxDelayPlusOne) * ErGainsI3dl2[tap]; tap++;
            out_ptr[ch_L] += Axfx2DelayTapOut(CurInPtr, CurBasePtr, ErTapTimes[tap], MaxDelayPlusOne) * ErGainsI3dl2[tap]; tap++;
            out_ptr[ch_L] += Axfx2DelayTapOut(CurInPtr, CurBasePtr, ErTapTimes[tap], MaxDelayPlusOne) * ErGainsI3dl2[tap]; tap++;

            out_ptr[ch_R] += Axfx2DelayTapOut(CurInPtr, CurBasePtr, ErTapTimes[tap], MaxDelayPlusOne) * ErGainsI3dl2[tap]; tap++;
            out_ptr[ch_R] += Axfx2DelayTapOut(CurInPtr, CurBasePtr, ErTapTimes[tap], MaxDelayPlusOne) * ErGainsI3dl2[tap]; tap++;
            out_ptr[ch_R] += Axfx2DelayTapOut(CurInPtr, CurBasePtr, ErTapTimes[tap], MaxDelayPlusOne) * ErGainsI3dl2[tap]; tap++;

            out_ptr[ch_L] += Axfx2DelayTapOut(CurInPtr, CurBasePtr, ErTapTimes[tap], MaxDelayPlusOne) * ErGainsI3dl2[tap]; tap++;
            out_ptr[ch_L] += Axfx2DelayTapOut(CurInPtr, CurBasePtr, ErTapTimes[tap], MaxDelayPlusOne) * ErGainsI3dl2[tap]; tap++;
            out_ptr[ch_L] += Axfx2DelayTapOut(CurInPtr, CurBasePtr, ErTapTimes[tap], MaxDelayPlusOne) * ErGainsI3dl2[tap]; tap++;
            out_ptr[ch_L] += Axfx2DelayTapOut(CurInPtr, CurBasePtr, ErTapTimes[tap], MaxDelayPlusOne) * ErGainsI3dl2[tap]; tap++;

            out_ptr[ch_R] += Axfx2DelayTapOut(CurInPtr, CurBasePtr, ErTapTimes[tap], MaxDelayPlusOne) * ErGainsI3dl2[tap]; tap++;
            out_ptr[ch_R] += Axfx2DelayTapOut(CurInPtr, CurBasePtr, ErTapTimes[tap], MaxDelayPlusOne) * ErGainsI3dl2[tap]; tap++;
            out_ptr[ch_R] += Axfx2DelayTapOut(CurInPtr, CurBasePtr, ErTapTimes[tap], MaxDelayPlusOne) * ErGainsI3dl2[tap]; tap++;
        }

        // apply early reflections level

        for (i = 0; i < NUM_2CHANS; i++)
        {
            out_ptr[i] *= state->_earlyLevel;
        }

        // low-pass filter inputs to er delay lines and tick er delays:
        state->_lpfZ = tempin * state->_lpfB0 + state->_lpfZ * state->_lpfA1; // LPF
        Axfx2DelayTick(earlyDelayLine, state->_lpfZ);                              // tick delay

        for (i = 0; i < I3dl2ReverbParameter::FeedbackDelayNetworkCount; i++)
        {
            // perform attenuation and high freq filtering with 1st-order shelf filter (transposed direct form II)
            filt_in = Axfx2DelayNextOut(&state->_fdnDelayLines[i]);
            filt_out = filt_in * state->_fdnFilterCoef[0][i] + state->_fdnFilterZ[i];
            state->_fdnFilterZ[i] = filt_in * state->_fdnFilterCoef[1][i] + filt_out * state->_fdnFilterCoef[2][i];
            temps[i] = filt_out;
        }

        const f32 late_in_i = (fdn_in * state->_lateLevel);

        // Jot/Stautner/Puckette mixing matrix plus late in.
        temps2[0] = temps[1] + temps[2] + late_in_i;
        temps2[1] = -temps[0] - temps[3] + late_in_i;
        temps2[2] = temps[0] - temps[3] + late_in_i;
        temps2[3] = temps[1] - temps[2] + late_in_i;

        // add inputs, then all-pass filter, then feedback to delays
//        for (i = 0; i < I3dl2ReverbParameter::FeedbackDelayNetworkCount; ++i)
        {
            fdn_out[0] = Axfx2AllPassTick(&state->_apDelayLine[0], &state->_ap2DelayLine[0], &state->_fdnDelayLines[0], temps2[0]);
            fdn_out[1] = Axfx2AllPassTick(&state->_apDelayLine[1], &state->_ap2DelayLine[1], &state->_fdnDelayLines[1], temps2[1]);

            // Second set specifically doesn't save output for later.  Doing this saves a few instructions
            Axfx2AllPassTick(&state->_apDelayLine[2], &state->_ap2DelayLine[2], &state->_fdnDelayLines[2], temps2[2]);
            Axfx2AllPassTick(&state->_apDelayLine[3], &state->_ap2DelayLine[3], &state->_fdnDelayLines[3], temps2[3]);
        }

        // mix in output from late reverb
        Lout[samp] = F32ToInt32(dryGain * Int32ToF32(L[samp]) + (out_ptr[ch_L] + fdn_out[0])); // Left
        Rout[samp] = F32ToInt32(dryGain * Int32ToF32(R[samp]) + (out_ptr[ch_R] + fdn_out[1])); // Right
    }
} // NOLINT(readability/fn_size)

NN_FORCEINLINE void ApplyI3dl2ReverbEffect4ch(I3dl2ReverbState* state, const int32_t** ppInData, int32_t** ppOutData, int sampleCount)
{
    int samp;
    uint32_t i;
    uint32_t tap;
    f32 temp_samp;
    f32 out_ptr[NUM_4CHANS];
    f32 tempin;
    f32 temps[I3dl2ReverbParameter::FeedbackDelayNetworkCount];
    f32 temps2[I3dl2ReverbParameter::FeedbackDelayNetworkCount];
    f32 filt_in;
    f32 filt_out;
    f32 late_in_i;
    f32 fdn_in;
    f32 fdn_out[I3dl2ReverbParameter::FeedbackDelayNetworkCount];
    f32 dryGain = state->_dryGain;
    const int32_t* L = ppInData[0];
    const int32_t* R = ppInData[1];
    const int32_t* Ls = ppInData[2];
    const int32_t* Rs = ppInData[3];
    int32_t* Lout = ppOutData[0];
    int32_t* Rout = ppOutData[1];
    int32_t* Lsout = ppOutData[2];
    int32_t* Rsout = ppOutData[3];

    Axfx2DelayLine *earlyDelayLine = &(state->_earlyDelayLine);
    uint32_t erLateTap = state->_erLateTap;

    for (samp = 0; (samp < sampleCount); ++samp)
    {
        // zero the output
        out_ptr[ch_L] = out_ptr[ch_R] = out_ptr[ch_LS] = out_ptr[ch_RS] = 0.0;
        // mix LFE into all inputs
        tempin = Int32ToF32(L[samp] + R[samp] + Ls[samp] + Rs[samp]);

        // get last er delay output for input to late reverb, mixing center into left and right
        fdn_in = Axfx2DelayTapOut(earlyDelayLine, erLateTap);

        // early reflections
        // get early reflections from delay line and add to output bus:
        for (tap = 0; (tap < I3dl2ReverbParameter::EarlyReflectionTaps); ++tap) // 20
        {
            temp_samp = Axfx2DelayTapOut(earlyDelayLine, state->_erTapTimes[tap]) * ErGainsI3dl2[tap];
            out_ptr[ErBuses4Channel[tap]] += temp_samp;
        }

        // low-pass filter inputs to er delay lines and tick er delays:
        state->_lpfZ = tempin * state->_lpfB0 + state->_lpfZ * state->_lpfA1; // LPF
        Axfx2DelayTick(earlyDelayLine, state->_lpfZ);                              // tick delay

        // apply early reflections level
        for (i = 0; i < NUM_4CHANS; i++)
        {
            out_ptr[i] *= state->_earlyLevel;
        }

        // get last samples out of delay lines, multiply by attenuation coef, and low-pass filter
        for (i = 0; i < I3dl2ReverbParameter::FeedbackDelayNetworkCount; i++)
        {
            // perform attenuation and high freq filtering with 1st-order shelf filter (transposed direct form II)
            filt_in = Axfx2DelayNextOut(&state->_fdnDelayLines[i]);
            filt_out = filt_in * state->_fdnFilterCoef[0][i] + state->_fdnFilterZ[i];
            state->_fdnFilterZ[i] = filt_in * state->_fdnFilterCoef[1][i] + filt_out * state->_fdnFilterCoef[2][i];
            temps[i] = filt_out;
        }

        // Jot/Stautner/Puckette mixing matrix
        temps2[0] = temps[1] + temps[2];
        temps2[1] = -temps[0] - temps[3];
        temps2[2] = temps[0] - temps[3];
        temps2[3] = temps[1] - temps[2];

        // add inputs, then all-pass filter, then feedback to delays
        for (i = 0; i < I3dl2ReverbParameter::FeedbackDelayNetworkCount; ++i)
        {
            late_in_i = (fdn_in * state->_lateLevel);
            fdn_out[i] = Axfx2AllPassTick(&state->_apDelayLine[i], (late_in_i + temps2[i]));
            fdn_out[i] = Axfx2AllPassTick(&state->_ap2DelayLine[i], fdn_out[i]);
            Axfx2DelayTick(&state->_fdnDelayLines[i], fdn_out[i]);
        }

        // mix in output from late reverb
        Lout[samp] = F32ToInt32(dryGain * Int32ToF32(L[samp]) + (out_ptr[ch_L] + fdn_out[0])); // Left
        Rout[samp] = F32ToInt32(dryGain * Int32ToF32(R[samp]) + (out_ptr[ch_R] + fdn_out[1])); // Right
        Lsout[samp] = F32ToInt32(dryGain * Int32ToF32(Ls[samp]) + (out_ptr[ch_LS] + fdn_out[2])); // Left Surround
        Rsout[samp] = F32ToInt32(dryGain * Int32ToF32(Rs[samp]) + (out_ptr[ch_RS] + fdn_out[3])); // Right Surround
    }
}

NN_FORCEINLINE void ApplyI3dl2ReverbEffect6ch(I3dl2ReverbState* state, const int32_t** ppInData, int32_t** ppOutData, int sampleCount)
{
    int samp;
    uint32_t i;
    uint32_t tap;
    f32 temp_samp;
    f32 out_ptr[NUM_6CHANS]; // 6
    f32 tempin;
    f32 temps[I3dl2ReverbParameter::FeedbackDelayNetworkCount];
    f32 temps2[I3dl2ReverbParameter::FeedbackDelayNetworkCount];
    f32 filt_in;
    f32 filt_out;
    f32 late_in_i;
    f32 tempc;
    f32 fdn_in;
    f32 fdn_out[I3dl2ReverbParameter::FeedbackDelayNetworkCount];
    f32 dryGain = state->_dryGain;
    const int32_t* L = ppInData[0];
    const int32_t* R = ppInData[1];
    const int32_t* Ls = ppInData[2];
    const int32_t* Rs = ppInData[3];
    const int32_t* Ct = ppInData[4];
    const int32_t* Sw = ppInData[5];
    int32_t* Lout = ppOutData[0];
    int32_t* Rout = ppOutData[1];
    int32_t* Lsout = ppOutData[2];
    int32_t* Rsout = ppOutData[3];
    int32_t* Ctout = ppOutData[4];
    int32_t* Swout = ppOutData[5];

    Axfx2DelayLine *earlyDelayLine = &(state->_earlyDelayLine);
    uint32_t  erLateTap = state->_erLateTap;

    for (samp = 0; (samp < sampleCount); ++samp)
    {
        // zero the output
        out_ptr[ch_L] = out_ptr[ch_R] = out_ptr[ch_C] = out_ptr[ch_LS] = out_ptr[ch_RS] = out_ptr[ch_LFE] = 0.0;
        // mix LFE into all inputs
        tempin = Int32ToF32(L[samp] + R[samp] + Ct[samp] + Ls[samp] + Rs[samp] + Sw[samp]);

        // get last er delay output for input to late reverb, mixing center into left and right
        fdn_in = Axfx2DelayTapOut(earlyDelayLine, erLateTap);

        // early reflections
        // get early reflections from delay line and add to output bus:
        for (tap = 0; (tap < I3dl2ReverbParameter::EarlyReflectionTaps); ++tap) // 20
        {
            temp_samp = Axfx2DelayTapOut(earlyDelayLine, state->_erTapTimes[tap]) * ErGainsI3dl2[tap];
            out_ptr[ErBuses6Channel[tap]] += temp_samp;
            out_ptr[ch_LFE] += temp_samp;
        }

        // apply early reflections level
        for (i = 0; i < NUM_6CHANS; i++)
        {
            out_ptr[i] *= state->_earlyLevel;
        }

        // low-pass filter inputs to er delay lines and tick er delays:
        state->_lpfZ = tempin * state->_lpfB0 + state->_lpfZ * state->_lpfA1; // LPF
        Axfx2DelayTick(earlyDelayLine, state->_lpfZ);                              // tick delay

        // get last samples out of delay lines, multiply by attenuation coef, and low-pass filter
        for (i = 0; i < I3dl2ReverbParameter::FeedbackDelayNetworkCount; i++)
        {
            //perform attenuation and high freq filtering with 1st-order shelf filter (transposed direct form II)
            filt_in = Axfx2DelayNextOut(&state->_fdnDelayLines[i]);
            filt_out = filt_in * state->_fdnFilterCoef[0][i] + state->_fdnFilterZ[i];
            state->_fdnFilterZ[i] = filt_in * state->_fdnFilterCoef[1][i] + filt_out * state->_fdnFilterCoef[2][i];
            temps[i] = filt_out;
        }

        // Jot/Stautner/Puckette mixing matrix
        temps2[0] = temps[1] + temps[2];
        temps2[1] = -temps[0] - temps[3];
        temps2[2] = temps[0] - temps[3];
        temps2[3] = temps[1] - temps[2];

        // add inputs, then all-pass filter, then feedback to delays
        for (i = 0; i < I3dl2ReverbParameter::FeedbackDelayNetworkCount; ++i)
        {
            late_in_i = (fdn_in * state->_lateLevel);
            fdn_out[i] = Axfx2AllPassTick(&state->_apDelayLine[i], (late_in_i + temps2[i]));
            fdn_out[i] = Axfx2AllPassTick(&state->_ap2DelayLine[i], fdn_out[i]);
            Axfx2DelayTick(&state->_fdnDelayLines[i], fdn_out[i]);
        }

        // sum and delay rears to get center
        tempc = Axfx2DelayTick(&state->_centerDelayLine, 0.5f * (fdn_out[2] - fdn_out[3]));

        // mix in output from late reverb
        Lout[samp] = F32ToInt32(dryGain * Int32ToF32(L[samp]) + (out_ptr[ch_L] + fdn_out[0])); // Left
        Rout[samp] = F32ToInt32(dryGain * Int32ToF32(R[samp]) + (out_ptr[ch_R] + fdn_out[1])); // Right
        Lsout[samp] = F32ToInt32(dryGain * Int32ToF32(Ls[samp]) + (out_ptr[ch_LS] + fdn_out[2])); // Left Surround
        Rsout[samp] = F32ToInt32(dryGain * Int32ToF32(Rs[samp]) + (out_ptr[ch_RS] + fdn_out[3])); // Right Surround
        Ctout[samp] = F32ToInt32(dryGain * Int32ToF32(Ct[samp]) + (out_ptr[ch_C] + tempc)); // Center
        Swout[samp] = F32ToInt32(dryGain *  Int32ToF32(Sw[samp]) + (out_ptr[ch_LFE] + fdn_out[3])); // LFE
    } // for samp
}

}}} // namespace nn::audio::detail

