﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdint>

#if defined(__ARM_NEON__) || defined(__ARM_NEON)
#include "detail/audio_Volume.neon.h"
#else  // defined(__ARM_NEON__) || defined(__ARM_NEON)
#include "detail/audio_Volume.generic.h"
#endif  // defined(__ARM_NEON__) || defined(__ARM_NEON)

namespace nn { namespace audio { namespace dsp {

void ApplyUniformGain1(int32_t* output, const int32_t* input, int32_t gain, int sampleCount) NN_NOEXCEPT
{
    const int Q = 15;
    for (auto i = 0; i < sampleCount; ++i)
    {
        output[i] = (static_cast<int64_t>(input[i]) * gain + (1 << (Q - 1))) >> Q;
    }
}

void ApplyUniformGain2(int32_t* output, const int32_t* input, int32_t gain, int sampleCount) NN_NOEXCEPT
{
    return detail::ApplyUniformGain2(output, input, gain, sampleCount);
}

void ApplyUniformGain4(int32_t* output, const int32_t* input, int32_t gain, int sampleCount) NN_NOEXCEPT
{
    return detail::ApplyUniformGain4(output, input, gain, sampleCount);
}

void ApplyUniformGain(int32_t* output, const int32_t* input, int32_t gain, int sampleCount) NN_NOEXCEPT
{
    if (!(sampleCount & 0x3))
    {
        return ApplyUniformGain4(output, input, gain, sampleCount);
    }
    else if (!(sampleCount & 0x1))
    {
        return ApplyUniformGain2(output, input, gain, sampleCount);
    }
    else
    {
        return ApplyUniformGain1(output, input, gain, sampleCount);
    }
}

void ApplyUniformGain1(float* output, const float* input, float gain, int sampleCount) NN_NOEXCEPT
{
    for (auto i = 0; i < sampleCount; ++i)
    {
        output[i] = input[i] * gain;
    }
}

void ApplyUniformGain4(float* output, const float* input, float gain, int sampleCount) NN_NOEXCEPT
{
    return detail::ApplyUniformGain4(output, input, gain, sampleCount);
}

void ApplyUniformGain(float* output, const float* input, float gain, int sampleCount) NN_NOEXCEPT
{
    if (!(sampleCount & 0x3))
    {
        return ApplyUniformGain4(output, input, gain, sampleCount);
    }
    else
    {
        return ApplyUniformGain1(output, input, gain, sampleCount);
    }
}

void ApplyLinearEnvelopeGain1(int32_t* output, const int32_t* input, int32_t gain, int32_t delta, int sampleCount) NN_NOEXCEPT
{
    const int Q = 15;
    for (auto i = 0; i < sampleCount; ++i)
    {
        output[i] = (static_cast<int64_t>(input[i]) * gain + (1 << (Q - 1))) >> Q;
        gain += delta;
    }
}

void ApplyLinearEnvelopeGain2(int32_t* output, const int32_t* input, int32_t gain, int32_t delta, int sampleCount) NN_NOEXCEPT
{
    return detail::ApplyLinearEnvelopeGain2(output, input, gain, delta, sampleCount);
}

void ApplyLinearEnvelopeGain4(int32_t* output, const int32_t* input, int32_t gain, int32_t delta, int sampleCount) NN_NOEXCEPT
{
    return detail::ApplyLinearEnvelopeGain4(output, input, gain, delta, sampleCount);
}

void ApplyLinearEnvelopeGain(int32_t* output, const int32_t* input, int32_t gain, int32_t delta, int sampleCount) NN_NOEXCEPT
{
    if (!(sampleCount & 0x3))
    {
        return ApplyLinearEnvelopeGain4(output, input, gain, delta, sampleCount);
    }
    else if (!(sampleCount & 0x1))
    {
        return ApplyLinearEnvelopeGain2(output, input, gain, delta, sampleCount);
    }
    else
    {
        return ApplyLinearEnvelopeGain1(output, input, gain, delta, sampleCount);
    }
}

void ApplyLinearEnvelopeGain1(float* output, const float* input, float gain, float delta, int sampleCount) NN_NOEXCEPT
{
    for (auto i = 0; i < sampleCount; ++i)
    {
        output[i] = input[i] * gain;
        gain += delta;
    }
}

void ApplyLinearEnvelopeGain4(float* output, const float* input, float gain, float delta, int sampleCount) NN_NOEXCEPT
{
    return detail::ApplyLinearEnvelopeGain4(output, input, gain, delta, sampleCount);
}

void ApplyLinearEnvelopeGain(float* output, const float* input, float gain, float delta, int sampleCount) NN_NOEXCEPT
{
    if (!(sampleCount & 0x3))
    {
        return ApplyLinearEnvelopeGain4(output, input, gain, delta, sampleCount);
    }
    else
    {
        return ApplyLinearEnvelopeGain1(output, input, gain, delta, sampleCount);
    }
}

}}}  // namespace nn::audio::dsp
