﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*
File:     audio_Upsampler.cpp
Block SRC
*---------------------------------------------------------------------------*/

#include "audio_DspCommon.h"
#include "audio_Cache.h"

#if defined(NN_AUDIO_ENABLE_NEON_OPTIMIZATION)
#include <arm_neon.h>
#endif

#include "audio_Upsampler.h"

namespace nn {
namespace audio {

/***********************************************************************
For the special case of 32k->48k conversion, 2 input samples will generate
3 output samples. Let's use t0 and (t0+T) to represent the time stamps of
the 2 input samples, the time stamps for the 3 output samples will be
t0, t0+(2/3)*T, and t0+T+(1/3)*T. This pattern repeats itself every 2 input samples.

To calculate the values for the output samples, we shift the Sinc waveform
along the time axis to place the maximum or original at the output sample position,
and multiply the Sinc values at the input sample positions with the corresponding
input values. For the output sample at (t0+(2/3)*T), we just need 10 Sinc Coefficients
that are 2/3 off the original time or zero crossing time to multiply with 10
corresponding input samples older than the current time, and 10 Sinc Coefficients
that are 1/3 off the original time or zero crossing time to multiply with 10
corresponding input samples after the current time (into the future).

Similar for t0+T+(1/3)*T, where it would be 1/3 and older data, 2/3 and newer data.

g_H1ThirdP[] holds the Sinc coefficients that are the values of the Sinc waveform
            located at 1/3 off the original time or zero crossings
g_H2ThirdP[] holds the Sinc coefficients that are the values of the Sinc waveform
            located at 2/3 off the original time or zero crossings
g_H5SixthP[] holds the Sinc coefficients that are the values of the Sinc waveform
            located at 5/6 off the original time or zero crossings
g_H1HalfP[] holds the Sinc coefficients that are the values of the Sinc waveform
            located at 1/2 off the original time or zero crossings

You can find A C project that generates the full Sinc Waveform and Octave scripts
that pick out the needed Sinc Coefficients and saved in h1thirdP[] etc. here:

http://spdlybra.nintendo.co.jp/confluence/display/SIGLO/New+back-end+SRC+bandlimited+Resampling

 **********************************************************************/

// choose which size coefficients to use
// 16 bit has so far found to be OK
#define NN_AUDIO_USE_16_BIT

#ifdef NN_AUDIO_USE_16_BIT
static const uint16_t Nhxn = 15; /* Q15 format */

static const int16_t g_H2ThirdP[] = { 13295,-4813,2491,-1365,728,-361,159,-58,16,-1 };
static const int16_t g_H1ThirdP[] = { 26971,-6279,3062,-1668,903,-460,212,-83,26,-4 };
static const int16_t g_H1SixthP[] = { 31253,-4218,1967,-1064,579,-299,140,-57,19,-3};
static const int16_t g_H5SixthP[] = {6076,-2463,1300,-712,376,-183,79,-27,7,0 };
static const int16_t g_H1HalfP[] = { 20640,-6316,3187,-1743,938,-471,213,-81,24,-3 };

#else

static const uint16_t Nhxn = 31; /* Q31 format */

static const int32_t g_H2ThirdP[] = { 871392825, -315438573, 163294607,-89495613,47733203,-23650871,10434740,-3868548,1077297,-157464};
static const int32_t g_H1ThirdP[] = { 1767630269,-411565628,200731944,-109373002,59219927,-30207743,13925643,-5517653,1723431,-339581 };
static const int32_t g_H1SixthP[] = { 2048279391,-276435166,128942359,-69768257,37983467,-19625850,9229136,-3767120,1235621,-271250 };
static const int32_t g_H5SixthP[] = { 398263767,-161470997,85230013,-46691930,24670253,-12026830,5179348,-1849757,481037,-56240 };
static const int32_t g_H1HalfP[] =  { 1352688745,-413983329,208864169,-114275934,61450562,-30910575,13950283,-5353437,1583914,-273548 };


#endif


/*! \fn    uint16_t checkBoundsUp(UpsamplerState *src_bl_config, uint16_t arrayIndex)
*  \brief
*/
inline static uint16_t checkBoundsUp(UpsamplerState *src_bl_config, uint16_t arrayIndex)
{
    if (arrayIndex > src_bl_config->input_cb_upperBound) /* CHECK ON BOUNDS */
    {
        arrayIndex -= 20 ; //2 * src_bl_config->nz_input; /* buffer size 2*src_bl_config->nz_input */
    }
    return arrayIndex;
}


#ifdef NN_AUDIO_USE_16_BIT
/*! \fn    int32_t FilterUpLeft(int16_t *h, uint16_t Xi, UpsamplerState *src_bl_config)
*  \brief
*/
static int32_t FilterUpLeft(const int16_t* h, uint16_t Xi, UpsamplerState* src_bl_config)
{
    int16_t* Hp = (int16_t *) h;
    int64_t  v = 0;
    uint16_t  j;
    uint16_t  jj = Xi - src_bl_config->input_cb_lowerBound + 1;

    uint16_t  loop1 = (jj > 10) ? 10 : jj;

    for (j = 0; j < loop1; j++)
    {
        v += static_cast<int64_t>(*Hp) * src_bl_config->input_samples[Xi];                                 /* The filter output */
        Hp++;                                   /* Filter coeff step */
        Xi--;
    }
    Xi = src_bl_config->input_cb_upperBound;
    for (; j < 10; j++)
    {
        v += static_cast<int64_t>(*Hp) * src_bl_config->input_samples[Xi];                                 /* The filter output */
        Hp++;                                   /* Filter coeff step */
        Xi--;                              /* Input signal step. */
    }
    v >>= Nhxn;

    return((int32_t)v);
}

/*! \fn    int32_t FilterUpRight(int16_t *h, uint16_t Xi, UpsamplerState *src_bl_config)
*  \brief
*/
static int32_t FilterUpRight(const int16_t* h, uint16_t Xi, UpsamplerState* src_bl_config)
{
    int16_t* Hp = (int16_t *)h;
    int64_t  v = 0;
    uint16_t  j;
    uint16_t  jj = src_bl_config->input_cb_upperBound - Xi + 1;

    uint16_t  loop1 = (jj > 10) ? 10 : jj;

    for (j = 0; j < loop1; j++)
    {
        v += static_cast<int64_t>(*Hp) * src_bl_config->input_samples[Xi];
        Hp++;
        Xi++;
    }

    Xi = src_bl_config->input_cb_lowerBound;

    for (; j < 10; j++)
    {
        v += static_cast<int64_t>(*Hp) * src_bl_config->input_samples[Xi];                                 /* The filter output */
        Hp++;                                   /* Filter coeff step */
        Xi++;                              /* Input signal step. */
    }

    v >>= Nhxn;

    return static_cast<int32_t>(v);
}

#else

#if defined(NN_AUDIO_ENABLE_NEON_OPTIMIZATION)
/*! \fn    int32_t FilterUpLeft(int16_t *h, uint16_t Xi, UpsamplerState *src_bl_config)
*  \brief
*/
static int32_t FilterUpLeft(const int32_t* h, uint16_t Xi, UpsamplerState* src_bl_config)
{
    int64_t  v = 0;
    uint16_t  j;
    uint16_t  jj = Xi - src_bl_config->input_cb_lowerBound + 1;
    uint16_t  loop1 = (jj > 10) ? 10 : ((jj >> 1)<<1);

    int32x2_t inputVec;
    int32x2_t coeffVec;
    int64x2_t sumVec;
    for (j = 0; j < loop1; j += 2)
    {
        //v += ((int64_t)(* Hp))*src_bl_config->input_samples[Xi];
        //Hp++;
        //Xi--;
        inputVec = vld1_s32(&(src_bl_config->input_samples[Xi - 1 - j]));
        inputVec = vrev64_s32(inputVec);
        coeffVec = vld1_s32(h + j);
        sumVec = vmlal_s32(sumVec, inputVec, coeffVec);
    }

    if (jj != ((jj >> 1)<<1))
    {
        inputVec = vld1_lane_s32(&(src_bl_config->input_samples[Xi - j]), inputVec, 0);
        jj = src_bl_config->input_cb_upperBound;
        inputVec = vld1_lane_s32(&(src_bl_config->input_samples[jj]), inputVec, 1);
        coeffVec = vld1_s32(h + j);
        sumVec = vmlal_s32(sumVec, inputVec, coeffVec);
        j += 2;
        jj--;
    }
    else
    {
        jj = src_bl_config->input_cb_upperBound;
    }

    for (; j < 10; j += 2)
    {
        inputVec = vld1_s32(&(src_bl_config->input_samples[jj - 1]));
        inputVec = vrev64_s32(inputVec);
        coeffVec = vld1_s32(h + j);
        sumVec = vmlal_s32(sumVec, inputVec, coeffVec);
        jj -= 2;
    }
    v = vgetq_lane_s64(sumVec, 0) + vgetq_lane_s64(sumVec, 1);
    v >>= Nhxn;

    return static_cast<int32_t>(v);
}

#else
/*! \fn    int32_t FilterUpLeft(int16_t *h, uint16_t Xi, UpsamplerState *src_bl_config)
*  \brief
*/
static int32_t FilterUpLeft(const int32_t* h, uint16_t Xi, UpsamplerState* src_bl_config)
{
    int32_t* Hp = (int32_t *) h;
    int64_t  v = 0;
    uint16_t  j;
    uint16_t  jj = Xi - src_bl_config->input_cb_lowerBound + 1;
    uint16_t  loop1 = (jj > 10) ? 10 : jj;

    for (j = 0; j < loop1; j++)
    {
        v += static_cast<int64_t>(*Hp) * src_bl_config->input_samples[Xi];                                 /* The filter output */
        Hp++;                                   /* Filter coeff step */
        Xi--;
    }
    Xi = src_bl_config->input_cb_upperBound;
    for (; j < 10; j++)
    {
        v += static_cast<int64_t>(*Hp) * src_bl_config->input_samples[Xi];                                 /* The filter output */
        Hp++;                                   /* Filter coeff step */
        Xi--;                              /* Input signal step. */
    }
    v >>= Nhxn;

    return static_cast<int32_t>(v);
}
#endif

#if defined(NN_AUDIO_ENABLE_NEON_OPTIMIZATION)
/*! \fn    int32_t FilterUpRight(int16_t *h, uint16_t Xi, UpsamplerState *src_bl_config)
*  \brief
*/
static int32_t FilterUpRight(const int32_t* h, uint16_t Xi, UpsamplerState* src_bl_config)
{
    int32_t* Hp = (int32_t *) h;
    int32_t* inputP = &(src_bl_config->input_samples[Xi]);
    int64_t  v = 0;
    uint16_t  j;
    uint16_t  jj = src_bl_config->input_cb_upperBound - Xi + 1;
    uint16_t  loop1 = (jj > 10) ? 10 : ((jj >> 1)<<1);

    int32x2_t inputVec;
    int32x2_t coeffVec;
    int64x2_t sumVec;
    for (j = 0; j < loop1; j += 2)
    {
        inputVec = vld1_s32(inputP);
        coeffVec = vld1_s32(Hp);
        sumVec = vmlal_s32(sumVec, inputVec, coeffVec);
        inputP += 2;
        Hp += 2;
    }

    if (jj != ((jj >> 1)<<1))
    {
        inputVec = vld1_lane_s32(inputP, inputVec, 0);
        inputP = &(src_bl_config->input_samples[src_bl_config->input_cb_lowerBound]);
        inputVec = vld1_lane_s32(inputP, inputVec, 1);
        coeffVec = vld1_s32(Hp);
        sumVec = vmlal_s32(sumVec, inputVec, coeffVec);
        inputP += 1;
        Hp += 2;
        j += 2;
    }
    else
    {
        inputP = &(src_bl_config->input_samples[src_bl_config->input_cb_lowerBound]);
    }

    for (; j < 10; j += 2)
    {
        inputVec = vld1_s32(inputP);
        coeffVec = vld1_s32(Hp);
        sumVec = vmlal_s32(sumVec, inputVec, coeffVec);
        inputP += 2;
        Hp += 2;
    }
    v = vgetq_lane_s64(sumVec, 0) + vgetq_lane_s64(sumVec, 1);
    v >>= Nhxn;

    return((int32_t)v);
}

#else
/*! \fn    int32_t FilterUpRight(int16_t *h, uint16_t Xi, UpsamplerState *src_bl_config)
*  \brief
*/
static int32_t FilterUpRight(const int32_t* h, uint16_t Xi, UpsamplerState* src_bl_config)
{
    int32_t* Hp = (int32_t *)h;// nullptr;
    int64_t  v = 0;
    uint16_t  j;
    uint16_t  jj = src_bl_config->input_cb_upperBound - Xi + 1;
    uint16_t  loop1 = (jj > 10) ? 10 : jj;

    for (j = 0; j < loop1; j++)
    {
        v += static_cast<int64_t>(*Hp) * src_bl_config->input_samples[Xi];                                 /* The filter output */
        Hp++;                                   /* Filter coeff step */
        Xi++;
    }

    Xi = src_bl_config->input_cb_lowerBound;

    for (; j < 10; j++)
    {
        v += static_cast<int64_t>(*Hp) * src_bl_config->input_samples[Xi];                                 /* The filter output */
        Hp++;                                   /* Filter coeff step */
        Xi++;                              /* Input signal step. */
    }

    v >>= Nhxn;

    return((int32_t)v);
}
#endif

#endif


/*! \fn   SrcInitBandlimitedSrc(UpsamplerState* src_bl_config, int32_t ratio)
*  \brief initialization for Bandlimited SRC.
*
*  Initialize block SRC configuration buffer
*
*  \param[in]  UpsamplerState *src_bl_config: The SRC state and configuration
*  \param[in]  uint32_t ratio:       SRC ratio
*  \param[out] Return number of history samples initialized
*/

int32_t SrcInitBandlimitedSrc(UpsamplerState* src_bl_config, int32_t ratio)
{
    /* Clear input sample buffer before write pointer */
    memset(src_bl_config->input_samples, 0, sizeof(src_bl_config->input_samples));

    src_bl_config->nz_input = NN_AUDIO_SRC_BL_NZ;
    src_bl_config->ratioRho = ratio;

    src_bl_config->input_current = (NN_AUDIO_SRC_BL_CB_MID - 1);
    src_bl_config->input_write = 0;
    src_bl_config->input_cb_lowerBound = static_cast<uint8_t>(NN_AUDIO_SRC_BL_CB_MID - src_bl_config->nz_input);
    src_bl_config->input_cb_upperBound = static_cast<uint8_t>(NN_AUDIO_SRC_BL_CB_MID - 1 + src_bl_config->nz_input);
    src_bl_config->outputIndex = 0;
    src_bl_config->isInitialized = 1;

    return NN_AUDIO_SRC_BL_NZ;
}

/*! \fn  SrcBl32kTo48k(int32_t* pOutData, int32_t* pInData, uint32_t outSampleCount, UpsamplerState *src_bl_config
*
*  \brief  run SRC for rho >= 1
*
*  \param[in]  AUDIO_BUFFER *pSrc   Audio context buffer
*  \param[in]  UpsamplerState *src_bl_config
*  \param[out]
*  \return 0
*/
static int32_t SrcBl32kTo48k(int32_t* pOutData, int32_t* pInData, uint32_t outSampleCount, UpsamplerState* src_bl_config)
{
    int32_t v_L;
    uint32_t inSamples = 0;
    int32_t* pOut = pOutData;
    uint32_t index;

    /*
        now generate a frames worth of output samples
    */
    for (index = 0; index < outSampleCount; index++)
    {
        switch (src_bl_config->outputIndex)
        {
        case 0:
            /* Read in a sample and update the current position */
            src_bl_config->input_samples[src_bl_config->input_write++] = (*pInData++ << 8); /* Q0 format */
            src_bl_config->input_write = checkBoundsUp(src_bl_config, src_bl_config->input_write);
            inSamples++;
            src_bl_config->input_current++;
            src_bl_config->input_current = checkBoundsUp(src_bl_config, src_bl_config->input_current);
            /* write output, currently assume outout to *output, need to change once interface defined  */
            *pOut++ = (src_bl_config->input_samples[src_bl_config->input_current] >> 8);
            src_bl_config->outputIndex++;
            break;

        case 1:
            /* apply left wing of the filter */
            v_L = FilterUpLeft(g_H2ThirdP, src_bl_config->input_current, src_bl_config);

            /* apply right wing of the filter */
            v_L += FilterUpRight(g_H1ThirdP, checkBoundsUp(src_bl_config, src_bl_config->input_current + 1), src_bl_config);

            /* write output, currently assume outout to *output, need to change once interface defined  */
            *pOut++ = (v_L >> 8);
            src_bl_config->outputIndex++;
            break;

        case 2:
            /* read in a sample */
            src_bl_config->input_samples[src_bl_config->input_write++] = (*pInData++ << 8); /* Q0 format */
            src_bl_config->input_write = checkBoundsUp(src_bl_config, src_bl_config->input_write);
            inSamples++;
            src_bl_config->input_current++;
            src_bl_config->input_current = checkBoundsUp(src_bl_config, src_bl_config->input_current);
            /* apply left wing of the filter */
            v_L = FilterUpLeft(g_H1ThirdP, src_bl_config->input_current, src_bl_config);

            /* apply right wing of the filter */
            v_L += FilterUpRight(g_H2ThirdP, checkBoundsUp(src_bl_config, src_bl_config->input_current + 1), src_bl_config);

            /* write output, currently assume outout to *output, need to change once interface defined  */
            *pOut++ = (v_L >> 8);
            src_bl_config->outputIndex = 0;
            break;

        default:
            break;
        }
    }

    return inSamples;
}




/*! \fn  SrcBl16kTo48k(int32_t* pOutData, int32_t* pInData, uint32_t outSampleCount, UpsamplerState *src_bl_config)
*
*  \brief  run SRC for rho >= 1
*
*  \param[in]  AUDIO_BUFFER *pSrc   Audio context buffer
*  \param[in]  UpsamplerState *src_bl_config
*  \param[out]
*  \return 0
*/
static int32_t SrcBl16kTo48k(int32_t* pOutData, int32_t* pInData, uint32_t outSampleCount, UpsamplerState* src_bl_config)
{
    int32_t v_L;
    int32_t input;
    uint32_t inSamples = 0;
    int32_t* pOut = pOutData;
    uint32_t index;

    /*
        generate a frames worth of output samples
    */
    for (index = 0; index < outSampleCount; index++)
    {
        switch (src_bl_config->outputIndex)
        {
        case 0:
            /* read in a sample and update the current position */
            input = *pInData++;
            src_bl_config->input_samples[src_bl_config->input_write++] = (input << 8); /* Q0 format */
            src_bl_config->input_write = checkBoundsUp(src_bl_config, src_bl_config->input_write);
            inSamples++;
            src_bl_config->input_current++;
            src_bl_config->input_current = checkBoundsUp(src_bl_config, src_bl_config->input_current);
            /* write output, currently assume outout to *output, need to change once interface defined  */
            *pOut++ = src_bl_config->input_samples[src_bl_config->input_current];
            src_bl_config->outputIndex++;
            break;

        case 1:
            /* apply left wing of the filter */
            v_L = FilterUpLeft(g_H1ThirdP, src_bl_config->input_current, src_bl_config);

            /* apply right wing of the filter */
            v_L += FilterUpRight(g_H2ThirdP, checkBoundsUp(src_bl_config, src_bl_config->input_current + 1), src_bl_config);

            /* write output, currently assume outout to *output, need to change once interface defined  */
            *pOut++ = v_L;
            src_bl_config->outputIndex++;
            break;

        case 2:
            /* apply left wing of the filter */
            v_L = FilterUpLeft(g_H2ThirdP, src_bl_config->input_current, src_bl_config);

            /* apply right wing of the filter */
            v_L += FilterUpRight(g_H1ThirdP, checkBoundsUp(src_bl_config, src_bl_config->input_current + 1), src_bl_config);

            /* write output, currently assume outout to *output, need to change once interface defined  */
            *pOut++ = v_L;
            src_bl_config->outputIndex = 0;
            break;

        default:
            break;
        }
    }

    return inSamples;
}




/*! \fn  SrcBl8kTo48k(int32_t* pOutData, int32_t* pInData, uint32_t outSampleCount, UpsamplerState *src_bl_config)
*
*  \brief  run SRC for rho >= 1
*
*  \param[in]  AUDIO_BUFFER *pSrc   Audio context buffer
*  \param[in]  UpsamplerState *src_bl_config
*  \param[out]
*  \return 0
*/
static int32_t SrcBl8kTo48k(int32_t* pOutData, int32_t* pInData, uint32_t outSampleCount, UpsamplerState* src_bl_config)
{
    int32_t v_L;
    int32_t input;
    uint32_t inSamples = 0;
    int32_t* pOut = pOutData;
    uint32_t index;

    /*
        generate a frames worth of output samples
    */
    for (index = 0; index < outSampleCount; index++)
    {
        switch (src_bl_config->outputIndex)
        {
        case 0:
            /* read in a sample and update the current position */
            input = *pInData++;
            src_bl_config->input_samples[src_bl_config->input_write++] = (input << 8); /* Q0 format */
            src_bl_config->input_write = checkBoundsUp(src_bl_config, src_bl_config->input_write);
            inSamples++;
            src_bl_config->input_current++;
            src_bl_config->input_current = checkBoundsUp(src_bl_config, src_bl_config->input_current);
            /* write output, currently assume outout to *output, need to change once interface defined  */
            *pOut++ = src_bl_config->input_samples[src_bl_config->input_current];
            src_bl_config->outputIndex++;
            break;

        case 1:
            /* apply left wing of the filter */
            v_L = FilterUpLeft(g_H1SixthP, src_bl_config->input_current, src_bl_config);

            /* apply right wing of the filter */
            v_L += FilterUpRight(g_H5SixthP, checkBoundsUp(src_bl_config, src_bl_config->input_current + 1), src_bl_config);

            /* write output, currently assume outout to *output, need to change once interface defined  */
            *pOut++ = v_L;
            src_bl_config->outputIndex++;
            break;

        case 2:
            /* apply left wing of the filter */
            v_L = FilterUpLeft(g_H1ThirdP, src_bl_config->input_current, src_bl_config);

            /* apply right wing of the filter */
            v_L += FilterUpRight(g_H2ThirdP, checkBoundsUp(src_bl_config, src_bl_config->input_current + 1), src_bl_config);

            /* write output, currently assume outout to *output, need to change once interface defined  */
            *pOut++ = v_L;
            src_bl_config->outputIndex++;
            break;

        case 3:
            /* apply left wing of the filter */
            v_L = FilterUpLeft(g_H1HalfP, src_bl_config->input_current, src_bl_config);

            /* apply right wing of the filter */
            v_L += FilterUpRight(g_H1HalfP, checkBoundsUp(src_bl_config, src_bl_config->input_current + 1), src_bl_config);

            /* write output, currently assume outout to *output, need to change once interface defined  */
            *pOut++ = v_L;
            src_bl_config->outputIndex++;
            break;

        case 4:
            /* apply left wing of the filter */
            v_L = FilterUpLeft(g_H2ThirdP, src_bl_config->input_current, src_bl_config);

            /* apply right wing of the filter */
            v_L += FilterUpRight(g_H1ThirdP, checkBoundsUp(src_bl_config, src_bl_config->input_current + 1), src_bl_config);

            /* write output, currently assume outout to *output, need to change once interface defined  */
            *pOut++ = v_L;
            src_bl_config->outputIndex++;
            break;

        case 5:
            /* apply left wing of the filter */
            v_L = FilterUpLeft(g_H5SixthP, src_bl_config->input_current, src_bl_config);

            /* apply right wing of the filter */
            v_L += FilterUpRight(g_H1SixthP, checkBoundsUp(src_bl_config, src_bl_config->input_current + 1), src_bl_config);

            /* write output, currently assume outout to *output, need to change once interface defined  */
            *pOut++ = v_L;
            src_bl_config->outputIndex = 0;
            break;

        default:
            break;
        }
    }

    return inSamples;
}

/*********************************************************************************/
/*! \fn  SrcProcessFrame(AUDIO_BUFFER *pSrc, int32_t *pSamples, UpsamplerState *src_bl_config)
*
*  \brief
*
*  \param[in]  AUDIO_BUFFER *pSrc   Audio context buffer
*  \param[in]  UpsamplerState *src_bl_config
*  \param[in/out] int32_t *pSamples     Input Samples, advanced after call
*  \return     voidr
*  \return 0
*/
void SrcProcessFrame(int32_t* pOutData, int32_t* pInData, uint32_t outSampleCount, uint32_t inSampleCount, UpsamplerState* src_bl_config)
{

    NN_AUDIO_TRACK_BUFFER(src_bl_config, sizeof(UpsamplerState), dsp::CachedBuffer_RequireInval | dsp::CachedBuffer_RequireFlush);
    if (!src_bl_config->isInitialized)
    {
        switch (inSampleCount)
        {
        case 160:
            SrcInitBandlimitedSrc(src_bl_config, NN_AUDIO_SRC_BL_RATIO_32_48);
            break;
        case 80:
            SrcInitBandlimitedSrc(src_bl_config, NN_AUDIO_SRC_BL_RATIO_16_48);
            break;
        case 40:
            SrcInitBandlimitedSrc(src_bl_config, NN_AUDIO_SRC_BL_RATIO_8_48);
            break;
        default:
            NN_AUDIO_DSP_ASSERT(0);
            break;
        }
    }
    if (src_bl_config->ratioRho == NN_AUDIO_SRC_BL_RATIO_32_48)  /*(1 << NN_AUDIO_NQRATIO))*1.5  up sampling */
    {
        SrcBl32kTo48k(pOutData, pInData, outSampleCount, src_bl_config);
    }
    else if (src_bl_config->ratioRho == NN_AUDIO_SRC_BL_RATIO_16_48)  /*(1 << NN_AUDIO_NQRATIO))*3  up sampling */
    {
        SrcBl16kTo48k(pOutData, pInData, outSampleCount, src_bl_config);
    }
    else if (src_bl_config->ratioRho == NN_AUDIO_SRC_BL_RATIO_8_48)  /*(1 << NN_AUDIO_NQRATIO))*6  up sampling */
    {
        SrcBl8kTo48k(pOutData, pInData, outSampleCount, src_bl_config);
    }
}

}  // namespace audio
}  // namespace nn
