﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "audio_DspCommon.h"
#include "audio_EffectConstants.h"

namespace nn {
namespace audio {

// Returns a fixed point (Sign:Magnitude:Fractional == 1:1:30) value between -1.0 and 1.0 inclusive between
// index range [0 .. 256].  The 256th value is equivalent to the 0th to aid in interpolation.
static const int32_t SinTable[] = { NN_AUDIO_DSP_SIN_TABLE };

// Returns a fixed point (Sign:Magnitude:Fractional == 1:1:30) value between -1.0 and 0.0 inclusive given an index between 0 and 255.
// This models 10.0^x where x is between [-5.3 to 0].
// POW2TABLE is generated via fxGenerateConstants.c and written to fxconst.h.
static const uint32_t Pow10Table[] = { POW10TABLE };

// Fixed point type
typedef int32_t qf;

/*---------------------------------------------------------------------------*
Fixed Point -- Sign:Magnitude:Fractional == 1:17:14

The format can be altered by changing QF_FRACTIONAL_BIT_COUNT below.  Constants in qconst.h will
need to be regenerated.
*---------------------------------------------------------------------------*/

// Fractional bit count
static const int QF_FRACTIONAL_BIT_COUNT = 14;

// Integer bit count relative to the fixed point type bit size and value of QF_FRACTIONAL_BIT_COUNT
static const int QI_INTEGER_BIT_COUNT = (NN_BITSIZEOF(qf) - QF_FRACTIONAL_BIT_COUNT);

#define QF_CONST(x) ( static_cast<int32_t>( (x) * ( 1 << QF_FRACTIONAL_BIT_COUNT)) )

// --------------------------------------------------------------------------------
// qf arithmetic operations
// --------------------------------------------------------------------------------
/* Convert fixed-point qf to int32_t */
static inline int32_t QfToS32(qf q) NN_NOEXCEPT
{
    return (int32_t)(q >> QF_FRACTIONAL_BIT_COUNT) + !!(q & (1 << (QF_FRACTIONAL_BIT_COUNT - 1)));
}

/* Convert s32 to fixed-point qf
*/
static inline qf S32ToQf(int32_t i) NN_NOEXCEPT
{
    return (qf)(i << QF_FRACTIONAL_BIT_COUNT);
}

/* Convert q8 to fixed-point qf
*/
static inline qf Q8ToQf(int32_t q8) NN_NOEXCEPT
{
    return (qf)(q8 << (QF_FRACTIONAL_BIT_COUNT - 8));
}

/* Convert fixed-point qf to q8
*/
static inline int32_t QfToQ8(qf q) NN_NOEXCEPT
{
    return (int32_t)(q >> (QF_FRACTIONAL_BIT_COUNT - 8));
}

/* Return magnitude/integer bits in a int32_t
*/
static inline int32_t QfMagnitude(qf q) NN_NOEXCEPT
{
    return (int32_t)(q >> QF_FRACTIONAL_BIT_COUNT);
}

/* Return qf with magnitude masked in
*/
static inline qf QfMagMasked(qf q) NN_NOEXCEPT
{
    return (q & ~((1 << QF_FRACTIONAL_BIT_COUNT) - 1));
}

/* Return fractional bits in a int32_t
*/
static inline int32_t QfFrac(qf q) NN_NOEXCEPT
{
    return (q & ((1 << QF_FRACTIONAL_BIT_COUNT) - 1));
}

static inline qf MultiQfQ015(qf a, qf b) NN_NOEXCEPT
{
    return (QfMagnitude(a) * b) + ((QfFrac(a) * b) >> QF_FRACTIONAL_BIT_COUNT);
}  //result in q0.15 format

static inline qf MultiQfS32(qf a, int32_t b) NN_NOEXCEPT
{
    return (a * b);
}

/* Multiply qf and qf
*/
static inline qf MultiQfQf(qf a, qf b) NN_NOEXCEPT
{
    return (qf)((int64_t)(a) * b >> QF_FRACTIONAL_BIT_COUNT);
}

static inline qf AddQfQf(qf a, qf b) NN_NOEXCEPT
{
    return (a + b);
}

static inline qf SubtractQfQf(qf a, qf b) NN_NOEXCEPT
{
    return (a - b);
} //result in qf format

static inline qf DivQfQf(qf dividend, qf divisor) NN_NOEXCEPT
{
    uint32_t shiftDivisor = 0, shiftDividend = 0, sign = 0, i, a, b, quotient = 0;

    if (0 == dividend) { return 0; }
    NN_AUDIO_DSP_ASSERT(0 != divisor, " division by zero: %08x / %08x", dividend, divisor);
    if (0 == divisor) { return 0; }

    if (dividend < 0) { a = -dividend; sign = 1; }
    else { a = dividend; }
    if (divisor < 0)  { b = -divisor;  sign ^= 1; }
    else { b = divisor; }

    // Shift the divisor and dividend bits until they align on the 30th bit
    while (!(a & (1 << 30)) && shiftDividend<31) { a <<= 1; ++shiftDividend; }
    while (!(b & (1 << 30)) && shiftDivisor<31) { b <<= 1; ++shiftDivisor; }

    if ((QF_FRACTIONAL_BIT_COUNT + 1) + shiftDivisor < shiftDividend)
    {
        return 0;
    }
    // Only need to compute a fixed set of bits dependent on the fractional bit count, divisor shift count, and dividend shift count
    for (i = (QF_FRACTIONAL_BIT_COUNT + 1) + shiftDivisor - shiftDividend; (i && a); --i) {
        quotient <<= 1; // Shift quotient for the next possible incoming bit.  Position 0 will get a 1 if divisor is subtracted
        if (b <= a) {
            a = SubtractQfQf(a, b);
            quotient |= 1;
        }
        a <<= 1; // Shift dividend
    }

    if (b <= a) ++quotient; // Round up if the theoretical next subtraction from the dividend would have been successful.
    if ( i > 0)
    {
        quotient <<= i; // If loop exited early, shift the quotient by the remaining loop count
    }
    return sign ? -(int32_t)quotient : (int32_t)quotient; // Adjust sign
}

static inline qf Pow10Qf(qf idx) NN_NOEXCEPT
{
    int32_t a, b;
    uint32_t i;
    qf ret;
    //return f64toqf(pow(10.0, qftof64(idx)));

    if (QF_CONST(0.0f) <= idx)
        ret = QF_CONST(1.0f);
    else if (idx <= QF_CONST(-5.3f))
        ret = QF_CONST(0.0f);
    else {

       idx = MultiQfQf(idx, QF_CONST(-48.30189f)); // Normalize the index between 0 to 255

        i = QfMagnitude(idx); // Consider the index' magnitude

        i = i & 255; // i = i % 256, just to be double sure that i is between 0 and 255
        // Consider the two consecutive sin values to interpolate.
        a = Pow10Table[i];
        b = Pow10Table[(i + 1)];

        // Interpolate based on the idx's qf fractional part.  b and a are both q30 but the multiply call works since the result is whatever fixed-point format
        ret = (MultiQfQ015(b - a, QfFrac(idx)) + a) >> (30 - QF_FRACTIONAL_BIT_COUNT);
    }

    return ret;
}

/* Compute sqrt by first approximating the value by shifting, the iterating (at most 5 times): guess = (q/guess + guess) / 2.
*/
static inline qf SqrtQf(qf q) NN_NOEXCEPT
{
    int shift;
    qf guess, last;

    if ((q >> QF_FRACTIONAL_BIT_COUNT) > 0x10000) shift = 9; // 262144.0
    else if ((q >> QF_FRACTIONAL_BIT_COUNT) > 0x04000) shift = 8; //  65536.0
    else if ((q >> QF_FRACTIONAL_BIT_COUNT) > 0x01000) shift = 7; //  16384.0
    else if ((q >> QF_FRACTIONAL_BIT_COUNT) > 0x00400) shift = 6; //   4096.0
    else if ((q >> QF_FRACTIONAL_BIT_COUNT) > 0x00100) shift = 5; //   1024.0
    else if ((q >> QF_FRACTIONAL_BIT_COUNT) > 0x00040) shift = 4; //    256.0
    else if ((q >> QF_FRACTIONAL_BIT_COUNT) > 0x00010) shift = 3; //     64.0
    else if ((q >> QF_FRACTIONAL_BIT_COUNT) > 0x00004) shift = 2; //     16.0
    else if ((q >> QF_FRACTIONAL_BIT_COUNT) > 0x00001) shift = 1; //      4.0
    else if (q > (QF_CONST(1.0f) >> 1)) shift = 0;
    else if (q > (QF_CONST(1.0f) >> 2)) shift = -1;
    else if (q > (QF_CONST(1.0f) >> 4)) shift = -2;
    else if (q > (QF_CONST(1.0f) >> 6)) shift = -3;
    else if (q > (QF_CONST(1.0f) >> 8)) shift = -4;
    else if (q > (QF_CONST(1.0f) >> 10)) shift = -5;
    else shift = -6;

    guess = shift > 0 ? q >> shift : q << -shift;

    for (auto i = 0; i < 7; ++i)
    {
        last = guess;
        guess = (DivQfQf(q, last) + guess) >> 1;
        if (last == guess)
        {
            break;
        }
    }
    return guess;
}

/* Computes sin given a 256-modulo fixed point value. */
static inline qf SinQf(qf idx) NN_NOEXCEPT
{
    uint32_t i = QfMagnitude(idx) & 255; // Consider the magnitude, modulo-256.
    int32_t a = SinTable[i]; // Consider the next consecutive table value to interpolate.
    // Interpolate based on the idx's qf fractional part.  (q30 * q_0_15) >> 15 == q30.
    return (MultiQfQ015(SinTable[i + 1] - a, QfFrac(idx)) + a) >> (30 - QF_FRACTIONAL_BIT_COUNT);
}

static inline qf CosQf(qf idx) NN_NOEXCEPT
{
    return SinQf(idx + QF_CONST(64.0f));
}

}  // namespace audio
}  // namespace nn

