﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdint>

#if defined(__ARM_NEON__) || defined(__ARM_NEON)
#include "detail/audio_Mix.neon.h"
#else  // defined(__ARM_NEON__) || defined(__ARM_NEON)
#include "detail/audio_Mix.generic.h"
#endif  // defined(__ARM_NEON__) || defined(__ARM_NEON)

namespace nn { namespace audio { namespace dsp {

void ApplyMix1(int32_t* output, const int32_t* input, int32_t gain, int sampleCount) NN_NOEXCEPT
{
    const int Q = 15;
    for (auto i = 0; i < sampleCount; ++i)
    {
        output[i] += (static_cast<int64_t>(input[i]) * gain + (1 << (Q - 1))) >> Q;
    }
}

void ApplyMix2(int32_t* output, const int32_t* input, int32_t gain, int sampleCount) NN_NOEXCEPT
{
    return detail::ApplyMix2(output, input, gain, sampleCount);
}

void ApplyMix4(int32_t* output, const int32_t* input, int32_t gain, int sampleCount) NN_NOEXCEPT
{
    return detail::ApplyMix4(output, input, gain, sampleCount);
}

void ApplyMix(int32_t* output, const int32_t* input, int32_t gain, int sampleCount) NN_NOEXCEPT
{
    if (!(sampleCount & 0x3))
    {
        return ApplyMix4(output, input, gain, sampleCount);
    }
    else if (!(sampleCount & 0x1))
    {
        return ApplyMix2(output, input, gain, sampleCount);
    }
    else
    {
        return ApplyMix1(output, input, gain, sampleCount);
    }
}

void ApplyMix1(float* output, const float* input, float gain, int sampleCount) NN_NOEXCEPT
{
    for (auto i = 0; i < sampleCount; ++i)
    {
        output[i] += input[i] * gain;
    }
}

void ApplyMix4(float* output, const float* input, float gain, int sampleCount) NN_NOEXCEPT
{
    return detail::ApplyMix4(output, input, gain, sampleCount);
}

void ApplyMix(float* output, const float* input, float gain, int sampleCount) NN_NOEXCEPT
{
    if (!(sampleCount & 0x3))
    {
        return ApplyMix4(output, input, gain, sampleCount);
    }
    else
    {
        return ApplyMix1(output, input, gain, sampleCount);
    }
}

int32_t ApplyMixRamp1(int32_t* output, const int32_t* input, int32_t gain, int32_t delta, int sampleCount) NN_NOEXCEPT
{
    const int Q = 15;
    int32_t x = 0;
    for (auto i = 0; i < sampleCount; ++i)
    {
        x = (static_cast<int64_t>(input[i]) * gain + (1 << (Q - 1))) >> Q;
        output[i] += x;
        gain += delta;
    }
    return x;
}

int32_t ApplyMixRamp2(int32_t* output, const int32_t* input, int32_t gain, int32_t delta, int sampleCount) NN_NOEXCEPT
{
    auto x = input[sampleCount - 1];
    detail::ApplyMixRamp2(output, input, gain, delta, sampleCount);
    const int Q = 15;
    return (static_cast<int64_t>(x) * (gain + delta * (sampleCount - 1)) + (1 << (Q - 1))) >> Q;  // TODO: would be better if this can be calculated in detail::ApplyMixRamp2()
}

int32_t ApplyMixRamp4(int32_t* output, const int32_t* input, int32_t gain, int32_t delta, int sampleCount) NN_NOEXCEPT
{
    auto x = input[sampleCount - 1];
    detail::ApplyMixRamp4(output, input, gain, delta, sampleCount);
    const int Q = 15;
    return (static_cast<int64_t>(x) * (gain + delta * (sampleCount - 1)) + (1 << (Q - 1))) >> Q;  // TODO: would be better if this can be calculated in detail::ApplyMixRamp2()
}

int32_t ApplyMixRamp(int32_t* output, const int32_t* input, int32_t gain, int32_t delta, int sampleCount) NN_NOEXCEPT
{
    if (!(sampleCount & 0x3))
    {
        return ApplyMixRamp4(output, input, gain, delta, sampleCount);
    }
    else if (!(sampleCount & 0x1))
    {
        return ApplyMixRamp2(output, input, gain, delta, sampleCount);
    }
    else
    {
        return ApplyMixRamp1(output, input, gain, delta, sampleCount);
    }
}

float ApplyMixRamp1(float* output, const float* input, float gain, float delta, int sampleCount) NN_NOEXCEPT
{
    float x = 0;
    for (auto i = 0; i < sampleCount; ++i)
    {
        x = input[i] * gain;
        output[i] += x;
        gain += delta;
    }
    return x;
}

float ApplyMixRamp4(float* output, const float* input, float gain, float delta, int sampleCount) NN_NOEXCEPT
{
    auto x = input[sampleCount - 1];
    detail::ApplyMixRamp4(output, input, gain, delta, sampleCount);
    return x * (gain + delta * (sampleCount - 1));  // TODO: would be better if this can be calculated in detail::ApplyMixRamp4()
}

float ApplyMixRamp(float* output, const float* input, float gain, float delta, int sampleCount) NN_NOEXCEPT
{
    if (!(sampleCount & 0x3))
    {
        return ApplyMixRamp4(output, input, gain, delta, sampleCount);
    }
    else
    {
        return ApplyMixRamp1(output, input, gain, delta, sampleCount);
    }
}

}}}  // namespace nn::audio::dsp
