﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <limits>
#include <nn/nn_SdkAssert.h>
#include "audio_MailBoxImplByAdspMailBoxCpu.h"

namespace nn { namespace audio { namespace dsp {

namespace {
const nn::TimeSpan DefaultTimeout = nn::TimeSpan::FromMilliSeconds(0);
const nn::TimeSpan DefaultRetryInterval = nn::TimeSpan::FromMilliSeconds(5);
} // anonymous namespace

void MailBoxImplByAdspMailBoxCpu::Initialize() NN_NOEXCEPT
{

}

void MailBoxImplByAdspMailBoxCpu::Finalize() NN_NOEXCEPT
{

}

void MailBoxImplByAdspMailBoxCpu::Open(int id, bool isServer) NN_NOEXCEPT
{
    NN_SDK_ASSERT(isServer == false);
    NN_SDK_ASSERT(id <= std::numeric_limits<int16_t>::max());
    int16_t mailBoxId = static_cast<int16_t>(id);

    nn::os::InitializeSemaphore(& m_MailBoxSemaphore, 0, nne::audio::adsp::MboxType::RCV_BUF_SZ + 1);
    NN_ABORT_UNLESS(nne::audio::adsp::Mbox::MboxOpen(&m_AdspMailBox, &mailBoxId, nullptr, &m_MailBoxSemaphore, nullptr, nullptr, nullptr) == 0);
}

void MailBoxImplByAdspMailBoxCpu::Close() NN_NOEXCEPT
{
    NN_ABORT_UNLESS(nne::audio::adsp::Mbox::MboxClose(&m_AdspMailBox) == 0);
    nn::os::FinalizeSemaphore(&m_MailBoxSemaphore);
}

void MailBoxImplByAdspMailBoxCpu::Send(int32_t message) NN_NOEXCEPT
{
    while(nne::audio::adsp::Mbox::MboxSendMsg(&m_AdspMailBox, message, false, DefaultTimeout) != 0)
    {
        nne::audio::adsp::Mbox::MboxClearErrCode(&m_AdspMailBox);
        nn::os::SleepThread(DefaultRetryInterval);
    }
}

int32_t MailBoxImplByAdspMailBoxCpu::Recv() NN_NOEXCEPT
{
    int32_t message;

    while(nne::audio::adsp::Mbox::MboxRecvMsg(&m_AdspMailBox, &message, false, DefaultTimeout) != 0)
    {
        nne::audio::adsp::Mbox::MboxClearErrCode(&m_AdspMailBox);
        nn::os::TimedAcquireSemaphore(&m_MailBoxSemaphore, DefaultRetryInterval);
    }

    return message;
}

}}}  // namespace nn::audio::dsp

