﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/audio/audio_Common.h>

#include "audio_EffectCommon.h"
#include "audio_DelayLine.h"
#include "audio_DspCommon.h"
#include "../common/audio_Util.h"

namespace nn {
namespace audio {

/**
 * @brief ReverbType を利用するのに必要な各種パラメータ
 */
struct ReverbParameter
{
    /**
     * @brief 初期反射音モードのプリセット値です。
     */
    enum EarlyMode
    {
        EarlyMode_SmallRoom = 0,     //!< 小さい部屋
        EarlyMode_LargeRoom,         //!< 大きい部屋
        EarlyMode_Hall,              //!< ホール
        EarlyMode_Cavern,            //!< 洞窟
        EarlyMode_NoEarlyReflection, //!< 初期反射音なし
        EarlyMode_Count
    };

    /**
     * @brief 後期残響音モードのプリセット値です。
     */
    enum LateMode
    {
        LateMode_Room,               //!< 部屋
        LateMode_Hall,               //!< ホール
        LateMode_MetalCorridor,      //!< 金属製の回廊
        LateMode_Cavern,             //!< 洞窟
        LateMode_MaximumDelay,       //!< 最大ディレイ（リバーブ）
        LateMode_Count
    };

    /**
    * @brief Reverb で利用可能な最大チャンネル数
    */
    static const int ChannelCountMax = 6;
    static const int SupportedChannelCountMax = 6;

    static const int LateReverbHighFrequency = 5000; // freq at which high frequency decay is controlled
    static const int EarlyReflectionCount = 10;
    static const int FeedbackDelayNetworkCount = 4;   // Number of channels in the feedback delay network on the late reverb
    static const int EarlyReflectionDelayMax = QF_CONST(350.0f);  // 350ms
    static const int EarlyReflectionTaps = 10;
    static const int CenterDelayMax = QF_CONST(5.0f);  // 5ms
    static const int EarlyReflectionDelayMax_350ms = QF_CONST(350.0f);
    static const int EarlyReflectionDelayMax_150ms = QF_CONST(150.0f);
    static const int HighFrequencyRatioMax = QF_CONST(0.995f);

    int8_t _input[ChannelCountMax];   // 入力ミックスバッファインデックス
    int8_t _output[ChannelCountMax];  // 出力ミックスバッファインデックス
    uint16_t _numChannelCountMax;
    uint16_t _numChannels;

    // user given parameter
    qf _sampleRate;                   // サンプルレート [kHz]
    EarlyMode _earlyMode; // 初期反射音モード
    qf _earlyGain;                    // 初期反射音のゲイン
    qf _predelayTime;                 // 初期反射音から後期残響音までの遅延時間

    LateMode _lateMode;   // 後期反射音モード
    qf _lateGain;                     // 後期残響音のゲイン
    qf _decayTime;                    // 後期残響音の継続時間
    qf _highFrequencyDecayRatio;      // 高周波成分の減衰率
    qf _coloration;                   // 残響音の音色

    qf _reverbGain;                   // リバーブエフェクトへの入力ゲイン
    qf _outGain;                      // 出力ゲイン
    qf _dryGain;                      // ドライゲイン

    EffectParameterStatus _parameterStatus;
    int8_t _padding[3];
};
NN_STATIC_ASSERT(sizeof(ReverbParameter) == 68);

struct NN_AUDIO_ALIGNAS_BUFFER_ALIGN ReverbState
{
    // early reflections
    FXDelayLine _earlyDelay;
    FXDelayLine _centerDelay;
    uint32_t _earlyTap[ReverbParameter::EarlyReflectionTaps];
    qf _earlyCoef[ReverbParameter::EarlyReflectionTaps];

    // late reverb
    uint32_t _lateTap;
    FXDelayLine _allPass[ReverbParameter::FeedbackDelayNetworkCount];
    FXDelayLine _fdnDelay[ReverbParameter::FeedbackDelayNetworkCount]; //
    qf _fdnFeedback[ReverbParameter::FeedbackDelayNetworkCount];       // feedback coefficients
    qf _fdnLowPassCoef[ReverbParameter::FeedbackDelayNetworkCount];    // low-pass filter coefficients
    qf _fdnLowPassHistory[ReverbParameter::FeedbackDelayNetworkCount];
};

inline bool IsSupportedReverbChannelCount(int channel) NN_NOEXCEPT
{
    return (channel == 1) ||
           (channel == 2) ||
           (channel == 4) ||
           (channel == 6);
}

void InitializeReverbEffect(const ReverbParameter *reverb,  ReverbState* state, void* workBuffer, bool longSizePreDelaySupported);
void UpdateReverbEffectParameter(const ReverbParameter* reverb, ReverbState* state);
void ApplyReverbEffect(const ReverbParameter *reverb, ReverbState* state, bool enabled, const int32_t **ppInData, int32_t **ppOutData, const uint32_t sampleCount);
size_t ReverbGetRequiredDelayLineBufferSize(int sampleRate, int channelCount);

} // namespace audio
} // namespace nn

