﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "audio_EffectCommon.h"
#include "audio_Axfx2DelayLine.h"

#include "audio_DspCommon.h"
#include <nn/audio/audio_Common.h>
#include "../common/audio_Util.h"

namespace nn {
namespace audio {

struct I3dl2ReverbParameter
{
    static const int ChannelCountMax = 6;
    static const int SupportedChannelCountMax = 6;

    static const int EarlyReflectionDelayMax = 300;  //msec
    static const int LateReverbDelayMax = 100;  //msec
    static const int LateReverbCenterDelay = 5;  // in msec
    static const int EarlyReflectionTaps = 20;
    static const int FeedbackDelayNetworkCount = 4;   // Number of channels in the feedback delay network on the late reverb

    int8_t _input[ChannelCountMax];   // 入力ミックスバッファインデックス
    int8_t _output[ChannelCountMax];  // 出力ミックスバッファインデックス
    uint16_t _numChannelCountMax;
    uint16_t _numChannels;

    // user controlled parameter
    uint8_t _preset_not_used;  // This variable has not been used now but do not reuse it
                               // to save a compatibility with applications built with ~NXAddon 5.x.x SDK (SIGLO-80051).

    uint32_t _sampleRate;      // sample_rate in Hz
    float _roomHf;
    float _hfReference;
    float _decayTime;
    float _hfDecayRatio;
    float _room;
    float _reflections;
    float _reverb;
    float _diffusion;
    float _reflectionsDelay;
    float _reverbDelay;
    float _density;
    float _dryGain;

    EffectParameterStatus _parameterStatus;
    int8_t _padding[3];
};
NN_STATIC_ASSERT(sizeof(I3dl2ReverbParameter) == 76);

struct NN_AUDIO_ALIGNAS_BUFFER_ALIGN I3dl2ReverbState
{
    // input LPF filters
    f32 _lpfZ;
    f32 _lpfA1;
    f32 _lpfB0;

    // early reflections
    Axfx2DelayLine _earlyDelayLine;
    uint32_t _erTapTimes[I3dl2ReverbParameter::EarlyReflectionTaps]; // 20  The actual delay lengths in samples
    f32 _earlyLevel;
    f32 _lateLevel;

    // 4 arm FDN late reverb
    uint32_t _erLateTap;          // tap time from early reflections to the late reverb
    Axfx2DelayLine _fdnDelayLines[I3dl2ReverbParameter::FeedbackDelayNetworkCount];  // 4 delay lines
    Axfx2DelayLine _apDelayLine[I3dl2ReverbParameter::FeedbackDelayNetworkCount];
    Axfx2DelayLine _ap2DelayLine[I3dl2ReverbParameter::FeedbackDelayNetworkCount];
    f32 _apCoef;
    Axfx2DelayLine _centerDelayLine;
    f32 _fdnFilterCoef[3][I3dl2ReverbParameter::FeedbackDelayNetworkCount]; // low-pass filter coefficients
    f32 _fdnFilterZ[I3dl2ReverbParameter::FeedbackDelayNetworkCount];       // shelf filter memory
    float _dryGain;
};

inline bool IsSupportedI3dl2ReverbChannelCount(int channel) NN_NOEXCEPT
{
    return (channel == 1) ||
           (channel == 2) ||
           (channel == 4) ||
           (channel == 6);
}

size_t I3DL2ReverbGetRequiredDelayLineBufferSize(int fs, int channelCount);
void InitializeI3dl2ReverbEffect(const I3dl2ReverbParameter *reverb, I3dl2ReverbState* state, void* workBuffer);
void UpdateI3dl2ReverbEffectParameter(const I3dl2ReverbParameter* reverb, I3dl2ReverbState* state, bool clearFlag);
void ApplyI3dl2ReverbEffect(const I3dl2ReverbParameter *reverb, I3dl2ReverbState* state, bool enabled, const int32_t **ppInData, int32_t **ppOutData, const uint32_t sampleCount);

} // namespace audio
} // namespace nn
