﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <limits>
#include <cstring>
#include <mutex>
#include <atomic>

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/util/util_BitUtil.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/os/os_Thread.h>
#include <nn/os/os_Event.h>
#include <nn/os/os_Mutex.h>
#include <nn/os/os_MessageQueue.h>
#include <nn/dd/dd_Types.h>
#include <nn/audio/server/audio_FirmwareDebugSettings.h>
#include <nn/audio/audio_AudioRendererApi-os.win32.h>
#include "audio_CommandProcessor.h"
#include "../common/audio_Command.h"
#include "audio_DeviceMixerManager.h"
#include "audio_Dsp.h"
#include "../common/audio_AudioRendererSession.h"
#include "audiodsp.h"
#include "audio_MailBox.h"

namespace nn { namespace audio { namespace dsp {

namespace {

MailBox g_MailBox;
nn::os::Mutex g_Mutex(false);

const int SessionCountMax = common::AudioRendererSessionCountMax;
struct CommandBufferInfo
{
    DspAddr addr;
    uint32_t size;
};

CommandBufferInfo g_CommandBufferInfoList[SessionCountMax];

nn::os::ThreadType g_MainThread;
NN_OS_ALIGNAS_THREAD_STACK int8_t g_MainStack[32768];

// FastRendering Mode
std::atomic_bool g_SkipDeviceOutput = { false };
nn::os::Event g_RenderingTriggerEvent(nn::os::EventClearMode_AutoClear);
nn::os::Event* g_pUserNotificationEvent = nullptr;
nn::os::Mutex g_FastRenderingModeMutex(false);

void MainThreadFunc(void* arg) NN_NOEXCEPT
{
    NN_UNUSED(arg);

    CommandListProcessor commandListProcessors[SessionCountMax];

    MailBox mailBox;
    mailBox.Open(NX_AUDIO_DSP_MBOXID, true);

    InitializeDeviceManager();

    NN_ABORT_UNLESS(mailBox.Recv() == NX_AUDIO_DSP_MSG_RENDER_BOOT);
    mailBox.Send(NX_AUDIO_DSP_MSG_RENDER_BOOT);
    mailBox.Send(NX_AUDIO_DSP_MSG_RENDER_END);

    for(;;)
    {
        // wait for render triger event from application
        bool FastRenderingEnabled = g_SkipDeviceOutput.load();
        if (FastRenderingEnabled)
        {
            std::lock_guard<nn::os::Mutex> lock(g_FastRenderingModeMutex);
            if (g_pUserNotificationEvent)
            {
                g_pUserNotificationEvent->Signal();
            }
        }

        while (FastRenderingEnabled &&
               g_RenderingTriggerEvent.TimedWait(nn::TimeSpan::FromMilliSeconds(100)) == false)
        {
            FastRenderingEnabled = g_SkipDeviceOutput.load();
        };

        auto message = mailBox.Recv();
        if(message == NX_AUDIO_DSP_MSG_RENDER_QUIT)
        {
            break;
        }
        else if(message == NX_AUDIO_DSP_MSG_RENDER_BEGIN)
        {
            int sampleCount;
            int index = 0;
            CommandListHeader* header = nullptr;
            while(header == nullptr && index < SessionCountMax)
            {
                header = reinterpret_cast<CommandListHeader*>(g_CommandBufferInfoList[index].addr);
                ++index;
            }

            for(int i = 0; i < SessionCountMax; ++i)
            {
                Device* pDevice = GetDevice("MainAudioOut", i); // \TBD need a way to access name used by Application
                NN_SDK_ASSERT_NOT_NULL(pDevice);

                int deviceSampleRate = GetDeviceSampleRate(pDevice);
                if(deviceSampleRate == 48000)
                {
                    sampleCount = 240;
                }
                else
                {
                    DownSamplerState* pState = GetDeviceDownsampleState(pDevice);
                    sampleCount = deviceSampleRate / 200; //first guess based on sample rate
                    // refine guess
                    ResampleOutputGetNextFrameSize(pState, pDevice->InputCountMax, 240, sampleCount, &sampleCount);
                }

                auto pHeader = reinterpret_cast<CommandListHeader*>(g_CommandBufferInfoList[i].addr);

                if(pHeader == nullptr)
                {
                    continue;
                }

                if (FastRenderingEnabled == false)
                {
                    WaitDevice(i, sampleCount);
                    BeginOutputToDevice(i, sampleCount);
                }

                const size_t size = g_CommandBufferInfoList[i].size;
                commandListProcessors[i].SetProcessTimeMax(std::numeric_limits<uint32_t>::max());
                commandListProcessors[i].Setup(pHeader, size);
                commandListProcessors[i].Process();

                if (FastRenderingEnabled == false)
                {
                    EndOutputToDevice(i, sampleCount);
                }
            }

            mailBox.Send(NX_AUDIO_DSP_MSG_RENDER_END);
        }
        else
        {
            NN_ABORT("Invalid command\n");
        }
    }

    FinalizeDeviceManager();

    mailBox.Send(NX_AUDIO_DSP_MSG_RENDER_QUIT);
    mailBox.Close();
}

} // anonymous namespace

Result Start() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(g_Mutex);

    MailBox::Initialize();
    g_MailBox.Open(NX_AUDIO_DSP_MBOXID, false);

    // TODO: This is win32-only thread, use NN_SYSTEM_THREAD_{PRIORITY,NAME} here?
    NN_RESULT_DO(nn::os::CreateThread(&g_MainThread, MainThreadFunc, nullptr, g_MainStack, sizeof(g_MainStack), nn::os::DefaultThreadPriority));
    nn::os::SetThreadName(&g_MainThread, "AudioDsp");
    nn::os::StartThread(&g_MainThread);

    g_MailBox.Send(NX_AUDIO_DSP_MSG_RENDER_BOOT);
    NN_ABORT_UNLESS(g_MailBox.Recv() == NX_AUDIO_DSP_MSG_RENDER_BOOT);

    NN_RESULT_SUCCESS;
}

void Stop() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(g_Mutex);

    g_MailBox.Send(NX_AUDIO_DSP_MSG_RENDER_QUIT);
    NN_ABORT_UNLESS(g_MailBox.Recv() == NX_AUDIO_DSP_MSG_RENDER_QUIT);

    g_MailBox.Close();
    MailBox::Finalize();

    nn::os::WaitThread(&g_MainThread);
    nn::os::DestroyThread(&g_MainThread);
}

void WaitForUnmap() NN_NOEXCEPT
{

}

void UnmapUserPointer(CpuAddr buffer, size_t size) NN_NOEXCEPT
{
    UnmapUserPointer(nn::os::InvalidNativeHandle, buffer, size);
}

void UnmapUserPointer(nn::dd::ProcessHandle processHandle, CpuAddr buffer, size_t size) NN_NOEXCEPT
{
    NN_UNUSED(processHandle);
    NN_UNUSED(buffer);
    NN_UNUSED(size);
}

void ProcessCleanup(nn::dd::ProcessHandle processHandle) NN_NOEXCEPT
{
    NN_UNUSED(processHandle);
}

DspAddr MapUserPointer(nn::dd::ProcessHandle processHandle, CpuAddr buffer, size_t size) NN_NOEXCEPT
{
    NN_UNUSED(size);
    NN_UNUSED(processHandle);

    // When using the CPU AudioRenderer, this is a no-op.
    return static_cast<DspAddr>(buffer.GetAddress());
}

DspAddr MapUserPointer(CpuAddr buffer, size_t size) NN_NOEXCEPT
{
    return MapUserPointer(nn::os::InvalidNativeHandle, buffer, size);
}

void InvalidateDspCache(DspAddr address, size_t size) NN_NOEXCEPT
{
    NN_UNUSED(address);
    NN_UNUSED(size);
}

void SendCommandBuffer(int sessionId, DspAddr command, size_t size, int renderingCycleLimit, bool isNew, uint64_t clientId)  NN_NOEXCEPT
{
    NN_UNUSED(renderingCycleLimit);
    NN_UNUSED(isNew);
    NN_UNUSED(clientId);
    NN_SDK_ASSERT_RANGE(sessionId, 0, SessionCountMax);

    auto& commandBufferInfo = g_CommandBufferInfoList[sessionId];
    commandBufferInfo.addr = command;
    commandBufferInfo.size = static_cast<uint32_t>(size);
}

void Wait() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(g_Mutex);

    NN_ABORT_UNLESS(g_MailBox.Recv() == NX_AUDIO_DSP_MSG_RENDER_END);
    memset(g_CommandBufferInfoList, 0, sizeof(g_CommandBufferInfoList));
}

int GetRemainCommandCount(int sessionId) NN_NOEXCEPT
{
    NN_SDK_ASSERT_RANGE(sessionId, 0, SessionCountMax);
    NN_UNUSED(sessionId);

    return 0;
}

void ClearRemainCommandCount(int sessionId) NN_NOEXCEPT
{
    NN_SDK_ASSERT_RANGE(sessionId, 0, SessionCountMax);
    NN_UNUSED(sessionId);
}

int64_t GetRenderingStartTick(int sessionId) NN_NOEXCEPT
{
    NN_UNUSED(sessionId);
    return nn::os::GetSystemTick().GetInt64Value();
}

void Signal() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(g_Mutex);

    g_MailBox.Send(NX_AUDIO_DSP_MSG_RENDER_BEGIN);
}

void SessionSuspend(AppletVolumeManager::SessionType type, int index) NN_NOEXCEPT
{
    NN_UNUSED(type);
    NN_UNUSED(index);
}

void SessionResume(AppletVolumeManager::SessionType type, int index) NN_NOEXCEPT
{
    NN_UNUSED(type);
    NN_UNUSED(index);
}

void SetVolume(AppletVolumeManager::SessionType type, int index, float volume, uint32_t durationUsec) NN_NOEXCEPT
{
    NN_UNUSED(type);
    NN_UNUSED(index);
    NN_UNUSED(volume);
    NN_UNUSED(durationUsec);
}

void SetRecordVolume(AppletVolumeManager::SessionType type, int index, float volume, uint32_t durationUsec) NN_NOEXCEPT
{
    NN_UNUSED(type);
    NN_UNUSED(index);
    NN_UNUSED(volume);
    NN_UNUSED(durationUsec);
}

void Wake() NN_NOEXCEPT
{
}

void Sleep() NN_NOEXCEPT
{
}

void DumpMappedPointer() NN_NOEXCEPT
{

}
} // namespace dsp

void SetFastRenderingMode(bool enabled, nn::os::Event* pEvent) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(dsp::g_FastRenderingModeMutex);
    dsp::g_SkipDeviceOutput.store(enabled);
    dsp::g_pUserNotificationEvent = (enabled) ? pEvent : nullptr;
}

bool GetFastRenderingMode() NN_NOEXCEPT
{
    return dsp::g_SkipDeviceOutput.load();
}

void TriggerRendering() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_EQUAL(GetFastRenderingMode(), true);
    dsp::g_RenderingTriggerEvent.Signal();
}

}}// namespace nn::audio

