﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <limits>
#include <cstring>
#include <mutex>

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/os/os_Event.h>
#include <nn/os/os_Mutex.h>
#include <nn/util/util_BitUtil.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/audio/detail/audio_Log.h>

#include "audio_Dsp.h"
#include "../common/audio_AudioRendererResourceControlParameter.h"
#include "../common/audio_AudioRendererSession.h"
#include "audio_DspExceptionNotifier.h"

#include <nne/audio/audio.h>
#include "audiodsp.h"
#include "audio_MailBox.h"
#include "audio_AudioHardwareConfiguration.h"

namespace nn { namespace audio { namespace dsp {
namespace {
const int SessionCountMax = common::AudioRendererSessionCountMax;

// DSP Variables
nn::os::Mutex g_DspMutex(false);
bool g_IsAdspActive = false;
bool g_IsAwake = true;
int64_t g_SignalTick;

MailBox g_MailBox;
MailBox g_RpcMailBox;

nn::os::Event     g_UnmapEvent (os::EventClearMode_ManualClear);
nn::os::Event     g_WakeEvent (os::EventClearMode_ManualClear);
nn::os::Event     g_WaitForDspSleepReadyEvent (os::EventClearMode_ManualClear);
nn::os::Mutex     g_RpcMutex (false);
nn::os::Mutex      g_PointerMapMutex (false);
nne::audio::adsp::Session * g_session = nullptr;

SharedMem_t *g_SharedMem = nullptr;
RendererDspSharedContext* g_pRendererDspSharedContext;

struct PointerMap
{
    enum class State
    {
        Free = 0,
        Mapped,
        DeferredUnmap,
    };

    uint64_t cpuAddr;
    DspAddr dspAddr;
    uint32_t size;
    PointerMap::State state;
    uint32_t referenceCount;
    nn::dd::ProcessHandle processHandle;

    void Clear() NN_NOEXCEPT
    {
        cpuAddr = 0;
        dspAddr = 0;
        size = 0;
        state = PointerMap::State::Free;
        referenceCount = 0;
        processHandle = nn::os::InvalidNativeHandle;
    }

    void Dump() NN_NOEXCEPT
    {
        NN_DETAIL_AUDIO_INFO("state:%c, cpu:%016llx, dsp:%08x, size:%08x, HandleID:%08x, refCount:%d\n",
            ConvertToCharactor(state),
            cpuAddr,
            dspAddr,
            size,
            processHandle,
            referenceCount);
    }

    char ConvertToCharactor(PointerMap::State state) const NN_NOEXCEPT
    {
        switch (state)
        {
        case PointerMap::State::Free:
            return 'F';
        case PointerMap::State::Mapped:
            return 'M';
        case PointerMap::State::DeferredUnmap:
            return 'D';
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }
};

const size_t POINTER_MAP_COUNT = 2048;
PointerMap m_PointerMap[POINTER_MAP_COUNT] = {{0, 0, 0, PointerMap::State::Free, 0, nn::os::InvalidNativeHandle}};
int g_NumPointerMapEntries = 0;

nne::audio::gmix::Session::Name SessionTypeToName(AppletVolumeManager::SessionType type) NN_NOEXCEPT
{
    switch(type)
    {
        case AppletVolumeManager::SessionType_AudioRenderer:
            return nne::audio::gmix::Session::Name::AudioRenderer;
        case AppletVolumeManager::SessionType_AudioOut:
            return nne::audio::gmix::Session::Name::AudioOut;
        case AppletVolumeManager::SessionType_AudioIn:
            return nne::audio::gmix::Session::Name::NearVoice;
        case AppletVolumeManager::SessionType_FinalOutputRecorder:
            return nne::audio::gmix::Session::Name::GameRecord;
        default:
            NN_UNEXPECTED_DEFAULT;
    }
}

void AdspRemapRpc(int32_t dspAddr, int32_t size) NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_IsAdspActive);

    g_SharedMem->rpcData.arg[0] = dspAddr;
    g_SharedMem->rpcData.arg[1] = size;

    g_RpcMailBox.Send(NX_AUDIO_DSP_RPC_MAP_ADDRESS);
    NN_ABORT_UNLESS(g_RpcMailBox.Recv() == NX_AUDIO_DSP_RPC_MAP_ADDRESS_ACK);
}

void SendAdspMapRpc() NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_IsAdspActive);

    if(g_IsAdspActive == false)
    {
        return;
    }

    int i;

    std::lock_guard<os::Mutex> lock(g_RpcMutex);
    for (i = 0; i < g_NumPointerMapEntries; i++)
    {
        if (m_PointerMap[i].dspAddr && m_PointerMap[i].state == PointerMap::State::Free)
        {
            g_SharedMem->rpcData.arg[0] = m_PointerMap[i].dspAddr;
            g_SharedMem->rpcData.arg[1] = m_PointerMap[i].size;

            g_RpcMailBox.Send(NX_AUDIO_DSP_RPC_MAP_ADDRESS);
            NN_ABORT_UNLESS(g_RpcMailBox.Recv() == NX_AUDIO_DSP_RPC_MAP_ADDRESS_ACK);

            m_PointerMap[i].state = PointerMap::State::Mapped;
        }
    }
}

void SendAdspUnmapRpc(DspAddr addr, uint32_t size) NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_IsAdspActive);

    if(g_IsAdspActive == false)
    {
        return;
    }

    std::lock_guard<os::Mutex> lock(g_RpcMutex);

    g_SharedMem->rpcData.arg[0] = addr;
    g_SharedMem->rpcData.arg[1] = size;

    g_RpcMailBox.Send(NX_AUDIO_DSP_RPC_UNMAP_ADDRESS);
    NN_ABORT_UNLESS(g_RpcMailBox.Recv() == NX_AUDIO_DSP_RPC_UNMAP_ADDRESS_ACK);
}

void DeferredUnmap() NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(g_PointerMapMutex);
    for (size_t i = 0; i < g_NumPointerMapEntries; i++)
    {
        if(m_PointerMap[i].cpuAddr != 0 && m_PointerMap[i].state == PointerMap::State::DeferredUnmap)
        {
            SendAdspUnmapRpc(m_PointerMap[i].dspAddr, m_PointerMap[i].size);
            nne::audio::iova::Unmap(m_PointerMap[i].cpuAddr, m_PointerMap[i].size, m_PointerMap[i].dspAddr);
            --g_NumPointerMapEntries;
            m_PointerMap[i] = m_PointerMap[g_NumPointerMapEntries];
            m_PointerMap[g_NumPointerMapEntries].Clear();
            --i;
        }
    }

    g_UnmapEvent.Signal();
}

void DspExceptionObserver(void* args) NN_NOEXCEPT
{
    NN_UNUSED(args);
    DumpMappedPointer();
}

void AppStartup() NN_NOEXCEPT
{
    DspExceptionNotifier::AddObserver(DspExceptionObserver);

    MailBox::Initialize();

    nne::audio::adsp::OpenSession(&g_session, "AudioRenderer");
    NN_ABORT_UNLESS(g_session != nullptr);

    g_SharedMem = (SharedMem_t*)g_session->getSharedMemory();
    g_pRendererDspSharedContext = g_SharedMem->rendererDspSharedContexts;
    memset(g_pRendererDspSharedContext, 0, sizeof(RendererDspSharedContext) * SessionCountMax);

    g_MailBox.Open(NX_AUDIO_DSP_MBOXID, false);
    g_RpcMailBox.Open(NX_AUDIO_DSP_RPC_MBOXID, false);

    g_MailBox.Send(NX_AUDIO_DSP_MSG_RENDER_BOOT);
    NN_ABORT_UNLESS(g_MailBox.Recv() == NX_AUDIO_DSP_MSG_RENDER_BOOT);
}

void AppStop() NN_NOEXCEPT
{
    g_RpcMailBox.Send(NX_AUDIO_DSP_RPC_QUIT);
    NN_ABORT_UNLESS(g_RpcMailBox.Recv() == NX_AUDIO_DSP_RPC_QUIT_ACK);

    g_MailBox.Send(NX_AUDIO_DSP_MSG_RENDER_QUIT);
    NN_ABORT_UNLESS(g_MailBox.Recv() == NX_AUDIO_DSP_MSG_RENDER_QUIT);
}

void AppFinalize() NN_NOEXCEPT
{
    nne::audio::adsp::CloseSession(g_session);

    g_MailBox.Close();
    g_RpcMailBox.Close();
    MailBox::Finalize();

    DspExceptionNotifier::RemoveObserver(DspExceptionObserver);
}

} // anonymous namespace

Result Start() NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(g_DspMutex);
    std::lock_guard<os::Mutex> lock2(g_PointerMapMutex);
    if (g_IsAdspActive)
    {
        NN_RESULT_SUCCESS;
    }

    memset(m_PointerMap, 0, sizeof(m_PointerMap));
    AppStartup();
    g_IsAdspActive = true;

    // Send any pending memory map changes.
    SendAdspMapRpc();

    NN_RESULT_SUCCESS;
}

void Stop() NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(g_DspMutex);
    if (!g_IsAdspActive)
    {
        return;
    }

    g_IsAdspActive = false;
    AppStop();

    {
        std::lock_guard<os::Mutex> lock(g_PointerMapMutex);
        // Unmap all memory from the DSP
        for (size_t i = 0; i < g_NumPointerMapEntries; i++)
        {
            if (m_PointerMap[i].cpuAddr != 0)
            {
                SendAdspUnmapRpc(m_PointerMap[i].dspAddr, m_PointerMap[i].size);
                nne::audio::iova::Unmap(m_PointerMap[i].cpuAddr, m_PointerMap[i].size, m_PointerMap[i].dspAddr);
                m_PointerMap[i].Clear();
            }
        }
        g_NumPointerMapEntries = 0;
    }

    g_UnmapEvent.Signal();
    AppFinalize();
}


void WaitForUnmap() NN_NOEXCEPT
{
    if(g_IsAdspActive)
    {
        g_UnmapEvent.Clear();
        g_UnmapEvent.Wait();
    }
}

void UnmapUserPointer(CpuAddr buffer, size_t size) NN_NOEXCEPT
{
    UnmapUserPointer(nn::dd::GetCurrentProcessHandle(), buffer, size);
}

void UnmapUserPointer(nn::dd::ProcessHandle processHandle, CpuAddr buffer, size_t size) NN_NOEXCEPT
{
    uint64_t alignedBuffer = nn::util::align_down(buffer.GetAddress(), nn::os::MemoryPageSize);
    std::lock_guard<os::Mutex> lock(g_PointerMapMutex);
    for (size_t i = 0; i < g_NumPointerMapEntries; i++)
    {
        if (processHandle == m_PointerMap[i].processHandle && m_PointerMap[i].cpuAddr == alignedBuffer)
        {
            --m_PointerMap[i].referenceCount;
            if(m_PointerMap[i].referenceCount == 0)
            {
                if(g_IsAdspActive)
                {
                    m_PointerMap[i].state = PointerMap::State::DeferredUnmap;
                }
                else
                {
                    // ADSP device session is closed so we can send unmapping requests to only sMMU.
                    nne::audio::iova::Unmap(m_PointerMap[i].cpuAddr, m_PointerMap[i].size, m_PointerMap[i].dspAddr);
                    --g_NumPointerMapEntries;
                    m_PointerMap[i] = m_PointerMap[g_NumPointerMapEntries];
                    m_PointerMap[g_NumPointerMapEntries].Clear();
                    --i;
                }
            }
        }
    }
}

void ProcessCleanup(nn::dd::ProcessHandle processHandle) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(g_PointerMapMutex);
    for (size_t i = 0; i < g_NumPointerMapEntries; i++)
    {
        if (m_PointerMap[i].processHandle == processHandle)
        {
            SendAdspUnmapRpc(m_PointerMap[i].dspAddr, m_PointerMap[i].size);
            nne::audio::iova::Unmap(m_PointerMap[i].cpuAddr, m_PointerMap[i].size, m_PointerMap[i].dspAddr);
            --g_NumPointerMapEntries;
            m_PointerMap[i] = m_PointerMap[g_NumPointerMapEntries];
            m_PointerMap[g_NumPointerMapEntries].Clear();
            --i;
        }
    }
}

DspAddr MapUserPointer(nn::dd::ProcessHandle processHandle, CpuAddr buffer, size_t size) NN_NOEXCEPT
{
    DspAddr pDsp = 0;
    uint64_t bufferAddr = buffer.GetAddress();

    std::lock_guard<os::Mutex> lock(g_PointerMapMutex);

    // If the buffer is already mapped, just return the ADSP mapping
    if (bufferAddr > 0)
    {
        size_t i = 0;
        size_t found = g_NumPointerMapEntries;
        for (; i < g_NumPointerMapEntries; i++)
        {
            if (processHandle == m_PointerMap[i].processHandle
                && (m_PointerMap[i].cpuAddr <= bufferAddr)
                && ((bufferAddr + size) <= (m_PointerMap[i].cpuAddr + m_PointerMap[i].size)))
            {
                pDsp = m_PointerMap[i].dspAddr + (bufferAddr - m_PointerMap[i].cpuAddr);
                NN_SDK_ASSERT(0 != pDsp);

                // Reset deferred unmap request
                if(m_PointerMap[i].state == PointerMap::State::DeferredUnmap)
                {
                    m_PointerMap[i].state = PointerMap::State::Mapped;
                }

                found = i;
                break;
            }
        }

        // Create a mapping (page aligned)
        if ((i == g_NumPointerMapEntries) && (0 == pDsp))
        {
            uint64_t alignedBuffer = nn::util::align_down(bufferAddr, nn::os::MemoryPageSize);
            size_t alignedSize = nn::util::align_up(size + (bufferAddr - alignedBuffer), nn::os::MemoryPageSize);

            NN_SDK_ASSERT(found < POINTER_MAP_COUNT);
            NN_SDK_ASSERT(m_PointerMap[found].cpuAddr == 0);

            uint32_t addr = nne::audio::iova::Map(processHandle, alignedBuffer, alignedSize, 0);
            if (addr == NULL)
            {
                // failed to map pointer.
                return NULL;
            }

            m_PointerMap[found].cpuAddr = alignedBuffer;
            m_PointerMap[found].dspAddr = addr;
            m_PointerMap[found].size = alignedSize;
            m_PointerMap[found].processHandle = processHandle;

            pDsp = addr + (bufferAddr - alignedBuffer);

            ++g_NumPointerMapEntries;
            SendAdspMapRpc();
        }

        ++m_PointerMap[found].referenceCount;

        NN_SDK_ASSERT(m_PointerMap[found].state == PointerMap::State::Mapped,
                      "map state failure | Entries:%d, found:%d, i:%d state:%d\n",
                      g_NumPointerMapEntries, found, i, m_PointerMap[found].state);
    }

    NN_SDK_ASSERT(buffer.IsNullPtr() || 0 != pDsp);

    return pDsp;
}

DspAddr MapUserPointer(CpuAddr buffer, size_t size) NN_NOEXCEPT
{
    return MapUserPointer(nn::dd::GetCurrentProcessHandle(), buffer, size);
}

void InvalidateDspCache(DspAddr address, size_t size) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(g_RpcMutex);

    NN_SDK_ASSERT(address != 0u);
    NN_SDK_ASSERT(size > 0);

    g_SharedMem->rpcData.arg[0] = address;
    g_SharedMem->rpcData.arg[1] = size;

    g_RpcMailBox.Send(NX_AUDIO_DSP_RPC_INVALIDATE_CACHE);
    NN_ABORT_UNLESS(g_RpcMailBox.Recv() == NX_AUDIO_DSP_RPC_INVALIDATE_CACHE_ACK);
}

void SendCommandBuffer(int sessionId, DspAddr command, size_t size, int renderingCycleLimit, bool isNew, uint64_t clientId)  NN_NOEXCEPT
{
    NN_SDK_ASSERT_RANGE(sessionId, 0, SessionCountMax);
    NN_SDK_ASSERT_MINMAX(renderingCycleLimit, 0, nn::audio::server::detail::AdspFrequency * 1000);

    g_pRendererDspSharedContext[sessionId].commandListAddr = command;
    g_pRendererDspSharedContext[sessionId].commandListSize = size;
    g_pRendererDspSharedContext[sessionId].isNew = isNew;
    g_pRendererDspSharedContext[sessionId].clientId = clientId;
    g_pRendererDspSharedContext[sessionId].avaliableProcessingTimePerAudioFrame = renderingCycleLimit;
}

void Wait() NN_NOEXCEPT
{
    NN_ABORT_UNLESS(g_MailBox.Recv() == NX_AUDIO_DSP_MSG_RENDER_END);

    DeferredUnmap();

    // Clear any old command lists
    for(int i = 0; i < SessionCountMax; ++i)
    {
        g_pRendererDspSharedContext[i].commandListAddr = 0;
        g_pRendererDspSharedContext[i].commandListSize = 0;
        g_pRendererDspSharedContext[i].isNew = false;
    }

    if( !g_IsAwake )
    {
        g_WaitForDspSleepReadyEvent.Signal();
        g_WakeEvent.Wait();

        NN_ABORT_UNLESS(g_MailBox.Recv() == NX_AUDIO_DSP_MSG_RENDER_END);
    }
}

int GetRemainCommandCount(int sessionId) NN_NOEXCEPT
{
    NN_SDK_ASSERT_RANGE(sessionId, 0, SessionCountMax);

    return g_pRendererDspSharedContext[sessionId].remainCommandCount;
}

int64_t GetRenderingStartTick(int sessionId) NN_NOEXCEPT
{
    NN_SDK_ASSERT_RANGE(sessionId, 0, SessionCountMax);

    const auto latencyUs = g_pRendererDspSharedContext[sessionId].renderingStartLatency;

    return g_SignalTick + nn::os::ConvertToTick(nn::TimeSpan::FromMicroSeconds(latencyUs)).GetInt64Value();
}

void ClearRemainCommandCount(int sessionId) NN_NOEXCEPT
{
     NN_SDK_ASSERT_RANGE(sessionId, 0, SessionCountMax);

    g_pRendererDspSharedContext[sessionId].remainCommandCount = 0;
}

void Signal() NN_NOEXCEPT
{
    g_SignalTick = nn::os::GetSystemTick().GetInt64Value();
    g_MailBox.Send(NX_AUDIO_DSP_MSG_RENDER_BEGIN);
}

void Wake() NN_NOEXCEPT
{
    bool shouldWakeup = g_IsAdspActive && !g_IsAwake;
    if( shouldWakeup )
    {
        AppStartup();
        {
            std::lock_guard<os::Mutex> lock(g_RpcMutex);
            for( int i = 0; i < g_NumPointerMapEntries; ++i )
            {
                if( m_PointerMap[i].state == PointerMap::State::Mapped )
                {
                    AdspRemapRpc(m_PointerMap[i].dspAddr, m_PointerMap[i].size);
                }
            }
        }
        g_IsAwake = true;
        g_WakeEvent.Signal();
        g_WakeEvent.Clear();
    }
    else
    {
        g_IsAwake = true;
    }
}

void Sleep() NN_NOEXCEPT
{
    bool shouldSleep = g_IsAdspActive && g_IsAwake;
    if( shouldSleep )
    {
        WaitForUnmap();
        // TODO: Workaround for SIGLO-38887
        // g_IsAwake must be set "false" after calling WaitForUnmap(),
        // otherwise the audio process sometimes fails to sleep by a deadlock between dsp::Wait() and WaitForUnmap().
        g_IsAwake = false;

        // Wait for completing AudioRenderer's rendering processing running on ADSP.
        g_WaitForDspSleepReadyEvent.Wait();
        g_WaitForDspSleepReadyEvent.Clear();

        AppStop();
        AppFinalize();
    }
    else
    {
        g_IsAwake = false;
    }
}

void DumpMappedPointer() NN_NOEXCEPT
{
    NN_DETAIL_AUDIO_INFO(" ======== Dump Adsp mapped addresses ========\n");
    NN_DETAIL_AUDIO_INFO("g_NumPointerMapEntries = %d\n", g_NumPointerMapEntries);
    for (auto i = 0; i < POINTER_MAP_COUNT; i++)
    {
        if (m_PointerMap[i].cpuAddr == 0)
        {
            continue;
        }
        NN_DETAIL_AUDIO_INFO("[%d] ", i);
        m_PointerMap[i].Dump();
    }
    NN_DETAIL_AUDIO_INFO(" ============================================\n");
}

void SetVolume(AppletVolumeManager::SessionType type, int index, float volume, uint32_t durationUsec) NN_NOEXCEPT
{
    nne::audio::gmix::Session::Name name = SessionTypeToName(type);
    nne::audio::gmix::SetSessionVolume(name, index, static_cast<int32_t>(volume * 256), durationUsec);
}

void SetRecordVolume(AppletVolumeManager::SessionType type, int index, float volume, uint32_t durationUsec) NN_NOEXCEPT
{
    nne::audio::gmix::Session::Name name = SessionTypeToName(type);
    nne::audio::gmix::SetSessionGameRecordVolume(name, index, static_cast<int32_t>(volume * 256), durationUsec);
}

void SessionSuspend(AppletVolumeManager::SessionType type, int index) NN_NOEXCEPT
{
    nne::audio::gmix::Session::Name name = SessionTypeToName(type);
    nne::audio::gmix::SuspendSession(name, index);
}

void SessionResume(AppletVolumeManager::SessionType type, int index) NN_NOEXCEPT
{
    nne::audio::gmix::Session::Name name = SessionTypeToName(type);
    nne::audio::gmix::ResumeSession(name, index);
}

}}}  // namespace nn::audio::dsp

