﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*
File:     audio_Downsampler.cpp
Block SRC
*---------------------------------------------------------------------------*/
#include "audio_DspCommon.h"

#include "audio_Downsampler.h"
#include "audio_DownsamplerFilter.h"

#include "audio_DspUtility.h"

namespace nn {
namespace audio {

#ifndef MAX
#define MAX(x, y) ((x) > (y) ? (x) : (y))
#endif
#ifndef MIN
#define MIN(x, y) ((x) < (y) ? (x) : (y))
#endif

/*! \fn    uint16_t checkBounds(DownSamplerState *pState, uint16_t arrayIndex)
*  \brief
*/
static uint16_t checkBounds(DownSamplerState *pState, uint16_t arrayIndex) NN_NOEXCEPT
{
    if (arrayIndex > pState->input_cb_upperBound) /* CHECK ON BOUNDS */
    {
        arrayIndex -= 2 * pState->nz_input; /* buffer size 2*pState->nz_input */
    }
    if (arrayIndex < pState->input_cb_lowerBound)
    {
        arrayIndex += 2 * pState->nz_input;
    }
    return arrayIndex;
}

/*! \fn    int32_t FilterUp(uint16_t Xi, uint32_t Ph, int16_t Inc, DownSamplerState *pState)
*  \brief
*/
static int32_t FilterUp(uint16_t Xi, uint32_t Ph, int16_t Inc, DownSamplerState *pState) NN_NOEXCEPT
{
    int16_t *Hp = NULL;
    uint32_t  a = 0;
    int64_t  v, t;
    uint16_t  i;
    int16_t *Imp;
    uint16_t  Nz2;

    Imp = pState->h;
    Nz2 = pState->nz_input;

    v = 0;
    Hp = &Imp[Ph >> NN_AUDIO_SRC_BL_DN_NNU];
    a = Ph & NN_AUDIO_SRC_BL_DN_NU_MASK;

    /* if (Ph == 0) TO DO: If the phase is zero... no need to do all these calculations, can save some computation */

    /* interpolation */

    for (i = 0; i < Nz2; i++)
    {
        int16_t Hdp = *(Hp + 1) - *Hp;
        t = (int64_t) * Hp;                      /* Get filter coeff */
        t += (((int64_t) Hdp) * a) >> NN_AUDIO_SRC_BL_DN_NNU;     /* t is now interp'd filter coeff */
        t *= pState->input_samples[Xi]; /* Mult coeff by input sample */
        if (t & (((int64_t)1) << (Nhxn - 1)))           /* Round, if needed */
            t += (((int64_t)1) << (Nhxn - 1));
        t >>= Nhxn;                             /* Leave some guard bits, but come back some */
        v += t;                                 /* The filter output */
        Hp += NN_AUDIO_SRC_BL_DN_L;                /* Filter coeff step */
        Xi += Inc;                              /* Input signal step. */
        Xi = checkBounds(pState, Xi);
    }

    return((int32_t)v);
}

/*! \fn    FilterDown(uint16_t Xi, uint32_t Ph, int16_t Inc, uint32_t L_step, DownSamplerState *pState)
*  \brief
*/
static int32_t FilterDown(uint16_t Xi, uint32_t Ph, int16_t Inc, uint32_t L_step, DownSamplerState *pState) NN_NOEXCEPT
{
    uint32_t  l, nu;
    int64_t  v, t;
    uint16_t  i;
    uint16_t  Nz2;
    uint32_t ratio;
    uint32_t temp;

    int16_t *Imp = pState->h;
    ratio = pState->ratioRho;
    Nz2 = pState->nz_input;

    v = 0;
    temp = (Ph * ratio);
    if (temp & (1 << (NQRATIO - 1)))            /* Round, if needed */
        temp += (1 << (NQRATIO - 1));
    Ph = temp >> NQRATIO;
    /*Ph = (Ph * (uint64_t) ratio) >> NQRATIO;*/

    /* if (Ph == 0) TO DO: If the phase is zero... no need to do all these calculations, can save some computation */

    for (i = 0; i < Nz2; i++)
    {
        l = Ph >> NN_AUDIO_SRC_BL_DN_NNU;                   /* l is how many 1/L-size steps we are between zero crossings */
        nu = Ph & NN_AUDIO_SRC_BL_DN_NU_MASK;               /* nu is how far we are between entries in the h table */
        t = (nu* (int64_t) (*(Imp + l + 1) - *(Imp + l))) >> NN_AUDIO_SRC_BL_DN_NNU;
        t += (int64_t) * (Imp + l);
        t *= pState->input_samples[Xi];
        if (t & ((int64_t)1) << (Nhxn - 1))             /* Round, if needed */
            t += ((int64_t)1) << (Nhxn - 1);
        t >>= Nhxn;                             /* Leave some guard bits, but come back some */
        v += t;                                 /* The filter output */
        Ph += L_step;
        Xi += Inc;                              /* Input signal step. */
        Xi = checkBounds(pState, Xi);
    }

    return((int32_t)v);
}

/*! \fn   SrcInitBandlimitedSrc(DownSamplerState *pState, AudioBuffer *pSrc, uint32_t ratio, int32_t *input)
*  \brief initialization for Bandlimited SRC.
*
*  Initialize block SRC configuration buffer, read input audio samples and save to the structure
*
*  \param[in]  DownSamplerState *pState: The SRC configuration
*  \param[in]  AudioBuffer    *pSrc: pointer to the input audio buffer
*  \param[in]  uint32_t ratio:       SRC ratio
*  \param[out] Return number of samples read
*/

static uint32_t InitializeState(DownSamplerState *pState, uint32_t ratio) NN_NOEXCEPT
{
    uint32_t inSamples;

    /* Clear input sample buffer before write pointer */
    memset(pState->input_samples, 0, sizeof(int32_t) * NN_AUDIO_SRC_BL_DN_CB_MID);

    /* limit the conversion ratio */
    ratio = MIN(ratio, static_cast<uint32_t>(NN_AUDIO_SRC_BL_DN_RATIO_MAX));
    ratio = MAX(ratio, static_cast<uint32_t>(NN_AUDIO_SRC_BL_DN_RATIO_MIN));

    if (ratio >= ((uint32_t)1 << NQRATIO))  // 1.0, Q"NQRATIO" format
    {
        pState->nz_input = NN_AUDIO_SRC_BL_DN_NZ;
    }
    else
    {
        pState->nz_input = (uint16_t)(((uint32_t)(NN_AUDIO_SRC_BL_DN_NZ) << NQRATIO) / ratio);
    }

    /* Set pState */
    pState->timeRegister = 0;
    pState->time_inc = (uint32_t)(((uint32_t)1 << (NN_AUDIO_SRC_BL_DN_NP + NQRATIO)) / ratio);
    pState->ratioRho = ratio;

    pState->input_current = (NN_AUDIO_SRC_BL_DN_CB_MID - 1);
    pState->input_write = (NN_AUDIO_SRC_BL_DN_CB_MID - 1);
    pState->input_cb_lowerBound = (NN_AUDIO_SRC_BL_DN_CB_MID - pState->nz_input);
    pState->input_cb_upperBound = (NN_AUDIO_SRC_BL_DN_CB_MID - 1 + pState->nz_input);

    pState->h = (int16_t *)h;           // Set filter coefficient table pointers

    /*  initialize input_buffer with silence */
    for (inSamples = 0; inSamples < pState->nz_input; inSamples++)
    {
        pState->input_samples[pState->input_write++] = 0;   //Q8 format
    }
    pState->numLeftovers = 0;
    memset(pState->leftoverSamples, 0, sizeof(pState->leftoverSamples));
    pState->isInitialized = 1;
    return inSamples;
}

/*! \fn  ResampleUpConverting(AudioBuffer *pSrc, int32_t *output, DownSamplerState *pState)
*
*  \brief  run SRC for rho >= 1
*
*  \param[in]  AudioBuffer *pSrc   Audio context buffer
*  \param[in]  DownSamplerState *pState
*  \param[out]
*  \return 0
*/
static int32_t ResampleUpConverting(
    int32_t numChannels, int32_t channel,
    const int32_t* pSrc, int16_t* pDest,
    int32_t inSamples, int32_t outSamples,
    DownSamplerState *pState) NN_NOEXCEPT
{
    int32_t index;
    int16_t* pOut = &pDest[channel];
    const int32_t* pIn = pSrc;
    /*
        now generate a frames worth of output samples
        */
    for (index = 0; index < outSamples; index++)
    {
        uint32_t P;
        int32_t inc;
        int32_t v_L;

        /* n- how many more new input samples needed */
        uint16_t n = (uint16_t)(pState->timeRegister >> NN_AUDIO_SRC_BL_DN_NP);
        pState->timeRegister += pState->time_inc;
        n = (uint16_t)(pState->timeRegister >> NN_AUDIO_SRC_BL_DN_NP) - n;

        /* if we need any originals get them now, calling function need to make sure enough samples in the input buffer
           if not, remaining locations should be set to 0 by calling function
           */
        if (0 != n)
        {
            uint16_t i;

            for (i = 0; i < n; i++)
            {
                /* Assume input from *pIn */
                int32_t input = *pIn++;
                pState->input_samples[pState->input_write++] = (input << 8); /* Q8 format */
                pState->input_write = checkBounds(pState, pState->input_write);
                inSamples++;
            }
            pState->input_current += n;
            pState->input_current = checkBounds(pState, pState->input_current);
        }

        /* apply left wing of the filter */
        P = (pState->timeRegister << NN_AUDIO_SRC_BL_DN_NN) >> NN_AUDIO_SRC_BL_DN_NN;
        inc = -1;
        v_L = FilterUp(pState->input_current, P, (int16_t)inc, pState);

        /* apply right wing of the filter */
        P = (1 << NN_AUDIO_SRC_BL_DN_NP) - P;
        inc = 1;
        v_L += FilterUp(checkBounds(pState, pState->input_current + 1), P, (int16_t)inc, pState);
        // /* write output  note channel interleave */
        *pOut = static_cast<int16_t>(Clamp<16>(static_cast<int32_t>(v_L >> (8))));
        pOut = &pOut[numChannels];
    }
    return 0;
}

/*! \fn  ResampleDownConverting(AudioBuffer *pSrc, int32_t *output, DownSamplerState *pState)
*
*  \brief  run SRC for rho < 1
*
*  \param[in]  AudioBuffer *pSrc   Audio context buffer
*  \param[in]  DownSamplerState *pState
*  \param[out] AudioBuffer *pOut  Audio context buffer updated
*  \return 0
*/
static int32_t ResampleDownConverting(
    int32_t numChannels, int32_t channel,
    const int32_t* pSrc, int16_t* pDest,
    int32_t inSamples, int32_t outSamples,
    DownSamplerState *pState) NN_NOEXCEPT
{
    int32_t index;
    int32_t used = 0;
    int16_t* pOut = &pDest[channel];
    const int32_t* pIn = pSrc;
    int32_t* pLeftover = &(pState->leftoverSamples[0]);
    /*
    now generate a frames worth of output samples
    NB: modified for specific 48K - 44K1 downsampling inline
    must consume precisely inSamples each frame time
    let outSamples float appropriately
    */
    for (index = 0; index < outSamples && used < inSamples; index++)
    {
        uint32_t P;
        int16_t inc;
        uint32_t L_step;
        int32_t v_L;
        int64_t temp;
        int32_t input = 0;
        /* n- how many more new input samples needed */
        uint16_t n = (uint16_t)(pState->timeRegister >> NN_AUDIO_SRC_BL_DN_NP);
        pState->timeRegister += pState->time_inc;
        n = (uint16_t)(pState->timeRegister >> NN_AUDIO_SRC_BL_DN_NP) - n;

        /* if we need any originals get them now, calling function need to make sure enough samples in the input buffer
           if not, remaining locations should be set to 0 by calling function
           */
        if (0 != n)
        {
            uint16_t i;

            for (i = 0; i < n && used < inSamples; i++)
            {
                if(pState->numLeftovers > 0)
                {
                    input = static_cast<int32_t>(*pLeftover++);
                    (pState->numLeftovers)--;
                }
                else
                {
                    input = *pIn++;
                    used++;
                }
                pState->input_samples[pState->input_write++] = (input << 8); /* Q8 format */
                pState->input_write = checkBounds(pState, pState->input_write);
            }
            pState->input_current += n;
            pState->input_current = checkBounds(pState, pState->input_current);
        }

        /* how far to step through filter table, expressed in same format as P */
        P = (pState->timeRegister << NN_AUDIO_SRC_BL_DN_NN) >> NN_AUDIO_SRC_BL_DN_NN;
        inc = -1;
        L_step = (uint32_t)(NN_AUDIO_SRC_BL_DN_L);
        L_step = L_step << NN_AUDIO_SRC_BL_DN_NNU;
        L_step = (L_step * pState->ratioRho) >> NQRATIO;

        v_L = FilterDown(pState->input_current, P, inc, L_step, pState);

        /* apply right wing of the filter  */
        P = ((uint32_t)(1) << NN_AUDIO_SRC_BL_DN_NP) - P;
        inc = 1;

        v_L += FilterDown(checkBounds(pState, pState->input_current + 1), P, inc, L_step, pState);

        /* write output, update t and y index, output samples in Q23.8 format */
        temp = ((int64_t)pState->ratioRho * v_L);
        if (temp & (1 << (NQRATIO - 1)))            // Round, if needed
            temp += (1 << (NQRATIO - 1));

        /* write output  note channel interleave */
        * pOut = static_cast<int16_t>(Clamp<16>(static_cast<int32_t>(temp >> (NQRATIO + 8))));
        pOut = &pOut[numChannels];

    }

    pState->numLeftovers = static_cast<int16_t>(inSamples - used);
    if(pState->numLeftovers > sizeof(pState->leftoverSamples) / sizeof(pState->leftoverSamples[0]))
    {
        pState->numLeftovers = 0;
        return -1;
    }

    for (auto k = 0; k < pState->numLeftovers; k++)
    {
        pState->leftoverSamples[k] = *pIn;
        ++pIn;
    }
    return 0;
}

/*! \fn  int32_t SrcProcessFrame(
*    DownSamplerState* pState,
*    int32_t numChannels,
*    int32_t channel,
*    int32_t* pIn,
*    int32_t* pOut,
*    int32_t inSamples,
*    int32_t outSamples,
*    int32_t* nextSamples)
*
*  \brief  run Resampler for rho < 1
*
*  \param[in]  DownSamplerState *pState
*  \param[in]  numChannels - number of interleaved channels in *pOut
*  \param[in]  channel - which channel position this resample is to occupy in the interleave
*  \param[in]  *pIn - pointer to single channel buffer of input samples
*  \param[in]  *pOut - pointer to interleaved output buffer base (ch 0)
*  \param[in]  inSamples - number of samples that must be consumed from *pIn
*  \param[in]  outSamples - number of samples that must be generated into *pOut
*  \param[out] *nextSamples - number of samples that must be generated on the next frame of inSamples size.
*              If pOut and pIn are both nullptr, only compute *nextSamples (first time up)
*  \return 0
*/
int32_t ResampleOutputFrame(
    DownSamplerState* pState,
    int32_t numChannels, int32_t channel,
    const int32_t* pIn, int16_t* pOut,
    int32_t inSamples, int32_t outSamples) NN_NOEXCEPT
{
    int32_t status = 0;

    if (0 == pState->isInitialized)
    {
        InitializeState(pState, NN_AUDIO_SRC_BL_DN_RATIO_48_441);
    }

    if (nullptr != pIn && nullptr != pOut)
    {
        if ((pState->ratioRho >= ((uint32_t)1 << NQRATIO))) /* up sampling*/
        {
            status  = ResampleUpConverting(numChannels, channel, pIn, pOut, inSamples, outSamples, pState);
        }
        else
        {
            status  = ResampleDownConverting(numChannels, channel, pIn, pOut, inSamples, outSamples, pState);
        }
    }
    return status;
}

int32_t ResampleOutputFrame(
    DownSamplerState* pState,
    int32_t numChannels, int32_t channel,
    const int32_t* pIn, int16_t* pOut,
    int32_t inSamples, int32_t outSamples, uint32_t ratio) NN_NOEXCEPT
{
    int32_t status = 0;

    if (0 == pState->isInitialized)
    {
        InitializeState(pState, ratio);
    }

    if (nullptr != pIn && nullptr != pOut)
    {
        if ((pState->ratioRho >= ((uint32_t)1 << NQRATIO))) /* up sampling*/
        {
            status  = ResampleUpConverting(numChannels, channel, pIn, pOut, inSamples, outSamples, pState);
        }
        else
        {
            status  = ResampleDownConverting(numChannels, channel, pIn, pOut, inSamples, outSamples, pState);
        }
    }
    return status;
}

int32_t ResampleOutputGetNextFrameSize(
    DownSamplerState* pState, int32_t numStates,
    int32_t inSamples, int32_t outSamples,
    int32_t* nextSamples) NN_NOEXCEPT
{
    int32_t status = 0;
    int i;
    inSamples += pState->numLeftovers;
    for (i = 0; i < numStates; i++)
    {
        if (0 != pState->isInitialized)
        {
            int32_t deltaN  = (((static_cast<int64_t>(pState->timeRegister) + static_cast<int64_t>(pState->time_inc) * outSamples) >> NN_AUDIO_SRC_BL_DN_NP) - (pState->timeRegister >> NN_AUDIO_SRC_BL_DN_NP)); // & 0x000000ff;
            while (deltaN < inSamples)
            {
                ++outSamples;
                deltaN  = (((static_cast<int64_t>(pState->timeRegister) + static_cast<int64_t>(pState->time_inc) * outSamples) >> NN_AUDIO_SRC_BL_DN_NP) - (pState->timeRegister >> NN_AUDIO_SRC_BL_DN_NP)); // & 0x000000ff;
            }
            while (deltaN > inSamples)
            {
                --outSamples;
                deltaN  = (((static_cast<int64_t>(pState->timeRegister) + static_cast<int64_t>(pState->time_inc) * outSamples) >> NN_AUDIO_SRC_BL_DN_NP) - (pState->timeRegister >> NN_AUDIO_SRC_BL_DN_NP)); // & 0x000000ff;
            }
            break;
        }
        ++pState;
    }
    *nextSamples = outSamples;

    return status;
}

}  // namespace audio
}  // namespace nn
