﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "audio_DspUtility.h" // Min
#include "audio_Qf.h"

namespace nn { namespace audio {

/*---------------------------------------------------------------------------*
FX Delay Line - 'buff' and 'end' address the first and last sample in the buffer.
'p' points to the next_out sample.

buff     end       max = 5
v        v     delay = 4    (Delay must be greater or equal to 1)
[ ][ ][ ][ ][ ]
^
p
*---------------------------------------------------------------------------*/
typedef struct
{
    uint32_t  max;   // Maximum delay
    uint32_t  delay; // Actual delay.  May be 1 through maxDelay.
    qf *buff;  // Allocated buffer
    qf *end;   // Last address (buff + delay - 1).  Must not exceed (buff + maxDelay - 1).
    qf *p;     // Read/write pointer.  Wraps when it gets to end.  Always pointing at the next out sample.
    qf  coef;  // Used by all pass filter
} FXDelayLine;

inline uint8_t DelayLineInit(FXDelayLine *delay, uint32_t maxDelay, qf* buffer) NN_NOEXCEPT
{
    delay->delay = delay->max = maxDelay;
    delay->buff = buffer;
    delay->end = (buffer + maxDelay - 1);
    delay->p = buffer;
    delay->coef = QF_CONST(1.0f);

    return 1;
}

inline uint32_t DelayLineGetDelay(FXDelayLine *delay) NN_NOEXCEPT
{
    return delay->delay;
}

inline uint32_t DelayLineGetMaxDelay(FXDelayLine *delay) NN_NOEXCEPT
{
    return delay->max;
}

inline qf DelayLineTapOut(FXDelayLine *delay, uint32_t rindex) NN_NOEXCEPT
{
    // the tap out delay should be limited to one sample shorter than the delay line circular buffer length
    // otherwise the tap out pointer wraps around and the actual delay is the remainder after the modulo
    if (rindex >= delay->delay)
    {
        rindex = delay->delay - 1;
    }
    qf *p = delay->p - (1 + rindex);

    if (p < delay->buff)
    {
        p += delay->delay;
    }

    return *p;
}

inline qf DelayLineTapOut(const qf *in, const qf *base, int32_t indexBack, const uint32_t delay) NN_NOEXCEPT
{
    const qf *p = in - indexBack; //max_delay + 1
    if(p < base)
    {
        p += delay;
    }
    return *p;
}

// Get the current out sample of the delay line
inline qf DelayLineNextOut(FXDelayLine *delayLine) NN_NOEXCEPT
{
    return *(delayLine->p);
}

// Calculate delay in terms of how many samples
inline int32_t DelayLineCalculateDelaySamples(qf khz, qf msec) NN_NOEXCEPT
{
    return QfToS32(MultiQfQf(khz, msec));
}

inline uint8_t DelayLineSetDelay(FXDelayLine *delayLine, uint32_t newDelay) NN_NOEXCEPT
{
    delayLine->delay = Min(newDelay, delayLine->max);

    // Reset end and current pointer
    delayLine->end = delayLine->buff + delayLine->delay - 1;
    delayLine->p = delayLine->buff;

    return 1;
}

// clear the delay line buffer
inline void DelayLineClear(FXDelayLine *delayLine) NN_NOEXCEPT
{
    memset(delayLine->buff, 0, sizeof(delayLine->buff[0]) * delayLine->delay);
}

// read out the current sample and queue the input sample into the delay buffer
inline qf DelayLineTick(FXDelayLine *delayLine, qf in) NN_NOEXCEPT
{
    qf out = *(delayLine->p);
    *(delayLine->p) = in;

    if (delayLine->end < ++delayLine->p)
    {
        delayLine->p = delayLine->buff;
    }

    return out;
}

// AllPass
// --------------------------------------------------------------------------------
inline uint8_t AllPassInit(FXDelayLine *allpass, uint32_t maxDelay, qf coef, qf* buffer) NN_NOEXCEPT
{
    uint8_t result;

    result = DelayLineInit(allpass, maxDelay, buffer);

    if (result)
    {
        allpass->coef = coef;
    }

    return result;
}

//  Name:      _AllPassTick
//  Description:
//             +------->------| coef >--->---+
//             |                             |
//  in >--(+)--+->-[////////////////]->-*->-(+)----> out
//         |                            |
//         +-----<----< -coef |----<----+

inline qf AllPassTick(FXDelayLine *allpass, qf in) NN_NOEXCEPT
{
    qf temp, out;
    qf coef = allpass->coef;

    temp = in - MultiQfQf(coef, DelayLineNextOut(allpass));
    out = MultiQfQf(coef, temp) + DelayLineTick(allpass, temp);

    return out;
}

inline uint32_t AllPassGetDelay(FXDelayLine *allpass) NN_NOEXCEPT
{
    return DelayLineGetDelay(allpass);
}

inline void AllPassSetCoef(FXDelayLine *allpass, qf coef) NN_NOEXCEPT
{
    allpass->coef = coef;
}

}} // namespace nn::audio
