﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "audio_DspCommon.h"

namespace nn {
namespace audio {
namespace dsp {

#ifdef NN_AUDIO_ENABLE_DEBUG_CACHE

enum CachedBufferFlags
{
    //Warning if this buffer is not flushed before Adsp frame ends
    CachedBuffer_RequireFlush = 1,

    //Warning if this buffer was not invalidated before the CachedBuffer was created
    CachedBuffer_RequireInval = 2,

    //Buffer is never used by CPU
    CachedBuffer_LocalBuffer   = 4,

    //Uncached buffers do not require any cache operations
    CachedBuffer_Uncached     = 8,
};

void FlushDebugCacheHistory(const void* addr, size_t size) NN_NOEXCEPT;
void InvalidateDebugCacheHistory(const void* addr, size_t size) NN_NOEXCEPT;
void ClearDebugCacheHistory(void) NN_NOEXCEPT;
void ValidateDebugCacheOperations(void) NN_NOEXCEPT;
void InsertDebugCacheBuffer(const void* addr, size_t size, int flags, const char* function, int line) NN_NOEXCEPT;

#define _NN_AUDIO_FLUSH_BUFFER(addr, size) ::nn::audio::dsp::FlushDebugCacheHistory(addr, size)
#define _NN_AUDIO_INVAL_BUFFER(addr, size) ::nn::audio::dsp::InvalidateDebugCacheHistory(addr, size)
#define NN_AUDIO_TRACK_BUFFER(addr, size, flags) ::nn::audio::dsp::InsertDebugCacheBuffer(addr, size, flags, __FILE__, __LINE__)
#define NN_AUDIO_DEBUG_CACHE_BEGIN() ::nn::audio::dsp::ClearDebugCacheHistory()
#define NN_AUDIO_DEBUG_CACHE_END()   ::nn::audio::dsp::ValidateDebugCacheOperations()

#else //NN_AUDIO_ENABLE_DEBUG_CACHE

#define _NN_AUDIO_FLUSH_BUFFER(addr, size)
#define _NN_AUDIO_INVAL_BUFFER(addr, size)
#define NN_AUDIO_TRACK_BUFFER(addr, size, flags)
#define NN_AUDIO_DEBUG_CACHE_BEGIN()
#define NN_AUDIO_DEBUG_CACHE_END()

#endif //NN_AUDIO_ENABLE_DEBUG_CACHE

}}} // namespace nn::audio::dsp
