﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "audio_CacheDebug.h"

#ifdef NN_AUDIO_ENABLE_DEBUG_CACHE
namespace nn {
namespace audio {
namespace dsp {

namespace{
//Track history of cache flush/invalidate calls
struct CacheHistory
{
    uint32_t addr;
    uint32_t size;
    int flushCount;
    int invalCount;
};

//Track buffers created by NN_AUDIO_TRACK_BUFFER
struct CachedBuffer
{
    uint32_t addr;
    size_t size;
    int flags;
    int line;
    const char* file;
    bool flushRequired;
};

//print warning buffer is flushed more than MaxFlushCount (or invalidated more than MaxInvalCount)
const int MaxFlushCount = 1;
const int MaxInvalCount = 1;

int g_HistoryIdx = 0;
const int CacheHistoryCount = 512;
CacheHistory g_CacheHistory[CacheHistoryCount];

const int CachedBufferCount = 512;
int g_BufferIdx = 0;
CachedBuffer g_UserBuffer[CachedBufferCount];

static int FindUserBuffer(uint32_t addr, uint32_t size) NN_NOEXCEPT;

static int FindHistoryBuffer(uint32_t addr, uint32_t size) NN_NOEXCEPT
{
    enter_critical_section();
    for (int i = 0; i < g_HistoryIdx; ++i)
    {
        auto h = &g_CacheHistory[i];
        auto end = addr + size;
        if (addr == h->addr && end == (h->addr + h->size))
        {
            exit_critical_section();
            return i;
        }
    }
    exit_critical_section();
    return g_HistoryIdx;
}

static void ClearCacheHistory() NN_NOEXCEPT
{
    enter_critical_section();
    g_HistoryIdx = 0;
    memset(g_CacheHistory, 0, sizeof(CacheHistory) * CacheHistoryCount);
    exit_critical_section();
}

static void ValidateSingleBufferHistory() NN_NOEXCEPT
{
    for (int i = 0; i < g_HistoryIdx; ++i)
    {
        auto h = &g_CacheHistory[i];
        if (h->invalCount > MaxInvalCount || h->flushCount > MaxFlushCount)
        {
            printf("0x%x-0x%x (size:0x%x) inval:%d flush:%d\n", h->addr, h->addr + h->size, h->size, h->invalCount, h->flushCount);
            auto bufferIdx = FindUserBuffer(h->addr, h->size);
            if(bufferIdx != g_BufferIdx)
            {
                auto b = &g_UserBuffer[bufferIdx];
                printf("    %s:%d\n", b->file, b->line);
            }
        }
    }
}

static bool IsOverlapped(uint32_t addr1, size_t size1, uint32_t addr2, uint32_t size2)
{
    auto end1 = addr1 + size1;
    auto end2 = addr2 + size2;

    return (addr1 >= addr2 && addr1 < end2) ||
           (addr2 >= addr1 && addr2 < end1);
}

static void ValidateOverlapBufferHistory() NN_NOEXCEPT
{
    for (int i = 0; i < g_HistoryIdx; ++i)
    {
        for(int j = i + 1; j < g_HistoryIdx; ++j)
        {
            auto h1 = &g_CacheHistory[i];
            auto h2 = &g_CacheHistory[j];
            int invalCount = h1->invalCount + h2->invalCount;
            int flushCount = h1->flushCount + h2->flushCount;
            if((h1->invalCount == 0 && h1->flushCount == 0) ||
               (h2->invalCount == 0 && h2->flushCount == 0) ||
               !IsOverlapped(h1->addr, h1->size, h2->addr, h2->size))
            {
                continue;
            }
            if (invalCount > MaxInvalCount || flushCount > MaxFlushCount)
            {
                printf("1:0x%x-0x%x (size:0x%x) inval:%d flush:%d\n", h1->addr, h1->addr + h1->size, h1->size, h1->invalCount, h1->flushCount);
                printf("2:0x%x-0x%x (size:0x%x) inval:%d flush:%d\n", h2->addr, h2->addr + h2->size, h2->size, h2->invalCount, h2->flushCount);
                auto bufferIdx = FindUserBuffer(h1->addr, h1->size);
                if(bufferIdx != g_BufferIdx)
                {
                    auto b = &g_UserBuffer[bufferIdx];
                    printf("1:    %s:%d\n", b->file, b->line);
                }
                bufferIdx = FindUserBuffer(h2->addr, h2->size);
                if(bufferIdx != g_BufferIdx)
                {
                    auto b = &g_UserBuffer[bufferIdx];
                    printf("2:    %s:%d\n", b->file, b->line);
                }
            }
        }
    }
}

static void ValidateCacheHistory() NN_NOEXCEPT
{
    enter_critical_section();
    //Check single buffers for duplicate cache operations
    ValidateSingleBufferHistory();

    ValidateOverlapBufferHistory();

    exit_critical_section();
}

static bool IsCacheHistoryInvalidated(uint32_t addr, size_t size) NN_NOEXCEPT
{
    bool invalidated = false;
    enter_critical_section();
    for (int i = 0; i < g_HistoryIdx; ++i)
    {
        auto h = &g_CacheHistory[i];
        auto end = addr + size;
        if (addr >= h->addr && end <= (h->addr + h->size))
        {
            if(h->invalCount > 0)
            {
                invalidated = true;
                break;
            }
        }
    }
    exit_critical_section();

    return invalidated;
}

static bool IsCacheHistoryFlushed(uint32_t addr, size_t size) NN_NOEXCEPT
{
    bool flushed = false;
    enter_critical_section();
    for (int i = 0; i < g_HistoryIdx; ++i)
    {
        auto h = &g_CacheHistory[i];
        auto end = addr + size;
        if (addr >= h->addr && end <= (h->addr + h->size))
        {
            if(h->flushCount > 0)
            {
                flushed = true;
                break;
            }
        }
    }
    exit_critical_section();

    return flushed;
}

static int InsertCacheHistory(uint32_t addr, uint32_t size) NN_NOEXCEPT
{
    enter_critical_section();
    if (g_HistoryIdx == CacheHistoryCount)
    {
        exit_critical_section();
        return -1;
    }
    auto h = &g_CacheHistory[g_HistoryIdx];
    h->addr = addr;
    h->size = size;
    ++g_HistoryIdx;
    exit_critical_section();
    return g_HistoryIdx - 1;
}

static void ClearUserBuffer() NN_NOEXCEPT
{
    enter_critical_section();
    g_BufferIdx = 0;
    memset(g_UserBuffer, 0, sizeof(CachedBuffer) * CachedBufferCount);
    exit_critical_section();
}

static int FindUserBuffer(uint32_t addr, uint32_t size) NN_NOEXCEPT
{
    enter_critical_section();
    for (int i = 0; i < g_BufferIdx; ++i)
    {
        auto b = &g_UserBuffer[i];
        auto end = addr + size;
        if (addr == b->addr && end == (b->addr + b->size))
        {
            exit_critical_section();
            return i;
        }
    }
    exit_critical_section();
    return g_BufferIdx;
}

static void ValidateUserBuffer() NN_NOEXCEPT
{
    enter_critical_section();
    //Missing invalidate checked when buffers are inserted
    for (int i = 0; i < g_BufferIdx; ++i)
    {
        auto& buf = g_UserBuffer[i];
        if (buf.flushRequired)
        {
            printf("no flush: 0x%x, 0x%x (created at %s:%d)\n", buf.addr, buf.size, buf.file, buf.line);
        }
    }
    exit_critical_section();
}

static void CheckUncoveredOperations() NN_NOEXCEPT
{
    int buffersFound = 0;
    //In history but not UserBuffer
    for (int historyIdx = 0; historyIdx < g_HistoryIdx; ++historyIdx)
    {
        auto history = &g_CacheHistory[historyIdx];
        auto bufferIdx = FindUserBuffer(history->addr, history->size);

        if (bufferIdx == g_BufferIdx)
        {
            printf("untracked buffer: 0x%x, 0x%x (inval:%d, flush:%d)\n", history->addr, history->size, history->invalCount, history->flushCount);
        }
        else
        {
            auto buffer = &g_UserBuffer[bufferIdx];
            bool unnecessaryInval = !(buffer->flags & CachedBuffer_RequireInval) && history->invalCount > 0;
            bool unnecessaryFlush = !(buffer->flags & CachedBuffer_RequireFlush) && history->flushCount > 0;

            if(unnecessaryFlush || unnecessaryInval)
            {
                printf("unnecessary operation: 0x%x, 0x%x (inval:%d, flush:%d)\n", history->addr, history->size, unnecessaryInval, unnecessaryFlush);
                printf("    %s:%d\n", buffer->file, buffer->line);
            }
            ++buffersFound;
        }
    }
    if(buffersFound != g_HistoryIdx)
    {
        printf("coverage = %d out of %d\n", buffersFound, g_HistoryIdx);
    }
}

} //anonymous namespace

void InvalidateDebugCacheHistory(const void* pAddr, size_t size) NN_NOEXCEPT
{
    uint32_t addr = (uint32_t)pAddr;
    enter_critical_section();
    int i = FindHistoryBuffer(addr, size);
    if (i == g_HistoryIdx)
    {
        i = InsertCacheHistory(addr, size);
        if (i != g_HistoryIdx)
        {
            ++g_CacheHistory[i].invalCount;
        }
        else
        {
            printf("History buffer full\n");
        }
    }
    else
    {
        ++g_CacheHistory[i].invalCount;
    }
    exit_critical_section();
}

void FlushDebugCacheHistory(const void* pAddr, size_t size) NN_NOEXCEPT
{
    uint32_t addr = (uint32_t)pAddr;
    enter_critical_section();
    int i = FindHistoryBuffer(addr, size);
    if (i == g_HistoryIdx)
    {
        i = InsertCacheHistory(addr, size);
        //arm_mmu_map_mem(addr, addr, size, MMU_FLAG_READWRITE | MMU_FLAG_CACHED_WRITE_BACK_ALLOCATE | MMU_FLAG_EXECUTE_NEVER);
    }
    if (i != g_HistoryIdx)
    {
        ++g_CacheHistory[i].flushCount;
        auto index = FindUserBuffer((uint32_t)addr, size);
        if(index != g_BufferIdx)
        {
            auto b = &g_UserBuffer[index];
            b->flushRequired = false;
        }
    }
    else
    {
        printf("History buffer full\n");
    }
    exit_critical_section();
}

void InsertDebugCacheBuffer(const void* addr, size_t size, int flags, const char* file, int line) NN_NOEXCEPT
{
    if (g_BufferIdx == CachedBufferCount)
    {
        printf("Cached buffer full\n");
        return;
    }
    enter_critical_section();
    auto index = FindUserBuffer((uint32_t)addr, size);
    auto b = &g_UserBuffer[index];
    if(index == g_BufferIdx)
    {
        index = g_BufferIdx;
        b->addr = (uint32_t)addr;
        b->size = size;
        b->flags = flags;
        b->file = file;
        b->line = line;
        ++g_BufferIdx;
    }
    else
    {
        b->flags |= flags;
    }
    if((b->flags & CachedBuffer_RequireInval) && !IsCacheHistoryInvalidated((uint32_t)addr, size))
    {
        printf("no inval: 0x%x, 0x%x (created at %s:%d)\n", b->addr, b->addr + b->size, b->file, b->line);
    }
    b->flushRequired = b->flags & CachedBuffer_RequireFlush;

    exit_critical_section();
}

void ClearDebugCacheHistory() NN_NOEXCEPT
{
    enter_critical_section();
    ClearCacheHistory();
    ClearUserBuffer();
    exit_critical_section();
}

void ValidateDebugCacheOperations() NN_NOEXCEPT
{
    enter_critical_section();
    //Reports any buffers that invalidate or flush more than once per frame
    ValidateCacheHistory();

    //Reports any buffers that set CachedBuffer_RequireFlush but was not flushed
    //Buffers that set CachedBuffer_RequireInval were checked when NN_AUDIO_TRACK_BUFFER was called.
    ValidateUserBuffer();

    //Reports any buffers that were flushed or invalidated, but were not tracked with NN_AUDIO_TRACK_BUFFER
    CheckUncoveredOperations();

    exit_critical_section();
}

}}} // namespace nn::audio::dsp

#endif // NN_AUDIO_ENABLE_DEBUG_CACHE
