﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstdint>
#include <limits>
#include <type_traits>
#include <nn/nn_Macro.h>
#include <nn/util/util_BytePtr.h>

#include "audio_Util.h"

namespace nn { namespace audio { namespace common {

typedef uint32_t RevisionInfo;
RevisionInfo GetCurrentRevision() NN_NOEXCEPT;
bool CheckValidRevision(RevisionInfo rev) NN_NOEXCEPT;
using BranchTag = RevisionInfo(*)();
bool CheckFeatureSupported(BranchTag tag, RevisionInfo rev) NN_NOEXCEPT;
struct SupportTags
{
private:
    SupportTags() {};
public:
    static const BranchTag Latest;
    static const BranchTag VoicePitchAndSrcSkipped;
    static const BranchTag VoicePlayedSampleCountResetAtLoopPoint;
    static const BranchTag SplitterBugFix; // See SIGLO-80635
    static const BranchTag FlushVoiceWaveBuffers;
    static const BranchTag ElapsedFrameCount;
    static const BranchTag CommandProcessingTimeEstimatorVersion2;
    static const BranchTag AudioRendererVariadicCommandBufferSize;
    static const BranchTag PerformanceMetricsDataFormatVersion2;
    static const BranchTag AudioRendererProcessingTimeLimit70Percent;
    static const BranchTag AudioRendererProcessingTimeLimit75Percent;
    static const BranchTag AudioRendererProcessingTimeLimit80Percent;
    static const BranchTag AudioUsbDeviceOutput;
    static const BranchTag AdpcmLoopContextBugFix; // See SIGLO-46249
    static const BranchTag Splitter;
    static const BranchTag LongSizePreDelay;
    static const BranchTag InitialRelease;
};

int GetRevisionNumber(RevisionInfo revision) NN_NOEXCEPT;
RevisionInfo GetInitialReleaseRevision() NN_NOEXCEPT;

struct  UpdateDataSizes
{
    uint32_t behavior;
    uint32_t memoryPool;
    uint32_t voice;
    uint32_t voiceChannelResource;
    uint32_t effect;
    uint32_t mix;
    uint32_t sink;
    uint32_t performance;
    uint32_t splitter;
    uint32_t rendererInfo;
    uint32_t reserved[4];
};

class NN_AUDIO_INFOTYPE_FILED_ALIGN UpdateDataHeader
{
private:
    RevisionInfo m_Revision;
    UpdateDataSizes m_InfoSizes;
    uint32_t m_TotalInfoSize;

public:
    void Initialize(RevisionInfo revision) NN_NOEXCEPT;
    RevisionInfo GetRevision() const NN_NOEXCEPT;
    size_t GetTotalSize() const NN_NOEXCEPT;
    void* GetCurrentPtr(uintptr_t base) const NN_NOEXCEPT;

    template<typename Field>
    void SetSize(Field f, size_t value) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(value <= std::numeric_limits<uint32_t>::max());
        this->m_InfoSizes.*f = static_cast<uint32_t>(value);
        this->m_TotalInfoSize += static_cast<uint32_t>(value);
    }
    template<typename Field>
    bool CheckSize(Field f, size_t value) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(value <= std::numeric_limits<uint32_t>::max());
        return this->m_InfoSizes.*f == static_cast<uint32_t>(value);
    }
    template<typename Field>
    size_t GetSize(Field f) const NN_NOEXCEPT
    {
        return this->m_InfoSizes.*f;
    }
};

NN_AUDIO_INFOTYPE_CHECK(UpdateDataHeader, 64);
NN_STATIC_ASSERT(std::is_standard_layout<UpdateDataHeader>::value);
NN_STATIC_ASSERT(std::is_trivial<UpdateDataHeader>::value);

}}}  // namespace nn::audio::common
