﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "audio_SortLogic.h"

namespace nn { namespace audio { namespace common {

EdgeMatrix::EdgeMatrix() NN_NOEXCEPT
    : m_NodeCount(0)
{
}

size_t EdgeMatrix::GetWorkBufferSize(int nodeCount) NN_NOEXCEPT
{
    return nn::util::BitArray::CalculateWorkMemorySize(nodeCount * nodeCount);
}

void EdgeMatrix::Initialize(void* workMemory, size_t workMemorySize, int nodeCount) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(workMemory);
    NN_SDK_ASSERT_GREATER_EQUAL(workMemorySize, GetWorkBufferSize(nodeCount));
    m_NodeCount = nodeCount;
    ResetWorkMemory(workMemory, workMemorySize, nodeCount * nodeCount);
    reset();
}

bool EdgeMatrix::Connected(int src, int dst) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_LESS(src, m_NodeCount);
    NN_SDK_ASSERT_LESS(dst, m_NodeCount);
    return test(src * m_NodeCount + dst);
}

void EdgeMatrix::Connect(int src, int dst) NN_NOEXCEPT
{
    NN_SDK_ASSERT_LESS(src, m_NodeCount);
    NN_SDK_ASSERT_LESS(dst, m_NodeCount);
    set(src* m_NodeCount + dst);
}

void EdgeMatrix::Disconnect(int src, int dst) NN_NOEXCEPT
{
    NN_SDK_ASSERT_LESS(src, m_NodeCount);
    NN_SDK_ASSERT_LESS(dst, m_NodeCount);
    reset(src* m_NodeCount + dst);
}

void EdgeMatrix::RemoveEdges(int src) NN_NOEXCEPT
{
    NN_SDK_ASSERT_LESS(src, m_NodeCount);
    for (auto i = 0; i < m_NodeCount; ++i)
    {
        Disconnect(src, i);
    }
}

int EdgeMatrix::GetNodeCount() const NN_NOEXCEPT
{
    return m_NodeCount;
}



size_t NodeStates::Stack::CalcBufferSize(int count) NN_NOEXCEPT
{
    return nn::util::PlacementArray<int>::CalculateWorkMemorySize(count);
}

NodeStates::Stack::Stack()  NN_NOEXCEPT
    : index(0)
    , countMax(0)
{}

void NodeStates::Stack::Reset(int* buffer, size_t bufferSize, int count) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(buffer);
    NN_SDK_ASSERT_GREATER_EQUAL(bufferSize, CalcBufferSize(count));
    ResetWorkMemory(buffer, bufferSize, count);
    index = 0;
    countMax = count;
}

int NodeStates::Stack::Count() const NN_NOEXCEPT
{
    return index;
}

void NodeStates::Stack::push(int val) NN_NOEXCEPT
{
    NN_SDK_ASSERT_LESS(index + 1, countMax);
    operator[](index) = val;
    ++index;
}

int NodeStates::Stack::top() const NN_NOEXCEPT
{
    NN_SDK_ASSERT_GREATER(index, 0);
    return operator[](index - 1);
}

int NodeStates::Stack::pop() NN_NOEXCEPT
{
    NN_SDK_ASSERT_GREATER(index, 0);
    --index;
    return operator[](index);
}


size_t NodeStates::GetWorkBufferSize(int nodeCount) NN_NOEXCEPT
{
    return nn::util::BitArray::CalculateWorkMemorySize(nodeCount) * 2
        + nn::util::PlacementArray<int>::CalculateWorkMemorySize(nodeCount) * 3
        + Stack::CalcBufferSize(nodeCount * nodeCount);
}

NodeStates::NodeStates() NN_NOEXCEPT
    : m_Count(0)
    , m_SortPosition(0)
{}

void NodeStates::Initialize(void* buffer, size_t bufferSize, int nodeCount) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(buffer);
    NN_SDK_ASSERT_GREATER_EQUAL(bufferSize, GetWorkBufferSize(nodeCount));
    NN_UNUSED(bufferSize);

    m_Count = nodeCount;

    auto ptr = nn::util::BytePtr(buffer);
    auto size = nn::util::BitArray::CalculateWorkMemorySize(nodeCount);
    m_Discovered.ResetWorkMemory(ptr.Get(), size, nodeCount);
    m_Discovered.reset();
    ptr += size;
    m_Finished.ResetWorkMemory(ptr.Get(), size, nodeCount);
    m_Finished.reset();
    ptr += size;

    size = nn::util::PlacementArray<int>::CalculateWorkMemorySize(nodeCount);
    m_SortedIndex.ResetWorkMemory(ptr.Get(), size, nodeCount);
    ptr += size;

    size += Stack::CalcBufferSize(nodeCount * nodeCount);
    m_WorkStack.Reset(ptr.Get<int>(), size, nodeCount * nodeCount);
    ptr += size;
}

bool NodeStates::Tsort(EdgeMatrix& edges)
{
    NN_SDK_ASSERT_EQUAL(GetNodeCount(), edges.GetNodeCount());
    return DepthFirstSearch(*this, edges, m_WorkStack);
}

NodeStates::SearchState NodeStates::GetState(int index) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_LESS(index, GetNodeCount());
    if (m_Discovered.test(index))
    {
        NN_SDK_ASSERT(m_Finished.test(index) == false);
        return SearchState::Discovered;
    }
    else if (m_Finished.test(index))
    {
        NN_SDK_ASSERT(m_Discovered.test(index) == false);
        return SearchState::Finished;
    }
    else
    {
        return SearchState::Initial;
    }
}

void NodeStates::PushTsortResult(int index) NN_NOEXCEPT
{
    NN_SDK_ASSERT_LESS(index, GetNodeCount());
    m_SortedIndex[m_SortPosition++] = index;
}

void NodeStates::SetState(int index, SearchState state) NN_NOEXCEPT
{
    NN_SDK_ASSERT_LESS(index, GetNodeCount());
    switch (state)
    {
    case NodeStates::SearchState::Initial:
        m_Discovered.reset(index);
        m_Finished.reset(index);
        break;
    case NodeStates::SearchState::Discovered:
        m_Discovered.set(index);
        m_Finished.reset(index);
        break;
    case NodeStates::SearchState::Finished:
        m_Discovered.reset(index);
        m_Finished.set(index);
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

void NodeStates::ResetState() NN_NOEXCEPT
{
    m_Discovered.reset();
    m_Finished.reset();
    m_SortPosition = 0;
    for (auto itr = m_SortedIndex.begin(); itr != m_SortedIndex.end(); ++itr)
    {
        *itr = -1;
    }
}

std::reverse_iterator<const int *> NodeStates::ResultBegin() const NN_NOEXCEPT
{
    return m_SortedIndex.crbegin();
}

std::reverse_iterator<const int *> NodeStates::ResultEnd() const NN_NOEXCEPT
{
    return m_SortedIndex.crend();
}

int NodeStates::GetNodeCount() const NN_NOEXCEPT
{
    return m_Count;
}

bool NodeStates::DepthFirstSearch( NodeStates& states, EdgeMatrix& edges, NodeStates::Stack& workStack) NN_NOEXCEPT
{
    states.ResetState();

    for (auto i = 0; i < states.GetNodeCount(); ++i)
    {
        if (states.GetState(i) == NodeStates::SearchState::Initial)
        {
            workStack.push(i);
        }

        while (workStack.Count() > 0)
        {
            const auto idx = workStack.top();
            if (states.GetState(idx) == NodeStates::SearchState::Initial)
            {
                states.SetState(idx, NodeStates::SearchState::Discovered);
            }
            else if (states.GetState(idx) == NodeStates::SearchState::Discovered)
            {
                states.SetState(idx, NodeStates::SearchState::Finished);
                states.PushTsortResult(idx);
                workStack.pop();
                continue;
            }
            else if (states.GetState(idx) == NodeStates::SearchState::Finished)
            {
                workStack.pop();
                continue;
            }

            for (auto j = 0; j < edges.GetNodeCount(); ++j)
            {
                if (edges.Connected(idx, j) == false)
                {
                    continue; // not connected
                }

                if (states.GetState(j) == NodeStates::SearchState::Initial)
                {
                    workStack.push(j);
                }
                else if (states.GetState(j) == NodeStates::SearchState::Discovered)
                {
                    // backward edge detected"
                    states.ResetState();
                    return false;
                }
            }
        }
    }
    return true;
}

}}} // namespace nn::audio::common

