﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <type_traits>
#include <bitset>

#include <nn/nn_Macro.h>
#include "audio_BuildDefinition.h"

namespace nn { namespace audio { namespace common {

// Dynamically allocatable bit-flag set.
// "std::bitset" compatible API prepared (partially).
//
// note:
// "StorageType" must be "unsigned" type
//
template<typename StorageType = uint32_t>
class BitArray
{
    // check unsigned
    NN_STATIC_ASSERT(std::is_unsigned<StorageType>::value);
    NN_DISALLOW_COPY(BitArray);

public: // static func
    /**
     * Calc required buffer size for Initialize()
     *
     * Please note that return value is round-up value by StorageType bit width.
     * i.e. BitArray<uint64_t>::CaclBufferSize(1) -> 8
     **/
    static size_t CalcBufferSize(uint32_t requiredBitCount) NN_NOEXCEPT
    {
        size_t size = 0;
        size += (requiredBitCount / BitCountPerStorage) * sizeof(StorageType);
        size += (requiredBitCount % BitCountPerStorage) > 0 ? sizeof(StorageType) : 0;
        return size;
    }

public:
    BitArray() NN_NOEXCEPT
        : m_Storage(nullptr)
        , m_StorageCount(0)
    {}

    virtual ~BitArray() NN_NOEXCEPT
    {
        Finalize();
    }

    /**
     * Initialize BitArray
     *
     * Please note instance is initialized as using maxmum bit width.
     * In the case bellows, bitArray instance accept 64 bit width.
     *
     * // required width is 8
     * int requiredBitWidth = 8;
     *
     * // but bufSize is rounded up with uint64_t bit width, so bufSize -> 8
     * size_t bufSize = BitArray<uint64_t>::CaclBufferSize(requiredBitWidth);
     * uint8_t buf = new uint8_t[bufSize];
     *
     * // BitArray uses max width of buf & bufSize. so bitArray uses 8 byte = 64 bit width.
     * BitArray<uint64_t> bitArray(buf, bufSize)
     *
     **/
    void Initialize(void* buffer, size_t bufferSize) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(buffer != nullptr);
        NN_SDK_ASSERT(bufferSize > 0);

        memset(buffer, 0, bufferSize);

        m_Storage = reinterpret_cast<StorageType*>(buffer);
        m_StorageCount = static_cast<uint32_t>(bufferSize / sizeof(StorageType));
    }

    void* Finalize() NN_NOEXCEPT
    {
        void* buf = m_Storage;
        m_Storage = nullptr;
        m_StorageCount = 0;
        return buf;
    }

    /**
     * Check if "pos" is true, index origin is zero.
     **/
    bool Test(uint32_t pos) const NN_NOEXCEPT
    {
        NN_SDK_ASSERT(pos < m_StorageCount * BitCountPerStorage);

        return (StorageOf_(pos) & BitMaskOf_(pos)) != 0u;
    }

    /**
     * Check if all bits are "true"
     **/
    bool All() const NN_NOEXCEPT
    {
        for (uint32_t i = 0; i < m_StorageCount; ++i)
        {
            if ((m_Storage[i] ^ BitMaskAll) != 0u)
            {
                return false;
            }
        }
        return true;
    }

    /**
     * Check if any bit is "true"
     **/
    bool Any() const NN_NOEXCEPT
    {
        for (uint32_t i = 0; i < m_StorageCount; ++i)
        {
            if ((m_Storage[i] & BitMaskAll) != 0u)
            {
                return true;
            }
        }
        return false;
    }

    /**
     * Check if all bits are "false"
     **/
    bool None() const NN_NOEXCEPT
    {
        return !Any();
    }

    /**
     * Count bits "true"
     **/
    uint32_t Count() const NN_NOEXCEPT
    {
        uint32_t cnt = 0;
        for (uint32_t i = 0; i < m_StorageCount; ++i)
        {
            cnt += static_cast<uint32_t>(std::bitset<BitCountPerStorage>(m_Storage[i]).count());
        }
        return cnt;
    }

    /**
     * Return the number of bits in the BitArray
     **/
    uint32_t Size() const NN_NOEXCEPT
    {
        return m_StorageCount * BitCountPerStorage;
    }

    /**
     * Set bit
     **/
    BitArray& Set(uint32_t pos, bool val = true) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(pos < m_StorageCount * BitCountPerStorage);

        if (val)
        {
            StorageOf_(pos) |= BitMaskOf_(pos);
        }
        else
        {
            StorageOf_(pos) &= ~BitMaskOf_(pos);
        }
        return *this;
    }

    /**
     * Set all bits as "true"
     **/
    BitArray& Set() NN_NOEXCEPT
    {
        for (uint32_t i = 0; i < m_StorageCount; ++i)
        {
            m_Storage[i] |= BitMaskAll;
        }
        return *this;
    }

    /**
     * Reset bit
     **/
    BitArray& Reset(uint32_t pos) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(pos < m_StorageCount * BitCountPerStorage);

        StorageOf_(pos) &= ~BitMaskOf_(pos);
        return *this;
    }

    /**
     * Reset all bits as "false"
     **/
    BitArray& Reset() NN_NOEXCEPT
    {
        for (uint32_t i = 0; i < m_StorageCount; ++i)
        {
            m_Storage[i] = 0;
        }
        return *this;
    }

    /**
     * Fli bit
     **/
    BitArray& Flip(uint32_t pos) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(pos < m_StorageCount * BitCountPerStorage);

        Set(pos, !Test(pos));
        return *this;
    }

    /**
     * Flip all bits
     **/
    BitArray& Flip() NN_NOEXCEPT
    {
        for (uint32_t i = 0; i < m_StorageCount; ++i)
        {
            m_Storage[i] = ~m_Storage[i];
        }
        return *this;
    }

    bool operator[](uint32_t pos) const NN_NOEXCEPT
    {
        return Test(pos);
    }

    bool operator==(const BitArray& rhs) const NN_NOEXCEPT
    {
        if ((m_Storage != nullptr && rhs.m_Storage != nullptr) &&
            (m_StorageCount == rhs.m_StorageCount))
        {
            for (uint32_t i = 0; i < m_StorageCount; ++i)
            {
                if (m_Storage[i] != rhs.m_Storage)
                {
                    return false;
                }
            }
            return true;
        }
        return false;
    }

    bool operator!=(const BitArray& rhs) NN_NOEXCEPT
    {
        return !(*this == rhs);
    }

private:
    NN_FORCEINLINE StorageType& StorageOf_(uint32_t pos) NN_NOEXCEPT
    {
        auto strgInd = pos / BitCountPerStorage;
        return m_Storage[strgInd];
    }

    NN_FORCEINLINE const StorageType& StorageOf_(uint32_t pos) const NN_NOEXCEPT
    {
        auto strgInd = pos / BitCountPerStorage;
        return m_Storage[strgInd];
    }

    NN_FORCEINLINE StorageType BitMaskOf_(uint32_t pos) NN_NOEXCEPT
    {
        auto bitInd = pos % BitCountPerStorage;
        return static_cast<StorageType>(1ull << bitInd);
    }

    NN_FORCEINLINE StorageType BitMaskOf_(uint32_t pos) const NN_NOEXCEPT
    {
        auto bitInd = pos % BitCountPerStorage;
        return static_cast<StorageType>(1ull << bitInd);
    }

private:
    static const uint32_t BitCountPerStorage = sizeof(StorageType) * 8;
    static const StorageType BitMaskAll = static_cast<StorageType>(-1);

    StorageType* m_Storage;
    uint32_t m_StorageCount;
};

}}}  // namespace nn::audio::common
