﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <nn/nn_SdkAssert.h>

#include <nn/audio/audio_AudioRendererTypes.h>
#include <nn/audio/audio_SubMix.h>
#include <nn/audio/audio_FinalMix.h>
#include "audio_MixManager.h"
#include "audio_SplitterInfoManager.h"
#include "audio_ResourceExclusionChecker.h"
#define NN_AUDIO_DETAIL_SUBMIX_EXCLUSION_SCOPED_CHEKCER(pSubMix) \
    detail::ScopedConfigInstanceAccessChecker scopedConfigInstanceAccessChecker(detail::FindResourceExclusionCheckerFromRegionInConfig(pSubMix->_pMixInfo))

namespace nn {
namespace audio {

bool AcquireSubMix(AudioRendererConfig* pConfig, SubMixType* pOutSubMix, int sampleRate, int bufferCount) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pConfig);
    NN_SDK_REQUIRES_NOT_NULL(pOutSubMix);
    NN_SDK_REQUIRES(sampleRate == 32000 || sampleRate == 48000);
    NN_SDK_REQUIRES_GREATER(bufferCount, 0);
    NN_SDK_REQUIRES_LESS_EQUAL(bufferCount, MixBufferCountMax);
    auto pMixManager = pConfig->_pMixManager;

    if(!pMixManager->Acquire(pOutSubMix, bufferCount))
    {
        return false;
    }

    NN_AUDIO_DETAIL_SUBMIX_EXCLUSION_SCOPED_CHEKCER(pOutSubMix);
    pOutSubMix->_pMixInfo->volume = 1.0f;
    pOutSubMix->_pMixInfo->sampleRate = sampleRate;
    memset(pOutSubMix->_pMixInfo->mixVolume, 0, sizeof(pOutSubMix->_pMixInfo->mixVolume));

    return true;
}

void ReleaseSubMix(AudioRendererConfig* pConfig, SubMixType* pSubMix) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pConfig);
    NN_SDK_REQUIRES_NOT_NULL(pSubMix);
    NN_SDK_REQUIRES_NOT_NULL(pSubMix->_pMixInfo);
    NN_AUDIO_DETAIL_SUBMIX_EXCLUSION_SCOPED_CHEKCER(pSubMix);
    auto pMixManager = pConfig->_pMixManager;
    NN_SDK_REQUIRES(pMixManager->IsValidMix(pSubMix));

    pMixManager->Release(pSubMix);
}

void SetSubMixDestination(AudioRendererConfig* pConfig, SubMixType* pSource, FinalMixType* pDestination) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pConfig);
    NN_SDK_REQUIRES_NOT_NULL(pSource);
    NN_SDK_REQUIRES_NOT_NULL(pDestination);
    NN_SDK_REQUIRES_NOT_NULL(pSource->_pMixInfo);
    NN_SDK_REQUIRES_NOT_NULL(pDestination->_pMixInfo);
    NN_SDK_REQUIRES(pConfig->_pMixManager->IsValidMix(pSource));
    NN_SDK_REQUIRES(pConfig->_pMixManager->IsValidMix(pDestination));
    NN_AUDIO_DETAIL_SUBMIX_EXCLUSION_SCOPED_CHEKCER(pSource);
    NN_UNUSED(pConfig);
    pSource->_pMixInfo->destinationMixId = pDestination->_pMixInfo->mixId;
}

void SetSubMixDestination(AudioRendererConfig* pConfig, SubMixType* pSource, SubMixType* pDestination) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pConfig);
    NN_SDK_REQUIRES_NOT_NULL(pSource);
    NN_SDK_REQUIRES_NOT_NULL(pDestination);
    NN_SDK_REQUIRES_NOT_NULL(pSource->_pMixInfo);
    NN_SDK_REQUIRES_NOT_NULL(pDestination->_pMixInfo);
    NN_SDK_REQUIRES(pConfig->_pMixManager->IsValidMix(pSource));
    NN_SDK_REQUIRES(pConfig->_pMixManager->IsValidMix(pDestination));
    NN_AUDIO_DETAIL_SUBMIX_EXCLUSION_SCOPED_CHEKCER(pSource);
    NN_UNUSED(pConfig);
    pSource->_pMixInfo->destinationMixId = pDestination->_pMixInfo->mixId;
}

void SetSubMixDestination(AudioRendererConfig* pConfig, SubMixType* pSource, SplitterType* pDestination) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pConfig);
    NN_SDK_REQUIRES_NOT_NULL(pSource);
    NN_SDK_REQUIRES_NOT_NULL(pSource->_pMixInfo);
    NN_SDK_REQUIRES_NOT_NULL(pDestination);
    NN_SDK_REQUIRES_NOT_NULL(pDestination->_pSplitterInfo);
    NN_SDK_REQUIRES(pConfig->_pMixManager->IsValidMix(pSource));
    NN_AUDIO_DETAIL_SUBMIX_EXCLUSION_SCOPED_CHEKCER(pSource);
    NN_UNUSED(pConfig);
    pSource->_pMixInfo->destinationMixId = Invalid_MixId;
    pSource->_pMixInfo->splitterInfoId = pDestination->_pSplitterInfo->GetId();
}

int GetSubMixSampleRate(const SubMixType* pSubMix) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSubMix);
    NN_SDK_REQUIRES_NOT_NULL(pSubMix->_pMixInfo);
    return pSubMix->_pMixInfo->sampleRate;
}

int GetSubMixBufferCount(const SubMixType* pSubMix) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSubMix);
    NN_SDK_REQUIRES_NOT_NULL(pSubMix->_pMixInfo);
    return pSubMix->_pMixInfo->bufferCount;
}

float GetSubMixVolume(const SubMixType* pSubMix) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSubMix);
    NN_SDK_REQUIRES_NOT_NULL(pSubMix->_pMixInfo);
    return pSubMix->_pMixInfo->volume;
}

void SetSubMixVolume(SubMixType* pSubMix, float volume) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSubMix);
    NN_SDK_REQUIRES_NOT_NULL(pSubMix->_pMixInfo);
    NN_SDK_REQUIRES_MINMAX(volume, nn::audio::SubMixType::GetVolumeMin(), nn::audio::SubMixType::GetVolumeMax());
    NN_AUDIO_DETAIL_SUBMIX_EXCLUSION_SCOPED_CHEKCER(pSubMix);
    pSubMix->_pMixInfo->volume = volume;
}

void SetSubMixMixVolume(SubMixType* pSource, FinalMixType* pDestination, float volume, int sourceIndex, int destinationIndex) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSource);
    NN_SDK_REQUIRES_NOT_NULL(pDestination);
    NN_SDK_REQUIRES_MINMAX(volume, nn::audio::SubMixType::GetVolumeMin(), nn::audio::SubMixType::GetVolumeMax());
    NN_SDK_REQUIRES_GREATER_EQUAL(sourceIndex, 0);
    NN_SDK_REQUIRES_LESS(sourceIndex, nn::audio::GetSubMixBufferCount(pSource));
    NN_SDK_REQUIRES_GREATER_EQUAL(destinationIndex, 0);
    NN_SDK_REQUIRES_LESS(destinationIndex, nn::audio::GetFinalMixBufferCount(pDestination));
    NN_SDK_REQUIRES(pSource->_pMixInfo->destinationMixId == pDestination->_pMixInfo->mixId, "pDestination is not a destination FinalMix");
    NN_SDK_REQUIRES_NOT_NULL(pSource->_pMixInfo);
    NN_SDK_REQUIRES_NOT_NULL(pDestination->_pMixInfo);
    NN_AUDIO_DETAIL_SUBMIX_EXCLUSION_SCOPED_CHEKCER(pSource);
    NN_UNUSED(pDestination);

    pSource->_pMixInfo->mixVolume[sourceIndex][destinationIndex] = volume;
}

float GetSubMixMixVolume(const SubMixType* pSource, const FinalMixType* pDestination, int sourceIndex, int destinationIndex) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSource);
    NN_SDK_REQUIRES_NOT_NULL(pDestination);
    NN_SDK_REQUIRES_GREATER_EQUAL(sourceIndex, 0);
    NN_SDK_REQUIRES_LESS(sourceIndex, nn::audio::GetSubMixBufferCount(pSource));
    NN_SDK_REQUIRES_GREATER_EQUAL(destinationIndex, 0);
    NN_SDK_REQUIRES_LESS(destinationIndex, nn::audio::GetFinalMixBufferCount(pDestination));
    NN_SDK_REQUIRES(pSource->_pMixInfo->destinationMixId == pDestination->_pMixInfo->mixId, "pDestination is not a destination FinalMix");
    NN_SDK_REQUIRES_NOT_NULL(pSource->_pMixInfo);
    NN_SDK_REQUIRES_NOT_NULL(pDestination->_pMixInfo);
    NN_UNUSED(pDestination);

    return pSource->_pMixInfo->mixVolume[sourceIndex][destinationIndex];
}

void SetSubMixMixVolume(SubMixType* pSource, SubMixType* pDestination, float volume, int sourceIndex, int destinationIndex) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSource);
    NN_SDK_REQUIRES_NOT_NULL(pDestination);
    NN_SDK_REQUIRES_MINMAX(volume, nn::audio::SubMixType::GetVolumeMin(), nn::audio::SubMixType::GetVolumeMax());
    NN_SDK_REQUIRES_GREATER_EQUAL(sourceIndex, 0);
    NN_SDK_REQUIRES_LESS(sourceIndex, nn::audio::GetSubMixBufferCount(pSource));
    NN_SDK_REQUIRES_GREATER_EQUAL(destinationIndex, 0);
    NN_SDK_REQUIRES_LESS(destinationIndex, nn::audio::GetSubMixBufferCount(pDestination));
    NN_SDK_REQUIRES(pSource->_pMixInfo->destinationMixId == pDestination->_pMixInfo->mixId, "pDestination is not a destination SubMix");
    NN_SDK_REQUIRES_NOT_NULL(pSource->_pMixInfo);
    NN_SDK_REQUIRES_NOT_NULL(pDestination->_pMixInfo);
    NN_AUDIO_DETAIL_SUBMIX_EXCLUSION_SCOPED_CHEKCER(pSource);
    NN_UNUSED(pDestination);

    pSource->_pMixInfo->mixVolume[sourceIndex][destinationIndex] = volume;
}

float GetSubMixMixVolume(const SubMixType* pSource, const SubMixType* pDestination, int sourceIndex, int destinationIndex) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSource);
    NN_SDK_REQUIRES_NOT_NULL(pDestination);
    NN_SDK_REQUIRES_GREATER_EQUAL(sourceIndex, 0);
    NN_SDK_REQUIRES_LESS(sourceIndex, nn::audio::GetSubMixBufferCount(pSource));
    NN_SDK_REQUIRES_GREATER_EQUAL(destinationIndex, 0);
    NN_SDK_REQUIRES_LESS(destinationIndex, nn::audio::GetSubMixBufferCount(pDestination));
    NN_SDK_REQUIRES(pSource->_pMixInfo->destinationMixId == pDestination->_pMixInfo->mixId, "pDestination is not a destination SubMix");
    NN_SDK_REQUIRES_NOT_NULL(pSource->_pMixInfo);
    NN_SDK_REQUIRES_NOT_NULL(pDestination->_pMixInfo);
    NN_UNUSED(pDestination);
    return pSource->_pMixInfo->mixVolume[sourceIndex][destinationIndex];
}

NodeId GetSubMixNodeId(const SubMixType* pSubMix) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSubMix);
    NN_SDK_REQUIRES_NOT_NULL(pSubMix->_pMixInfo);
    return pSubMix->_pMixInfo->nodeId;
}

}  // namespace audio
}  // namespace nn
