﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <limits>

#include <nn/util/util_BitUtil.h>
#include <nn/nn_SdkAssert.h>

#include <nn/audio/audio_Common.h>
#include <nn/audio/audio_AudioRendererTypes.h>
#include <nn/audio/audio_VoiceTypes.h>
#include <nn/audio/audio_FinalMixApi.h>
#include <nn/audio/audio_FinalMixTypes.h>
#include <nn/audio/audio_SubMixTypes.h>
#include <nn/audio/audio_SubMixApi.h>
#include <nn/audio/audio_Splitter.h>

#include "audio_VoiceInfoManager.h"
#include "audio_SplitterInfoManager.h"
#include "audio_MixManager.h"
#include "common/audio_NodeIdManager.h"
#include "common/audio_CommonWaveBuffer.h"
#include "audio_ResourceExclusionChecker.h"

#define NN_AUDIO_DETAIL_SPLITTER_SCOPED_CHEKCER(pSplitter) \
    detail::ScopedConfigInstanceAccessChecker scopedConfigInstanceAccessCheckerForSink(detail::FindResourceExclusionCheckerFromRegionInConfig(pSplitter->_pSplitterInfo))

namespace nn {
namespace audio {

namespace
{
bool CheckInitialized(const SplitterType* pSplitter)
{
    return (pSplitter != nullptr) && (pSplitter->_pSplitterInfo != nullptr);
}

}

bool AcquireSplitter(AudioRendererConfig* pConfig, SplitterType* pOutSplitter, int sampleRate, int sourceChannelCount ,int destinationCount) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pConfig);
    NN_SDK_REQUIRES_NOT_NULL(pOutSplitter);
    // TODO: check sampleRate

    auto pSplitterInfo = pConfig->_pSplitterInfoManager->AcquireSplitterInfo(sampleRate, sourceChannelCount, destinationCount);
    pOutSplitter->_pSplitterInfo = pSplitterInfo;
    NN_AUDIO_DETAIL_SPLITTER_SCOPED_CHEKCER(pOutSplitter);

    return pSplitterInfo != nullptr;
}

void ReleaseSplitter(AudioRendererConfig* pConfig, SplitterType* pOutSplitter) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pConfig);
    NN_SDK_REQUIRES(CheckInitialized(pOutSplitter), "pSplitter is not initialized");
    NN_AUDIO_DETAIL_SPLITTER_SCOPED_CHEKCER(pOutSplitter);

    if (CheckInitialized(pOutSplitter) == false)
    {
        return;
    }

    auto pSplitterInfo = pOutSplitter->_pSplitterInfo;
    pConfig->_pSplitterInfoManager->ReleaseSplitterInfo(pSplitterInfo);
    pOutSplitter->_pSplitterInfo = nullptr;
}

int GetSplitterDestinationCount(const SplitterType* pSplitter) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(CheckInitialized(pSplitter), "pSplitter is not initialized");
    return pSplitter->_pSplitterInfo->GetDestinationCount();
}

int GetSplitterSourceChannelCount(const SplitterType* pSplitter) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(CheckInitialized(pSplitter), "pSplitter is not initialized");
    return pSplitter->_pSplitterInfo->GetChannelCount();
}

int GetSplitterSampleRate(const SplitterType* pSplitter) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(CheckInitialized(pSplitter), "pSplitter is not initialized");
    return pSplitter->_pSplitterInfo->GetSampleRate();
}

void SetSplitterDestination(AudioRendererConfig* pConfig, SplitterType* pSource, int sendIndex, MixId destinationId) NN_NOEXCEPT
{
    NN_AUDIO_DETAIL_SPLITTER_SCOPED_CHEKCER(pSource);
    NN_UNUSED(pConfig);
    if (CheckInitialized(pSource) == false)
    {
        return;
    }

    for (auto chIdx = 0; chIdx < pSource->_pSplitterInfo->GetChannelCount(); ++chIdx)
    {
        pSource->_pSplitterInfo->GetDestinationData(sendIndex, chIdx)->SetDestinationId(destinationId);
    }
    // Service 側での更新を簡素化するために SplitterInfo も Mark する。
    pSource->_pSplitterInfo->MarkAsUpdated();
}

void SetSplitterDestination(AudioRendererConfig* pConfig, SplitterType* pSource, int sendIndex, FinalMixType* pDestination) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(CheckInitialized(pSource), "pSource is not initialized");
    NN_SDK_REQUIRES_NOT_NULL(pDestination);
    NN_SDK_REQUIRES_EQUAL(GetFinalMixSampleRate(pDestination), GetSplitterSampleRate(pSource));
    NN_SDK_REQUIRES_LESS(sendIndex, GetSplitterDestinationCount(pSource));
    NN_AUDIO_DETAIL_SPLITTER_SCOPED_CHEKCER(pSource);
    SetSplitterDestination(pConfig, pSource, sendIndex, pDestination->_pMixInfo->mixId);
}

void SetSplitterDestination(AudioRendererConfig* pConfig, SplitterType* pSource, int sendIndex, SubMixType* pDestination) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(CheckInitialized(pSource), "pSource is not initialized");
    NN_SDK_REQUIRES_NOT_NULL(pDestination);
    NN_SDK_REQUIRES_EQUAL(GetSubMixSampleRate(pDestination), GetSplitterSampleRate(pSource));
    NN_SDK_REQUIRES_LESS(sendIndex, GetSplitterDestinationCount(pSource));
    NN_AUDIO_DETAIL_SPLITTER_SCOPED_CHEKCER(pSource);
    SetSplitterDestination(pConfig, pSource, sendIndex, pDestination->_pMixInfo->mixId);
}

void SetSplitterMixVolume(SplitterType* pSource, int destinationIndex, MixInfo* pMixInfo, float volume, int sourceChannelIndex, int destinationChannelIndex) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(CheckInitialized(pSource), "pSource is not initialized");
    NN_AUDIO_DETAIL_SPLITTER_SCOPED_CHEKCER(pSource);
    if (CheckInitialized(pSource) == false)
    {
        return;
    }
    NN_SDK_REQUIRES_MINMAX(volume, VoiceType::GetVolumeMin(), VoiceType::GetVolumeMax());
    NN_SDK_REQUIRES_LESS(destinationIndex, GetSplitterDestinationCount(pSource));
    auto splitterDestinationData = pSource->_pSplitterInfo->GetDestinationData(destinationIndex, sourceChannelIndex);
    NN_SDK_REQUIRES_EQUAL(pMixInfo->mixId, splitterDestinationData->GetDestinationId());
    NN_UNUSED(pMixInfo);
    splitterDestinationData->SetMixVolume(volume, destinationChannelIndex);
}

void SetSplitterMixVolume(SplitterType* pSource, int destinationIndex, SubMixType* pDestination, float volume, int sourceChannelIndex, int destinationChannelIndex) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(CheckInitialized(pSource), "pSource is not initialized");
    NN_SDK_REQUIRES_RANGE(destinationChannelIndex, 0, GetSubMixBufferCount(pDestination));
    NN_AUDIO_DETAIL_SPLITTER_SCOPED_CHEKCER(pSource);
    SetSplitterMixVolume(pSource, destinationIndex, pDestination->_pMixInfo, volume, sourceChannelIndex, destinationChannelIndex);
}

void SetSplitterMixVolume(SplitterType* pSource, int destinationIndex, FinalMixType* pDestination, float volume, int sourceChannelIndex, int destinationChannelIndex) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(CheckInitialized(pSource), "pSource is not initialized");
    NN_SDK_REQUIRES_RANGE(destinationChannelIndex, 0, GetFinalMixBufferCount(pDestination));
    NN_AUDIO_DETAIL_SPLITTER_SCOPED_CHEKCER(pSource);
    SetSplitterMixVolume(pSource, destinationIndex, pDestination->_pMixInfo, volume, sourceChannelIndex, destinationChannelIndex);
}

float GetSplitterMixVolume(const SplitterType* pSource, int destinationIndex, const MixInfo* pMixInfo, int sourceChannelIndex, int destinationChannelIndex) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_LESS(destinationIndex, GetSplitterDestinationCount(pSource));
    if (CheckInitialized(pSource) == false)
    {
        return 0.0f;
    }
    auto splitterDestinationData = pSource->_pSplitterInfo->GetDestinationData(destinationIndex, sourceChannelIndex);
    NN_SDK_REQUIRES_EQUAL(pMixInfo->mixId, splitterDestinationData->GetDestinationId());
    NN_UNUSED(pMixInfo);
    return splitterDestinationData->GetMixVolume(destinationChannelIndex);
}

float GetSplitterMixVolume(const SplitterType* pSource, int destinationIndex, const SubMixType* pDestination, int sourceChannelIndex, int destinationChannelIndex) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(CheckInitialized(pSource), "pSource is not initialized");
    NN_SDK_REQUIRES_RANGE(destinationChannelIndex, 0, GetSubMixBufferCount(pDestination));
    return GetSplitterMixVolume(pSource, destinationIndex, pDestination->_pMixInfo, sourceChannelIndex, destinationChannelIndex);
}

float GetSplitterMixVolume(const SplitterType* pSource, int destinationIndex, const FinalMixType* pDestination, int sourceChannelIndex, int destinationChannelIndex) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(CheckInitialized(pSource), "pSource is not initialized");
    NN_SDK_REQUIRES_RANGE(destinationChannelIndex, 0, GetFinalMixBufferCount(pDestination));
    return GetSplitterMixVolume(pSource, destinationIndex, pDestination->_pMixInfo, sourceChannelIndex, destinationChannelIndex);
}

}  // namespace audio
}  // namespace nn
