﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <cstring>
#include <nn/nn_TimeSpan.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/audio/audio_Result.h>
#include "audio_SinkManager.h"
#include "common/audio_NodeIdManager.h"

namespace nn {
namespace audio {

NN_DEFINE_STATIC_CONSTANT(const int SinkInfo::InputCountMax);

SinkInfo::SinkInfo() NN_NOEXCEPT
{
    CleanUp();
}

void SinkInfo::CleanUp() NN_NOEXCEPT
{
    this->_type = common::SinkType_Invalid;
    this->_isInUse = false;
}

common::SinkType SinkInfo::GetType() const NN_NOEXCEPT
{
    return this->_type;
}

size_t SinkInfo::StoreInParameter(common::SinkInParameter& param) NN_NOEXCEPT
{
    param._type = this->_type;
    param._isInUse = this->_isInUse;
    param._nodeId = this->_nodeId;
    switch(this->_type)
    {
    case common::SinkType_CircularBuffer:
        param._circular = this->circularInParam;
        break;
    case common::SinkType_Device:
        param._device = this->deviceInParam;
        break;
    case common::SinkType_Invalid:
        // pass
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
    return sizeof(common::SinkInParameter);
}

size_t SinkInfo::StoreOutStatus(common::SinkOutStatus& outStatus) NN_NOEXCEPT
{
    this->circularOutStatus = outStatus.circular;
    return sizeof(common::SinkOutStatus);
}

bool SinkInfo::IsUsed() const NN_NOEXCEPT
{
    return this->_isInUse;
}

SinkManager::SinkManager(int count, SinkInfo* pInfo, int32_t sampleCount, int32_t sampleRate) NN_NOEXCEPT
    : m_Count(count)
    , m_FinalMixSampleCount(sampleCount)
    , m_FinalMixSampleRate(sampleRate)
    , m_Infos(pInfo)
    , m_Mutex(true)
{
}

SinkInfo* SinkManager::Allocate(common::SinkType type) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    SinkInfo* p = nullptr;
    for (int i = 0; i < m_Count; ++i)
    {
        if (m_Infos[i].IsUsed() == false)
        {
            p = &m_Infos[i];
            p->_type = type;
            p->_isInUse = true;
            p->_nodeId = common::NodeIdManager::GetNodeId(common::NodeIdManager::NodeIdType::Sink, i, 0);
            memset(p->_reserved, 0, sizeof(p->_reserved));
            break;
        }
    }
    return p;
}

void SinkManager::Free(SinkInfo* pInfo) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    pInfo->CleanUp();
}

int32_t SinkManager::GetSampleCount() const NN_NOEXCEPT
{
    return m_FinalMixSampleCount;
}

int32_t SinkManager::GetSampleRate() const NN_NOEXCEPT
{
    return m_FinalMixSampleRate;
}

bool SinkManager::IsValidSinkInfo(const SinkInfo* pInfo) const NN_NOEXCEPT
{
    return m_Infos <= pInfo
        && pInfo <= (m_Infos + m_Count)
        && (reinterpret_cast<uintptr_t>(pInfo) - reinterpret_cast<uintptr_t>(m_Infos)) % sizeof(SinkInfo) == 0;
}

size_t SinkManager::UpdateSinkInParameter(common::SinkInParameter* pSink) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    size_t size = 0;
    for (auto i = 0; i < m_Count; ++i)
    {
        size += m_Infos[i].StoreInParameter(pSink[i]);
    }
    return size;
}

size_t SinkManager::UpdateSinkOutStatus(common::SinkOutStatus* pOutStatus) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    size_t size = 0;
    for (auto i = 0; i < m_Count; ++i)
    {
        size += m_Infos[i].StoreOutStatus(pOutStatus[i]);
    }
    return size;
}

}  // namespace audio
}  // namespace nn
