﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <mutex>
#include <cstdint>
#include <atomic>
#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/audio/audio_AudioRendererTypes.h>
#include <nn/os/os_Mutex.h>
#include <nn/nn_SdkLog.h>

namespace nn { namespace audio { namespace detail {

class ResourceExclusionChecker
{
public:
    ResourceExclusionChecker() NN_NOEXCEPT
    {
        Reset();
    }

    bool NotifyRequestUpdateAudioRendererBegin() NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_IsUsed);

        return m_RequestUpdateAudioRendererAccessingCount++ == 0 && m_ConfigAccessingCount == 0;
    }

    bool NotifyRequestUpdateAudioRendererEnd() NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_IsUsed);

        return --m_RequestUpdateAudioRendererAccessingCount == 0 && m_ConfigAccessingCount == 0;
    }

    bool NotifyAccessAudioRendererConfigBegin() NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_IsUsed);

        return m_ConfigAccessingCount++ >= 0 && m_RequestUpdateAudioRendererAccessingCount == 0;
    }

    bool NotifyAccessAudioRendererConfigEnd() NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_IsUsed);

        return --m_ConfigAccessingCount >= 0 && m_RequestUpdateAudioRendererAccessingCount == 0;
    }

    void SetUsed(bool isUsed) NN_NOEXCEPT
    {
        m_IsUsed = isUsed;
    }

    bool IsUsed() NN_NOEXCEPT
    {
        return m_IsUsed;
    }

    bool IsInstanceIncludedInConfig(void* p) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(p);
        NN_SDK_ASSERT(m_IsUsed);

        if(m_pConfigBuffer == nullptr || m_ConfigBufferSize == 0)
        {
            return false;
        }

        const auto ConfigBufferBegin = reinterpret_cast<uintptr_t>(m_pConfigBuffer);
        const auto ConfigBufferEnd = ConfigBufferBegin + m_ConfigBufferSize - 1;
        const auto InstanceBegin = reinterpret_cast<uintptr_t>(p);

        return ConfigBufferBegin <= InstanceBegin && InstanceBegin <= ConfigBufferEnd;
    }

    bool RegisterConfig(void* pConfigBuffer, size_t size) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pConfigBuffer);
        NN_SDK_ASSERT_GREATER(size, 0u);
        NN_SDK_ASSERT(m_IsUsed);

        if(m_pConfigBuffer != nullptr && m_pConfigBuffer != pConfigBuffer)
        {
            return false;
        }

        m_pConfigBuffer = pConfigBuffer;
        m_ConfigBufferSize = size;

        return true;
    }

    bool RegisterInternalHandle(void* pHandle) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pHandle);
        NN_SDK_ASSERT(m_IsUsed);

        if(m_pInternalHandle != nullptr && m_pInternalHandle != pHandle)
        {
            return false;
        }

        m_pInternalHandle = pHandle;

        return true;
    }

    void Reset() NN_NOEXCEPT
    {
        m_IsUsed = false;
        m_pInternalHandle = nullptr;
        m_pConfigBuffer = nullptr;
        m_ConfigBufferSize = 0;
        m_ConfigAccessingCount = 0;
        m_RequestUpdateAudioRendererAccessingCount = 0;
    }

    const void* GetInternalHandle() NN_NOEXCEPT
    {
        return m_pInternalHandle;
    }

private:
    std::atomic<bool> m_IsUsed;
    const void* m_pInternalHandle;
    const void* m_pConfigBuffer;
    size_t m_ConfigBufferSize;
    std::atomic_int m_ConfigAccessingCount;
    std::atomic_int m_RequestUpdateAudioRendererAccessingCount;
};

extern std::atomic<bool> g_IsResourceExclusionCheckEnabled;
ResourceExclusionChecker* FindResourceExclusionCheckerFromRegionInConfigImpl(void* p) NN_NOEXCEPT;
ResourceExclusionChecker* FindResourceExclusionCheckerFromInternalHandleImpl(void* pInternalHandle) NN_NOEXCEPT;

ResourceExclusionChecker* AcquireResourceExclusionChecker() NN_NOEXCEPT;
void ReleaseResourceExclusionChecker(ResourceExclusionChecker *pChecker) NN_NOEXCEPT;

inline ResourceExclusionChecker* FindResourceExclusionCheckerFromRegionInConfig(void* p) NN_NOEXCEPT
{
    if(!g_IsResourceExclusionCheckEnabled)
    {
        return nullptr;
    }
    else
    {
        return FindResourceExclusionCheckerFromRegionInConfigImpl(p);
    }
}

inline ResourceExclusionChecker* FindResourceExclusionCheckerFromInternalHandle(void* pInternalHandle) NN_NOEXCEPT
{
    if(!g_IsResourceExclusionCheckEnabled)
    {
        return nullptr;
    }
    else
    {
        return FindResourceExclusionCheckerFromInternalHandleImpl(pInternalHandle);
    }
}

inline bool IsResourceExclusionCheckEnabled() NN_NOEXCEPT
{
    return g_IsResourceExclusionCheckEnabled;
}

inline void SetResourceExclusionCheckEnabled(bool enabled) NN_NOEXCEPT
{
    g_IsResourceExclusionCheckEnabled = enabled;
}

class ScopedConfigInstanceAccessChecker
{
private:
    ResourceExclusionChecker* m_pChecker;
public:
    explicit ScopedConfigInstanceAccessChecker(ResourceExclusionChecker* pChecker) NN_NOEXCEPT
    {
        m_pChecker = pChecker;
        if(m_pChecker)
        {
            NN_SDK_ASSERT(!IsResourceExclusionCheckEnabled() || m_pChecker->NotifyAccessAudioRendererConfigBegin(), "This function must not be called when RequestUpdateAudioRenderer() is being called.");
        }
    }

    ~ScopedConfigInstanceAccessChecker() NN_NOEXCEPT
    {
        if(m_pChecker)
        {
            NN_SDK_ASSERT(!IsResourceExclusionCheckEnabled() || m_pChecker->NotifyAccessAudioRendererConfigEnd(), "This function must not be called when RequestUpdateAudioRenderer() is being called.");
        }
    }
};

class ScopedHandleAccessChecker
{
private:
    ResourceExclusionChecker* m_pChecker;
public:
    explicit ScopedHandleAccessChecker(ResourceExclusionChecker* pChecker) NN_NOEXCEPT
    {
        m_pChecker = pChecker;
        if(m_pChecker)
        {
            NN_SDK_ASSERT(!IsResourceExclusionCheckEnabled() || m_pChecker->NotifyRequestUpdateAudioRendererBegin(), "This function must not be called when AudioRendererConfig is being accessed.");
        }
    }

    ~ScopedHandleAccessChecker() NN_NOEXCEPT
    {
        if(m_pChecker)
        {
            NN_SDK_ASSERT(!IsResourceExclusionCheckEnabled() || m_pChecker->NotifyRequestUpdateAudioRendererEnd(), "This function must not be called when AudioRendererConfig is being accessed.");
        }
    }
};

}}} // nn::audio::detail
