﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/audio/audio_PerformanceMetrics.h>
#include <nn/audio/audio_Voice.h>
#include <nn/audio/audio_SubMix.h>
#include <nn/audio/audio_FinalMix.h>

#include "common/audio_PerformanceMetricsCommon.h"
#include "common/audio_NodeIdManager.h"
#include "audio_AudioRendererUtil.h"
#include "audio_PerformanceBufferManager.h"
#include "audio_ResourceExclusionChecker.h"

namespace nn { namespace audio {

size_t GetRequiredBufferSizeForPerformanceFrames(const AudioRendererParameter& parameter) NN_NOEXCEPT
{
    auto infoSize = common::GetRequiredBufferSizeForPerformanceMetricsPerFrame<PerformanceFrameHeader, PerformanceEntry, PerformanceDetail>(detail::ConvertToAudioRendererParameterInternal(parameter)) * (parameter.performanceFrameCount);
    auto marginSize = (infoSize > 0) ? sizeof(PerformanceFrameHeader) /* for last frame margin */ : 0;
    return nn::util::align_up(infoSize + marginSize, nn::audio::BufferAlignSize);
}

void* SetPerformanceFrameBuffer(AudioRendererConfig* pConfig, void* buffer, size_t bufferSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pConfig);

    auto pManager = pConfig->_pPerformanceBufferManager;
    auto rt = pManager->GetCurrentPerformanceFrameBuffer();
    pManager->SetPerformanceFrameBuffer(buffer, bufferSize);
    return rt;
}

void SetPerformanceDetailTarget(AudioRendererConfig* pConfig, const VoiceType* pVoice) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pConfig);
    NN_SDK_REQUIRES_NOT_NULL(pVoice);
    NN_SDK_REQUIRES(IsVoiceValid(pVoice));

    auto pManager = pConfig->_pPerformanceBufferManager;
    pManager->SetDetailTarget(GetVoiceNodeId(pVoice));
}

void SetPerformanceDetailTarget(AudioRendererConfig* pConfig, const SubMixType* pSubMix) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pConfig);
    NN_SDK_REQUIRES_NOT_NULL(pSubMix);
    NN_SDK_REQUIRES_NOT_NULL(pSubMix->_pMixInfo);

    auto pManager = pConfig->_pPerformanceBufferManager;
    pManager->SetDetailTarget(GetSubMixNodeId(pSubMix));
}

void SetPerformanceDetailTarget(AudioRendererConfig* pConfig, const FinalMixType* pFinalMix) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pConfig);
    NN_SDK_REQUIRES_NOT_NULL(pFinalMix);
    NN_SDK_REQUIRES_NOT_NULL(pFinalMix->_pMixInfo);

    auto pManager = pConfig->_pPerformanceBufferManager;
    pManager->SetDetailTarget(GetFinalMixNodeId(pFinalMix));
}

void ClearPerformanceDetailTarget(AudioRendererConfig* pConfig) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pConfig);

    auto pManager = pConfig->_pPerformanceBufferManager;
    pManager->SetDetailTarget(common::NodeIdManager::GetInvalidNodeId());
}

}} // namespace nn::audio
