﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring> // std::memcpy
#include <mutex> // std::lock_guard
#include <algorithm> // std::min

#include <nn/nn_SdkAssert.h>
#include <nn/util/util_BytePtr.h>
#include <nn/util/util_PlacementArray.h>
#include <nn/os/os_Tick.h>

#include <nn/audio/audio_PerformanceMetrics.h>

#include "common/audio_Util.h"
#include "common/audio_PerformanceMetricsCommon.h"
#include "audio_PerformanceBufferManager.h"

namespace nn {
namespace audio {

NN_STATIC_ASSERT(sizeof(PerformanceEntry) % 4 == 0);
NN_STATIC_ASSERT(sizeof(PerformanceDetail) % 4 == 0);

PerformanceInfo::PerformanceInfo() NN_NOEXCEPT
    : m_Buffer(nullptr)
    , m_BufferSize(0)
    , m_Header(nullptr)
    , m_Entries(nullptr)
    , m_Details(nullptr)
{
}

PerformanceInfo::~PerformanceInfo() NN_NOEXCEPT
{
    m_Buffer = nullptr;
    m_BufferSize = 0;
    m_Header = nullptr;
    m_Entries = nullptr;
    m_Details = nullptr;
}

bool PerformanceInfo::SetBuffer(const void* frameBuffer, size_t bufferSize) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(frameBuffer);

    m_Buffer = frameBuffer;
    m_BufferSize = bufferSize;

    nn::util::ConstBytePtr addr(frameBuffer);
    m_Header = reinterpret_cast<const PerformanceFrameHeader*>(addr.Get());
    addr += sizeof(PerformanceFrameHeader);

    if (m_Header->frameMagic != common::CreateMagic(common::PerformanceFrameHeaderMagic))
    {
        m_Buffer = nullptr;
        m_Header = nullptr;
        m_Entries = nullptr;
        m_Details = nullptr;
        return false;
    }

    m_Entries = (m_Header->entryCount != 0) ? reinterpret_cast<const PerformanceEntry*>(addr.Get()) : nullptr;
    addr += sizeof(PerformanceEntry) * m_Header->entryCount;
    m_Details = (m_Header->detailCount != 0) ? reinterpret_cast<const PerformanceDetail*>(addr.Get()) : nullptr;

    return true;
}

int32_t PerformanceInfo::GetTotalProcessingTime() const NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_Header != nullptr, "PerformanceInfo is not initialized");
    return m_Header->totalProcessingTime;
}

bool PerformanceInfo::IsRenderingTimeLimitExceeded() const NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_Header != nullptr, "PerformanceInfo is not initialized");
    return m_Header->isRenderingTimeLimitExceeded;
}

int32_t PerformanceInfo::GetVoiceDropCount() const NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_Header != nullptr, "PerformanceInfo is not initialized");
    return m_Header->voiceDropCount;
}


nn::os::Tick PerformanceInfo::GetStartTime() const NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_Header != nullptr, "PerformanceInfo is not initialized");
    return nn::os::Tick(m_Header->startTime);
}

uint32_t PerformanceInfo::GetFrameIndex() const NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_Header != nullptr, "PerformanceInfo is not initialized");
    return m_Header->frameIndex;
}

const PerformanceEntry* PerformanceInfo::GetEntries(int* pOutCount) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutCount);
    NN_SDK_ASSERT(m_Header != nullptr, "PerformanceInfo is not initialized");
    *pOutCount = m_Header->entryCount;
    return m_Entries;
}

const PerformanceDetail* PerformanceInfo::GetDetails(int* pOutCount) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutCount);
    NN_SDK_ASSERT(m_Header != nullptr, "PerformanceInfo is not initialized");
    *pOutCount = m_Header->detailCount;
    return m_Details;
}

bool PerformanceInfo::MoveToNextFrame() NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_Header != nullptr, "PerformanceInfo is not initialized");
    nn::util::ConstBytePtr addr(m_Header);
    if (m_Header->offsetToNextHeader > 0)
    {
        addr += m_Header->offsetToNextHeader;
        return this->SetBuffer(addr.Get(), m_BufferSize);
    }
    else
    {
        return false;
    }
}

//================================================================================
// PerformanceBufferManager
//================================================================================
PerformanceBufferManager::PerformanceBufferManager() NN_NOEXCEPT
    : m_BufferBase(nullptr)
    , m_BufferSize(0)
    , m_Mutex(true)
{
}

void* PerformanceBufferManager::GetCurrentPerformanceFrameBuffer() NN_NOEXCEPT
{
    return m_BufferBase.Get();
}

void PerformanceBufferManager::SetPerformanceFrameBuffer(void* buffer, size_t bufferSize) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    if (buffer)
    {
        NN_SDK_ASSERT_NOT_NULL(buffer);
        NN_SDK_ASSERT_ALIGNED(buffer, nn::audio::BufferAlignSize);
    }

    if (buffer != nullptr)
    {
        memset(buffer, 0, bufferSize);
    }

    m_BufferBase.Reset(buffer);
    m_BufferSize = bufferSize;
    m_ConsumedSize = 0;
}

void PerformanceBufferManager::SetDetailTarget(NodeId id) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    m_DetailTarget = id;
}

void* PerformanceBufferManager::GetBufferBase() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    return (m_BufferBase.Get() != nullptr) ?
        (m_BufferBase + m_ConsumedSize).Get() :
        nullptr;
}

size_t PerformanceBufferManager::GetBufferSize() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    NN_SDK_ASSERT_GREATER_EQUAL(m_BufferSize, m_ConsumedSize);
    return (m_BufferBase.Get() != nullptr) ?
        (m_BufferSize - m_ConsumedSize) :
        0;
}

size_t PerformanceBufferManager::UpdateInParameter(void* pInParameter) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    auto p = reinterpret_cast<PerformanceBufferInfo::InParameter*>(pInParameter);
    p->detailTarget = m_DetailTarget;
    return sizeof(PerformanceBufferInfo::InParameter);
}

size_t PerformanceBufferManager::UpdateOutStatus(const void* pOutStatus) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    auto p = reinterpret_cast<const PerformanceBufferInfo::OutStatus*>(pOutStatus);
    m_ConsumedSize += p->consumedSize;
    m_ConsumedSize = std::min(m_ConsumedSize, m_BufferSize);
    return sizeof(PerformanceBufferInfo::OutStatus);
}

} // namespace audio
} // namespace nn
