﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once


namespace nn {
namespace audio {

/**
 * @brief       AuxBuffer から size 数のサンプルを読み込みます。
 * @param[in]   pAuxInfo                 AuxBufferInfo へのポインタ
 * @param[in]   buffer                   pOutBuffer に紐づくバッファ
 * @param[in]   countMax                 buffer が保持可能な最大サンプル数
 * @param[out]  pOutData                 指定数のサンプル列が読み込まれるバッファ
 * @param[in]   readCount                読み込みサンプル数を指定
 * @return      実際に読み込まれたサンプル数を返す
 * @pre
 * - pBuffer != nullptr
 * - pBuffer は初期化されている
 * - pOutData != nullptr
 * - size >= 0
 */
int32_t ReadAuxBuffer(AuxBufferInfo* pAuxInfo, int32_t* buffer, uint32_t countMax, int32_t* pOutData, uint32_t readCount) NN_NOEXCEPT;

/**
 * @brief       AuxBuffer へ size 数のサンプルを書き込みます。
 * @param[in]   pAuxInfo                 AuxBufferInfo へのポインタ
 * @param[in]   buffer                   pOutBuffer に紐づくバッファ
 * @param[in]   countMax                 buffer が保持可能な最大サンプル数
 * @param[in]   pData                    pAuxInfo の管理する buffer に書き出すサンプルを保持するバッファ
 * @param[in]   writeCount               buffer に書き出すサンプル数
 * @param[in]   writeOffset              書き出し先の writeOffset
 * @param[in]   updateCount              buffer に書き出すサンプル数
 * @return      実際に書き込まれたサンプル数を返す
 * @pre
 * - pBuffer != nullptr
 * - pBuffer は初期化されている
 * - pData != nullptr
 * - size >= 0
 * @details
 * 引数の writeOffset と AuxBufferInfo._writeOffsetCount の合計オフセットから pData の内容が、 writeCount 分書き出されます。
 * AuxBufferInfo._writeOffsetCount の値は updateCount の分だけ加算されます。よって updateCount に 0 を指定した場合は、
 * この関数の呼び出し前後で AuxBufferInfo._writeOffsetCount の値は変化しません。
 */
int32_t WriteAuxBuffer(AuxBufferInfo* pOutBuffer, int32_t* buffer, uint32_t countMax, const int32_t* pData, const uint32_t writeCount) NN_NOEXCEPT;

/**
 * @brief       Aux に渡す AuxBuffer の初期化を行います。
 * @param[out]  pOutAuxBuffer AuxBuffer へのポインタ
 * @param[in]   pBuffer                  AuxBuffer の内部に保持する buffer へのポインタ
 * @param[in]   size                     pBuffer のサイズ [byte]
 * @return      pOutAuxBufferInfo が管理するバッファの先頭アドレスを返します。
 * @pre
 * - pOutAuxBuffer != nullptr
 * - pBuffer != nullptr
 * - size > 0
 */
void* InitializeAuxBufferInfo(AuxBufferInfo** pOutAuxBufferInfo, void* pBuffer, int32_t size) NN_NOEXCEPT;

/**
 * @brief       AuxBuffer を破棄します。
 * @param[out]  pOutAuxBuffer AuxBuffer へのポインタ
 * @pre
 * - pOutAuxBuffer != nullptr
 * @post
 * - AuxBuffer は未初期化状態
 */
void FinalizeAuxBuffer( AuxBufferInfo* pOutAuxBufferInfo ) NN_NOEXCEPT;

/**
 * @brief           AuxBuffer の内部状態を初期化します。書き込み/読み込みの済んでいないサンプルは破棄されます。
 * @param[in]       *pOutAuxBuffer
 * @pre
 * - pOutAuxBuffer != nullptr
 * - pOutAuxBuffer は初期化されている
 */
void ResetAuxBuffer(AuxBufferInfo* pOutAuxBufferInfo) NN_NOEXCEPT;

}  // namespace audio
}  // namespace nn
