﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>

#include <nn/nn_SdkAssert.h>
#include <nn/nn_TimeSpan.h>
#include <nn/util/util_BitUtil.h>
#include <nn/util/util_BytePtr.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/audio/audio_Result.h>
#include <nn/audio/audio_AudioRendererTypes.h>
#include <nn/audio/audio_EffectTypes.h>
#include <nn/audio/audio_SubMixTypes.h>
#include <nn/audio/audio_FinalMixTypes.h>

#include "audio_EffectManager.h"
#include "audio_AuxTypes.h"
#include <nn/os/os_Cache.h>

namespace nn {
namespace audio {


namespace
{
uint32_t GetWritableCount(const AuxBufferInfo* pAuxInfo, uint32_t countMax) NN_NOEXCEPT
{
    uint32_t writableCount = 0;
    if (pAuxInfo->_cpu._readOffsetCount <= pAuxInfo->_cpu._writeOffsetCount)
    {
        writableCount += countMax - pAuxInfo->_cpu._writeOffsetCount;
        writableCount += pAuxInfo->_cpu._readOffsetCount;
    }
    else
    {
        writableCount += pAuxInfo->_cpu._readOffsetCount - pAuxInfo->_cpu._writeOffsetCount;
    }
    return writableCount;
}

}

int32_t ReadAuxBuffer(AuxBufferInfo* pAuxInfo, int32_t* buffer, uint32_t countMax, int32_t* pOutData, uint32_t count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pAuxInfo);
    NN_SDK_REQUIRES_NOT_NULL(pOutData);
    NN_SDK_REQUIRES_NOT_NULL(buffer);

    if (countMax == 0)
    {
        return 0;
    }

    os::FlushDataCache(reinterpret_cast<void*>(&pAuxInfo->_dsp), sizeof(AuxInfoDsp));
    pAuxInfo->_cpu._writeOffsetCount = pAuxInfo->_dsp._writeOffsetCount;

    // get read size.
    uint32_t readCount = 0;
    if (pAuxInfo->_cpu._writeOffsetCount < pAuxInfo->_cpu._readOffsetCount)
    {
        readCount += countMax - pAuxInfo->_cpu._readOffsetCount;
        readCount += pAuxInfo->_cpu._writeOffsetCount;
    }
    else
    {
        readCount += pAuxInfo->_cpu._writeOffsetCount - pAuxInfo->_cpu._readOffsetCount;
    }
    readCount = std::min(readCount, count);

    // copy data.
    int32_t* dst = pOutData;
    int32_t* srcBase = buffer;// reinterpret_cast<int32_t*>(pOutBuffer->_buffer.GetPointer());
    uint32_t  offset = pAuxInfo->_cpu._readOffsetCount;
    uint32_t left = readCount;

    while (left > 0)
    {
        int32_t* base = srcBase + offset;
        uint32_t cnt = std::min((countMax - offset), left);

        os::FlushDataCache(base, cnt * sizeof(int32_t));

        for (uint32_t i = 0; i < cnt; i++)
        {
            *dst++ = *base++;
        }
        offset += cnt;
        offset %= countMax;
        left -= cnt;
    }

    // update parameter.
    pAuxInfo->_cpu._readOffsetCount += readCount;
    pAuxInfo->_cpu._readOffsetCount %= countMax;
    os::FlushDataCache(&pAuxInfo->_cpu, sizeof(AuxInfoCpu));

    return readCount;

}

int32_t WriteAuxBuffer(AuxBufferInfo* pAuxInfo, int32_t* buffer, uint32_t countMax, const int32_t* pData, const uint32_t writeCount) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pAuxInfo);
    NN_SDK_REQUIRES_NOT_NULL(pData);
    NN_SDK_REQUIRES_NOT_NULL(buffer);

    if (countMax == 0)
    {
        return 0;
    }

    // write data into buffer
    const int32_t* src = pData;
    int32_t* dstBase = buffer;
    uint32_t offset = pAuxInfo->_cpu._writeOffsetCount;

    // reflect readOffset from DSP
    nn::os::FlushDataCache(&pAuxInfo->_dsp, sizeof(AuxInfoDsp));
    pAuxInfo->_cpu._readOffsetCount = pAuxInfo->_dsp._readOffsetCount;

    auto wtble = GetWritableCount(pAuxInfo, countMax);
    uint32_t wc = std::min(writeCount, wtble);

    uint32_t left = wc;

    while (left > 0)
    {
        int32_t* base = dstBase + offset;
        uint32_t cnt = std::min((countMax - offset), left);
        for (uint32_t i = 0; i < cnt; i++)
        {
            *base++ = *src++;
        }

        os::FlushDataCache(dstBase + offset, cnt * sizeof(int32_t));

        offset += cnt;
        offset %= countMax;
        left -= cnt;
    }

    pAuxInfo->_cpu._writeOffsetCount += wc;
    pAuxInfo->_cpu._writeOffsetCount %= countMax;
    os::FlushDataCache(&pAuxInfo->_cpu, sizeof(AuxInfoCpu));

    return wc;
}

void* InitializeAuxBufferInfo(AuxBufferInfo** pOutAuxBufferInfo, void* pBuffer, int32_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutAuxBufferInfo);
    NN_SDK_REQUIRES_NOT_NULL(pBuffer);
    NN_SDK_REQUIRES(size > 0);
    NN_SDK_REQUIRES(nn::util::is_aligned(reinterpret_cast<uintptr_t>(pBuffer), nn::audio::BufferAlignSize));
    NN_UNUSED(size);

    memset(pBuffer, 0, size);
    nn::os::FlushDataCache(pBuffer, size);

    const size_t infoSize = nn::util::align_up(sizeof(nn::audio::AuxBufferInfo), nn::audio::BufferAlignSize);

    AuxBufferInfo* aux = reinterpret_cast<AuxBufferInfo*>(pBuffer);

    aux->_cpu._readOffsetCount = 0;
    aux->_cpu._writeOffsetCount = 0;
    aux->_cpu._unpushedCount = 0;

    aux->_dsp._readOffsetCount = 0;
    aux->_dsp._writeOffsetCount = 0;
    aux->_dsp._unpushedCount = 0;

    *pOutAuxBufferInfo = aux;

    return nn::util::BytePtr(pBuffer, infoSize).Get();
}

void FinalizeAuxBuffer(AuxBufferInfo* pOutAuxBufferInfo) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutAuxBufferInfo);
    pOutAuxBufferInfo->_cpu._readOffsetCount = 0;
    pOutAuxBufferInfo->_cpu._writeOffsetCount = 0;
}

void ResetAuxBuffer(AuxBufferInfo* pOutAuxBuffer) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutAuxBuffer);
    pOutAuxBuffer->_cpu._readOffsetCount = 0;
    pOutAuxBuffer->_cpu._writeOffsetCount = 0;
    pOutAuxBuffer->_dsp._unpushedCount = 0;
}

}  // namespace audio
}  // namespace nn
