﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <nn/nn_Macro.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_StaticAssert.h>
#include <nn/util/util_BitUtil.h>
#include <nn/audio/audio_Adpcm.h>

namespace nn {
namespace audio {

struct AdpcmHeader
{
    uint32_t sampleCount;
    uint32_t nibbleCount;
    uint32_t sampleRate;
    uint16_t loop;
    uint16_t format;
    uint32_t loopBeginSampleAddress;
    uint32_t loopEndSampleAddress;
    uint32_t ca;
    AdpcmParameter coefficients;
    uint16_t gain;
    AdpcmContext initialContext;
    AdpcmContext loopContext;
    uint16_t reserved[11];
};
namespace {

const int HeaderNibbleCount = 2;
const int NibblesPerFrame = 16;
const int SamplesPerFrame = 14;

}

int ConvertAdpcmNibbleCountToSampleCount(int nibbleCount) NN_NOEXCEPT
{
    int frameCount = nibbleCount / NibblesPerFrame;
    int tailNibbleCount = nibbleCount % NibblesPerFrame;

    return (frameCount * SamplesPerFrame) + (tailNibbleCount - HeaderNibbleCount);
}

int ConvertAdpcmSampleCountToNibbleCount(int sampleCount) NN_NOEXCEPT
{
    int frameCount = sampleCount / SamplesPerFrame;
    int tailSampleCount = sampleCount % SamplesPerFrame;

    return (frameCount * NibblesPerFrame) + (tailSampleCount + HeaderNibbleCount);
}

void ParseAdpcmHeader(AdpcmHeaderInfo* pOutInfo, const void* header, std::size_t size) NN_NOEXCEPT
{
    NN_STATIC_ASSERT(sizeof(AdpcmHeader) == AdpcmHeaderSize);

    NN_SDK_REQUIRES_NOT_NULL(pOutInfo);
    NN_SDK_REQUIRES_NOT_NULL(header);
    NN_SDK_REQUIRES_ALIGNED(reinterpret_cast<uintptr_t>(header), sizeof(int32_t));
    NN_SDK_REQUIRES_GREATER_EQUAL(size, AdpcmHeaderSize);
    NN_UNUSED(size);

    const AdpcmHeader* p = static_cast<const AdpcmHeader*>(header);

    pOutInfo->parameter = p->coefficients;
    pOutInfo->loopContext = p->loopContext;
    pOutInfo->sampleCount = p->sampleCount;
    pOutInfo->sampleRate = p->sampleRate;
    pOutInfo->loop = (p->loop != 0);
    pOutInfo->loopBeginSamplePosition = ConvertAdpcmNibbleCountToSampleCount(p->loopBeginSampleAddress);
    pOutInfo->loopEndSamplePosition = ConvertAdpcmNibbleCountToSampleCount(p->loopEndSampleAddress);
}

}  // namespace audio
}  // namespace nn
