﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/applet/applet_LibraryAppletSelf.h>

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Abort.h>
#include <nn/result/result_HandlingUtility.h>

#include <utility>
#include <nn/am/am_Result.h>
#include <nn/am/am_Shim.h>
#include <nn/applet/applet_Storage.h>
#include "applet_StorageUtility.h"

namespace nn { namespace applet {

using detail::GetPointer;
using detail::GetShared;
using detail::PopStorage;

struct LibraryAppletSelfHandleObject
{
    am::service::ILibraryAppletSelfAccessor* p;
    am::service::IProcessWindingController* pWindingController;
    bool isInteractiveInEventValid;
    os::SystemEventType interactiveInEvent;
};

namespace {

    os::Mutex g_Mutex{false};
    LibraryAppletSelfHandleObject g_Objects[2];

    LibraryAppletSelfHandleObject* CreateObject(sf::SharedPointer<am::service::ILibraryAppletSelfAccessor> p, sf::SharedPointer<am::service::IProcessWindingController> pWindingController) NN_NOEXCEPT
    {
        std::lock_guard<decltype(g_Mutex)> lk(g_Mutex);
        for (auto&& e : g_Objects)
        {
            if (!e.p)
            {
                e.p = p.Detach();
                e.pWindingController = pWindingController.Detach();
                e.isInteractiveInEventValid = false;
                return &e;
            }
        }
        NN_ABORT("too many applications");
    }

    void DeleteObject(LibraryAppletSelfHandleObject* p) NN_NOEXCEPT
    {
        std::lock_guard<decltype(g_Mutex)> lk(g_Mutex);
        auto&& e = *p;
        if (e.isInteractiveInEventValid)
        {
            os::DestroySystemEvent(&e.interactiveInEvent);
        }
        sf::ReleaseSharedObject(e.p);
        sf::ReleaseSharedObject(e.pWindingController);
        e.p = nullptr;
    }

    LibraryAppletSelfHandleObject* GetObject(LibraryAppletSelfHandle handle) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(handle != InvalidLibraryAppletSelfHandle);
        return static_cast<LibraryAppletSelfHandleObject*>(handle._p);
    }

    am::service::ILibraryAppletSelfAccessor* GetPointer(LibraryAppletSelfHandle handle) NN_NOEXCEPT
    {
        return GetObject(handle)->p;
    }

    am::service::IProcessWindingController* GetWindingController(LibraryAppletSelfHandle handle) NN_NOEXCEPT
    {
        return GetObject(handle)->pWindingController;
    }

} // anon

LibraryAppletSelfHandle CreateLibraryAppletSelf(sf::SharedPointer<am::service::ILibraryAppletSelfAccessor> p, sf::SharedPointer<am::service::IProcessWindingController> pWindingController) NN_NOEXCEPT
{
    return {CreateObject(std::move(p), std::move(pWindingController))};
}

void CloseLibraryAppletSelf(LibraryAppletSelfHandle selfHandle) NN_NOEXCEPT
{
    DeleteObject(static_cast<LibraryAppletSelfHandleObject*>(selfHandle._p));
}

void UnpopToInChannel(LibraryAppletSelfHandle selfHandle, StorageHandle storageHandle) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(GetPointer(selfHandle)->UnpopInData(GetShared(storageHandle)));
    ReleaseStorage(storageHandle);
}

bool TryPopFromInChannel(StorageHandle* pOut, LibraryAppletSelfHandle selfHandle) NN_NOEXCEPT
{
    return PopStorage(pOut, [&] (sf::SharedPointer<am::service::IStorage>* pOut) -> Result
    {
        return GetPointer(selfHandle)->PopInData(pOut);
    });
}

bool TryPopFromInteractiveInChannel(StorageHandle* pOut, LibraryAppletSelfHandle selfHandle) NN_NOEXCEPT
{
    return PopStorage(pOut, [&] (sf::SharedPointer<am::service::IStorage>* pOut) -> Result
    {
        return GetPointer(selfHandle)->PopInteractiveInData(pOut);
    });
}

os::SystemEventType* GetPopFromInteractiveInChannelEvent(LibraryAppletSelfHandle selfHandle) NN_NOEXCEPT
{
    auto&& e = *GetObject(selfHandle);
    std::lock_guard<decltype(g_Mutex)> lk(g_Mutex);
    if (!e.isInteractiveInEventValid)
    {
        sf::NativeHandle eventHandle;
        NN_ABORT_UNLESS_RESULT_SUCCESS(GetPointer(selfHandle)->GetPopInteractiveInDataEvent(&eventHandle));
        os::AttachReadableHandleToSystemEvent(&e.interactiveInEvent, eventHandle.GetOsHandle(), eventHandle.IsManaged(), os::EventClearMode_ManualClear);
        eventHandle.Detach();
        e.isInteractiveInEventValid = true;
    }
    return &e.interactiveInEvent;
}

void PushToOutChannel(LibraryAppletSelfHandle selfHandle, StorageHandle storageHandle) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(GetPointer(selfHandle)->PushOutData(GetShared(storageHandle)));
    ReleaseStorage(storageHandle);
}

void PushToInteractiveOutChannel(LibraryAppletSelfHandle selfHandle, StorageHandle storageHandle) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(GetPointer(selfHandle)->PushInteractiveOutData(GetShared(storageHandle)));
    ReleaseStorage(storageHandle);
}

void PushToContextStack(LibraryAppletSelfHandle selfHandle, StorageHandle storageHandle) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(GetWindingController(selfHandle)->PushContext(GetShared(storageHandle)));
    ReleaseStorage(storageHandle);
}

bool TryPopFromContextStack(StorageHandle* pOut, LibraryAppletSelfHandle selfHandle) NN_NOEXCEPT
{
    return PopStorage(pOut, [&] (sf::SharedPointer<am::service::IStorage>* pOut) -> Result
    {
        return GetWindingController(selfHandle)->PopContext(pOut);
    });
}

vi::IndirectProducerHandleType GetIndirectLayerProducerHandle(LibraryAppletSelfHandle selfHandle) NN_NOEXCEPT
{
    vi::IndirectProducerHandleType ret;
    NN_ABORT_UNLESS_RESULT_SUCCESS(GetPointer(selfHandle)->GetIndirectLayerProducerHandle(&ret));
    return ret;
}

}}
