﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/applet/applet_ApplicationSelf.h>
#include <nn/applet/applet_ApplicationControlForQuest.h>
#include <nn/applet/applet_SubProgramJump.h>

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Abort.h>
#include <nn/result/result_HandlingUtility.h>

#include <utility>
#include <nn/applet/applet_Storage.h>
#include "applet_StorageUtility.h"

namespace nn { namespace applet {

bool TryPopFromApplicationParameterChannel(StorageHandle* pOut, LaunchParameterKind kind) NN_NOEXCEPT
{
    return detail::PopStorage(pOut, [&] (sf::SharedPointer<am::service::IStorage>* pOut) -> Result
    {
        return am::GetApplicationFunctions()->PopLaunchParameter(pOut, kind);
    });
}

void RequestToRelaunchApplication(StorageHandle storageHandle) NN_NOEXCEPT
{
    RequestToLaunchApplication(ncm::ApplicationId::GetInvalidId(), storageHandle);
}

void RequestToLaunchApplication(ncm::ApplicationId applicationId, StorageHandle storageHandle) NN_NOEXCEPT
{
    if (InvalidStorageHandle == storageHandle)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(am::GetApplicationFunctions()->CreateApplicationAndRequestToStart(applicationId));
    }
    else
    {
        sf::SharedPointer<am::service::IStorage> storage(static_cast<am::service::IStorage*>(storageHandle._p), false);
        NN_ABORT_UNLESS_RESULT_SUCCESS(am::GetApplicationFunctions()->CreateApplicationAndPushAndRequestToStart(applicationId, std::move(storage)));
    }
}

// 試遊台向け
void RequestToLaunchApplicationForQuest(ncm::ApplicationId applicationId, StorageHandle storageHandle, const ApplicationAttributeForQuest* pAttribute) NN_NOEXCEPT
{
    if (InvalidStorageHandle == storageHandle)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(am::GetApplicationFunctions()->CreateApplicationAndRequestToStartForQuest(applicationId, pAttribute->playableTime, pAttribute->idleDetectionTime));
    }
    else
    {
        sf::SharedPointer<am::service::IStorage> storage(static_cast<am::service::IStorage*>(storageHandle._p), false);
        NN_ABORT_UNLESS_RESULT_SUCCESS(am::GetApplicationFunctions()->CreateApplicationAndPushAndRequestToStartForQuest(applicationId, std::move(storage), pAttribute->playableTime, pAttribute->idleDetectionTime));
    }
}

// キーボード配列設定
void SetDesirableKeyboardLayout(Bit32 layout) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(am::GetSelfController()->SetDesirableKeyboardLayout(layout));
}

namespace {

Result ExecuteProgramImpl(am::service::ProgramSpecifyKind kind, Bit64 id, StorageHandle storageHandle) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(am::GetApplicationFunctions()->ClearUserChannel());
    if (storageHandle != InvalidStorageHandle)
    {
        sf::SharedPointer<am::service::IStorage> storage(static_cast<am::service::IStorage*>(storageHandle._p), false);
        NN_RESULT_DO(am::GetApplicationFunctions()->UnpopToUserChannel(std::move(storage)));
    }
    return am::GetApplicationFunctions()->ExecuteProgram(kind, id);
}

}

Result RestartProgram(StorageHandle storageHandle) NN_NOEXCEPT
{
    return ExecuteProgramImpl(am::service::ProgramSpecifyKind::Same, 0, storageHandle);
}

Result ExecuteProgram(uint8_t programIndex, StorageHandle storageHandle) NN_NOEXCEPT
{
    return ExecuteProgramImpl(am::service::ProgramSpecifyKind::SubProgramIndex, programIndex, storageHandle);
}

int32_t GetPreviousProgramIndex() NN_NOEXCEPT
{
    int32_t ret;
    NN_ABORT_UNLESS_RESULT_SUCCESS(am::GetApplicationFunctions()->GetPreviousProgramIndex(&ret));
    return ret;
}

Result JumpToSubApplicationProgramForDevelopment(ncm::ApplicationId applicationId, StorageHandle storageHandle) NN_NOEXCEPT
{
    return ExecuteProgramImpl(am::service::ProgramSpecifyKind::ApplicationId, applicationId.value, storageHandle);
}

}}
